<?php
/**
 * Plugin Name: ACE Backyard Landing Page
 * Description: High-converting backyard remodeling landing page with admin management
 * Version: 1.6.0
 * Author: ACE Design &amp; Build
 */

if (!defined('ABSPATH')) exit;

define('ACE_BLP_VERSION', '1.6.3');
define('ACE_BLP_PATH', plugin_dir_path(__FILE__));
define('ACE_BLP_URL', plugin_dir_url(__FILE__));

// Include classes
require_once ACE_BLP_PATH . 'includes/class-admin.php';
require_once ACE_BLP_PATH . 'includes/class-frontend.php';
require_once ACE_BLP_PATH . 'includes/class-leads.php';
require_once ACE_BLP_PATH . 'includes/class-calculator.php';
require_once ACE_BLP_PATH . 'includes/class-visualizer.php';

// Activation
register_activation_hook(__FILE__, 'ace_blp_activate');
function ace_blp_activate() {
    ACE_BLP_Leads::create_table();
    ace_blp_maybe_upgrade_db();
    ace_blp_set_defaults();
    ace_blp_register_rewrite();
    flush_rewrite_rules();
}

// Upgrade DB - add zip_code column if missing
function ace_blp_maybe_upgrade_db() {
    global $wpdb;
    $table = $wpdb->prefix . 'ace_backyard_leads';
    $columns = $wpdb->get_col("SHOW COLUMNS FROM $table");
    if (!in_array('zip_code', $columns)) {
        $wpdb->query("ALTER TABLE $table ADD COLUMN zip_code VARCHAR(20) DEFAULT '' AFTER address");
    }
}

// Upgrade content - force-overwrite projects and reviews from defaults
function ace_blp_upgrade_content() {
    $settings = get_option('ace_backyard_lp_settings');
    if (!$settings || !is_array($settings)) return;

    $defaults = ace_blp_get_defaults();

    // Force-overwrite projects, reviews, and calculator prices with latest defaults
    $settings['projects_items'] = $defaults['projects_items'];
    $settings['reviews_items'] = $defaults['reviews_items'];
    $settings['calc_features'] = $defaults['calc_features'];

    update_option('ace_backyard_lp_settings', $settings);
}

// Deactivation
register_deactivation_hook(__FILE__, 'ace_blp_deactivate');
function ace_blp_deactivate() {
    flush_rewrite_rules();
    // Unschedule visualizer cleanup cron
    if (function_exists('ace_visualizer_deactivate_cleanup')) {
        ace_visualizer_deactivate_cleanup();
    }
}

// Set default options
function ace_blp_set_defaults() {
    if (get_option('ace_backyard_lp_settings')) return;

    $defaults = ace_blp_get_defaults();
    update_option('ace_backyard_lp_settings', $defaults);
}

function ace_blp_get_defaults() {
    return array(
        // General
        'slug' => 'backyard',
        'phone' => '(310) 438-6866',
        'email' => get_option('admin_email'),
        'enabled' => 1,
        'scroll_nav_enabled' => 1,
        'scroll_nav_labels' => array(
            array('target' => '#calculator', 'label' => 'Calculator'),
            array('target' => '#projects', 'label' => 'Our Work'),
            array('target' => '#reviews', 'label' => 'Reviews'),
            array('target' => '#why-ace', 'label' => 'Why ACE'),
            array('target' => '#process', 'label' => 'Process'),
            array('target' => '#faq', 'label' => 'FAQ'),
            array('target' => '#final', 'label' => 'Get Quote'),
        ),
        'seo_title' => 'Backyard Remodeling Los Angeles - ACE Design & Build',
        'seo_description' => 'Transform your backyard with ACE Design & Build. Custom pools, outdoor kitchens, hardscape, and more. Free 3D design. Serving Los Angeles & San Fernando Valley.',
        'og_image' => '',

        // Hero
        'hero_badge' => 'Serving Los Angeles & San Fernando Valley',
        'hero_headline' => "Your Backyard's Best Days Are Ahead",
        'hero_gold_word' => 'Ahead',
        'hero_subheadline' => 'ACE Design & Build transforms ordinary backyards into resort-style escapes - from concept to completion, with a free 3D design to see it before we build it.',
        'hero_bg_image' => 'https://images.unsplash.com/photo-1600596542815-ffad4c1539a8?w=1920',
        'hero_form_title' => 'Get Your Free Backyard Plan',
        'hero_form_subtitle' => 'Tell us a little about your project. We respond within 2 hours.',
        'hero_form_button' => 'Get My Free Plan',
        'hero_trust' => array(
            array('icon' => 'fas fa-star', 'text' => '4.9 Google (120+ Reviews)'),
            array('icon' => 'fas fa-trophy', 'text' => '15+ Years Experience'),
            array('icon' => 'fas fa-shield-halved', 'text' => 'Licensed & Insured'),
            array('icon' => 'fas fa-cube', 'text' => 'Free 3D Design'),
        ),
        'hero_enabled' => 1,

        // Proof Strip
        'proof_images' => array(),
        'proof_speed' => 30,
        'proof_enabled' => 1,

        // Transformation
        'transform_label' => 'The Transformation',
        'transform_headline' => 'Stop Dreaming. Start Living.',
        'transform_gold_word' => 'Living.',
        'transform_subheadline' => "Most homeowners know their backyard could be more - but they don't know where to start. Here's what waiting is really costing you:",
        'transform_before_image' => '',
        'transform_after_image' => '',
        'transform_pain_points' => array(
            array('icon' => 'fas fa-eye-slash', 'text' => 'A yard you avoid instead of enjoy - unused square footage going to waste'),
            array('icon' => 'fas fa-chart-line-down', 'text' => "Home value stagnating while your neighbors' properties appreciate"),
            array('icon' => 'fas fa-clock', 'text' => 'Weekend DIY projects that never get finished - and cost more than a pro'),
        ),
        'transform_enabled' => 1,

        // Calculator
        'calc_headline' => 'What Does a Backyard Remodel Cost?',
        'calc_subheadline' => 'Use our calculator to get a ballpark estimate in 30 seconds. Then let us give you an exact quote.',
        'calc_sizes' => array(
            array('name' => 'Small', 'desc' => '< 500 sq ft', 'multiplier' => '0.6'),
            array('name' => 'Medium', 'desc' => '500 - 1,000 sq ft', 'multiplier' => '1.0'),
            array('name' => 'Large', 'desc' => '1,000 - 1,500 sq ft', 'multiplier' => '1.4'),
            array('name' => 'XL', 'desc' => '1,500+ sq ft', 'multiplier' => '1.8'),
        ),
        'calc_features' => array(
            array('name' => 'Patio / Hardscape', 'icon' => 'fas fa-border-all', 'std' => 8000, 'prem' => 17000, 'lux' => 30000),
            array('name' => 'Swimming Pool', 'icon' => 'fas fa-water-ladder', 'std' => 38000, 'prem' => 60000, 'lux' => 90000),
            array('name' => 'Outdoor Kitchen', 'icon' => 'fas fa-fire-burner', 'std' => 12000, 'prem' => 30000, 'lux' => 60000),
            array('name' => 'Fire Feature', 'icon' => 'fas fa-fire', 'std' => 1500, 'prem' => 6000, 'lux' => 12000),
            array('name' => 'Landscaping', 'icon' => 'fas fa-seedling', 'std' => 4000, 'prem' => 12000, 'lux' => 23000),
            array('name' => 'Patio Cover', 'icon' => 'fas fa-umbrella', 'std' => 4000, 'prem' => 9000, 'lux' => 15000),
            array('name' => 'Lighting', 'icon' => 'fas fa-lightbulb', 'std' => 1500, 'prem' => 5000, 'lux' => 8000),
            array('name' => 'Artificial Turf', 'icon' => 'fas fa-spa', 'std' => 2500, 'prem' => 6000, 'lux' => 12000),
        ),
        'calc_finishes' => array(
            array('name' => 'Standard', 'desc' => 'Quality materials, clean design', 'key' => 'std'),
            array('name' => 'Premium', 'desc' => 'Upgraded materials & features', 'key' => 'prem'),
            array('name' => 'Luxury', 'desc' => 'High-end, custom everything', 'key' => 'lux'),
        ),
        'calc_low_mult' => 0.85,
        'calc_high_mult' => 1.15,
        'calc_result_note' => 'This is a ballpark estimate. Your actual price depends on site conditions, materials, and design complexity. Get a precise quote - it is free and takes 48 hours.',
        'calc_form_button' => 'Get My Exact Quote',
        'calc_default_features' => array(0, 4), // Patio + Landscaping indices
        'calc_default_size' => 1, // Medium
        'calc_default_finish' => 1, // Premium
        'calc_enabled' => 1,

        // Services
        'services_label' => 'What We Build',
        'services_headline' => 'Everything Your Backyard Needs',
        'services_gold_word' => 'Needs',
        'services_items' => array(
            array('icon' => 'fas fa-layer-group', 'title' => 'Hardscape & Pavers', 'desc' => 'Travertine, porcelain, and interlocking paver patios.', 'image' => ''),
            array('icon' => 'fas fa-water-ladder', 'title' => 'Swimming Pools & Spas', 'desc' => 'Custom pools, heated spas, and water features.', 'image' => ''),
            array('icon' => 'fas fa-fire-burner', 'title' => 'Outdoor Kitchens & BBQ', 'desc' => 'Built-in islands, countertops, and dining areas.', 'image' => ''),
            array('icon' => 'fas fa-umbrella', 'title' => 'Patio Covers & Pergolas', 'desc' => 'Insulated roofs, wood pergolas, shade structures.', 'image' => ''),
            array('icon' => 'fas fa-seedling', 'title' => 'Landscape Design', 'desc' => 'Drought-tolerant planting, irrigation, and greenery.', 'image' => ''),
            array('icon' => 'fas fa-fire', 'title' => 'Fire Pits & Fireplaces', 'desc' => 'Custom gas and wood-burning fire features.', 'image' => ''),
            array('icon' => 'fas fa-spa', 'title' => 'Artificial Turf', 'desc' => 'Realistic, maintenance-free lawns and greens.', 'image' => ''),
            array('icon' => 'fas fa-lightbulb', 'title' => 'Lighting & Ambiance', 'desc' => 'LED landscape lighting and evening atmosphere.', 'image' => ''),
        ),
        'services_enabled' => 1,

        // Projects
        'projects_label' => 'Featured Projects',
        'projects_headline' => 'Real Backyards. Real Families. Real Results.',
        'projects_gold_word' => 'Results.',
        'projects_subheadline' => '',
        'projects_items' => array(
            array(
                'name' => 'The Martinez Family Retreat',
                'location' => 'Encino, CA',
                'scope' => 'Pool + Outdoor Kitchen + Paver Patio + Full Landscaping + Lighting',
                'budget' => '$85K - $95K',
                'timeline' => '10 Weeks',
                'area' => '1,200 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600596542815-ffad4c1539a8?w=800',
            ),
            array(
                'name' => 'The Cohen Entertainment Oasis',
                'location' => 'Sherman Oaks, CA',
                'scope' => 'Infinity Pool + Spa + Fire Pit + Pergola + Turf + Outdoor Bar',
                'budget' => '$120K - $140K',
                'timeline' => '14 Weeks',
                'area' => '1,800 sq ft',
                'image' => 'https://images.unsplash.com/photo-1564013799919-ab600027ffc6?w=800',
            ),
            array(
                'name' => 'The Park Family Sanctuary',
                'location' => 'Woodland Hills, CA',
                'scope' => 'Paver Patio + Landscape Design + Patio Cover + Fire Pit + Lighting',
                'budget' => '$45K - $55K',
                'timeline' => '6 Weeks',
                'area' => '800 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600585154340-be6161a56a0c?w=800',
            ),
            array(
                'name' => 'The Rivera Modern Escape',
                'location' => 'Studio City, CA',
                'scope' => 'Plunge Pool + Outdoor Kitchen + Concrete Patio + Turf + Lighting',
                'budget' => '$70K - $80K',
                'timeline' => '8 Weeks',
                'area' => '950 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600607687939-ce8a6c25118c?w=800',
            ),
            array(
                'name' => 'The Nguyen Resort Oasis',
                'location' => 'Encino, CA',
                'scope' => 'Pool + Spa + Outdoor Kitchen + Pergola + Full Landscaping + Lighting',
                'budget' => '$150K - $175K',
                'timeline' => '16 Weeks',
                'area' => '2,200 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600566753190-17f0baa2a6c3?w=800',
            ),
            array(
                'name' => 'The Thompson Family Retreat',
                'location' => 'Tarzana, CA',
                'scope' => 'Paver Patio + BBQ Island + Fire Pit + Turf + Landscape Design + String Lighting',
                'budget' => '$55K - $65K',
                'timeline' => '7 Weeks',
                'area' => '900 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600573472592-401b489a3cdc?w=800',
            ),
            array(
                'name' => 'The Goldstein Entertainment Yard',
                'location' => 'Beverly Hills, CA',
                'scope' => 'Infinity Pool + Outdoor Bar + Lounge Area + Fire Feature + Premium Landscaping + Smart Lighting',
                'budget' => '$200K - $240K',
                'timeline' => '20 Weeks',
                'area' => '3,000 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600047509807-ba8f99d2cdde?w=800',
            ),
            array(
                'name' => 'The Ramirez Backyard Makeover',
                'location' => 'Van Nuys, CA',
                'scope' => 'Concrete Patio + Pergola + Turf + Planter Walls + LED Pathway Lighting',
                'budget' => '$35K - $42K',
                'timeline' => '5 Weeks',
                'area' => '650 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600585154526-990dced4db0d?w=800',
            ),
            array(
                'name' => 'The Kim Modern Escape',
                'location' => 'Studio City, CA',
                'scope' => 'Plunge Pool + Hardscape Deck + Outdoor Kitchen + Vertical Garden + Accent Lighting',
                'budget' => '$95K - $110K',
                'timeline' => '12 Weeks',
                'area' => '1,400 sq ft',
                'image' => 'https://images.unsplash.com/photo-1600210492486-724fe5c67fb0?w=800',
            ),
        ),
        'projects_cta_text' => 'Your Backyard Could Be Next',
        'projects_enabled' => 1,

        // CTA Bar
        'cta_text' => 'Ready to Love Your Backyard?',
        'cta_gold_text' => 'It Starts with a Free Plan.',
        'cta_button' => 'Get Started',
        'cta_enabled' => 1,

        // Reviews
        'reviews_label' => 'Reviews',
        'reviews_headline' => "Don't Take Our Word For It",
        'reviews_gold_word' => '',
        'reviews_items' => array(
            array('stars' => 5, 'quote' => 'ACE turned our boring backyard into something out of a magazine. The team was professional, on time, and the 3D design they showed us was almost identical to the final result. Could not be happier.', 'name' => 'Jessica & Tom R.', 'location' => 'Encino, CA'),
            array('stars' => 5, 'quote' => 'From start to finish, ACE made the entire process seamless. They handled permits, HOA approvals, and kept us updated every week. Our outdoor kitchen is now the envy of the neighborhood.', 'name' => 'Marcus D.', 'location' => 'Sherman Oaks, CA'),
            array('stars' => 5, 'quote' => 'We got quotes from four different contractors. ACE was not the cheapest, but they were the most transparent. No hidden fees, no surprises. The quality of work speaks for itself.', 'name' => 'Rachel & Brian K.', 'location' => 'Woodland Hills, CA'),
            array('stars' => 5, 'quote' => 'ACE completely transformed our backyard. The pool came out better than what we saw in the 3D rendering. Their attention to detail is unreal.', 'name' => 'David & Sarah M.', 'location' => 'Woodland Hills, CA'),
            array('stars' => 5, 'quote' => 'We interviewed 5 contractors before choosing ACE. Best decision we ever made. They were transparent about costs from day one and delivered exactly what they promised.', 'name' => 'Michael T.', 'location' => 'Encino, CA'),
            array('stars' => 5, 'quote' => 'The outdoor kitchen ACE built for us has become the centerpiece of every family gathering. Professional crew, clean worksite, and they finished a week early.', 'name' => 'Jennifer & Mark L.', 'location' => 'Sherman Oaks, CA'),
            array('stars' => 5, 'quote' => 'From the first consultation to the final walkthrough, ACE made the whole process feel easy. Our neighbors keep asking who did our yard.', 'name' => 'Robert K.', 'location' => 'Studio City, CA'),
            array('stars' => 5, 'quote' => 'We had a small backyard and a tight budget. ACE designed something that made it feel twice as big. The paver patio and lighting completely changed the space.', 'name' => 'Lisa P.', 'location' => 'Van Nuys, CA'),
            array('stars' => 5, 'quote' => 'ACE handled everything - permits, HOA approval, design, construction. We did not have to lift a finger. The fire pit area is our favorite spot now.', 'name' => 'James & Diana W.', 'location' => 'Tarzana, CA'),
            array('stars' => 5, 'quote' => 'Third time using ACE. They did our front yard, then backyard, now pool. Consistent quality every single time. Would not trust anyone else.', 'name' => 'Steven R.', 'location' => 'Beverlywood, CA'),
            array('stars' => 4, 'quote' => 'Great work overall. The hardscape and landscaping look amazing. Only reason for 4 stars is that the project took 2 weeks longer than estimated, but the result was worth the wait.', 'name' => 'Amanda G.', 'location' => 'West Hills, CA'),
            array('stars' => 5, 'quote' => 'The 3D design sold us immediately. Being able to see exactly what our backyard would look like before committing gave us so much confidence. The final result matched perfectly.', 'name' => 'Chris & Emily N.', 'location' => 'Reseda, CA'),
            array('stars' => 5, 'quote' => 'ACE turned our dirt lot into a resort. Pool, spa, outdoor kitchen, turf - the works. Every contractor we talked to said it could not be done in our timeline. ACE did it.', 'name' => 'Daniel & Maria F.', 'location' => 'Playa del Rey, CA'),
        ),
        'reviews_google_score' => '4.9',
        'reviews_google_count' => '120+',
        'reviews_enabled' => 1,

        // Why ACE
        'why_label' => 'Why ACE',
        'why_headline' => 'Why 500+ LA Homeowners Chose ACE',
        'why_gold_word' => 'ACE',
        'why_subheadline' => '',
        'why_image' => '',
        'why_items' => array(
            array('icon' => 'fas fa-compass-drafting', 'title' => 'Design-Build Under One Roof', 'desc' => 'Design, permits, construction - all managed by one team. Faster timelines, fewer headaches.'),
            array('icon' => 'fas fa-cube', 'title' => 'Free 3D Design', 'desc' => 'See your backyard in photorealistic 3D before a single nail is driven. Every project includes a complimentary rendering.'),
            array('icon' => 'fas fa-file-invoice-dollar', 'title' => 'Transparent Pricing', 'desc' => 'No hidden fees. No surprise change orders. Your quote is your quote. Period.'),
            array('icon' => 'fas fa-shield-halved', 'title' => 'Licensed, Insured, Guaranteed', 'desc' => 'CSLB Licensed. Fully insured. Written warranty on every project we deliver.'),
        ),
        'why_enabled' => 1,

        // Process
        'process_label' => 'Our Process',
        'process_headline' => 'From Vision to Backyard in 4 Steps',
        'process_gold_word' => '4 Steps',
        'process_items' => array(
            array('icon' => 'fas fa-comments', 'title' => 'Free In-Home Consultation', 'desc' => 'We visit your home, listen to your vision, and assess the space together.'),
            array('icon' => 'fas fa-cube', 'title' => '3D Design', 'desc' => 'See your backyard in full 3D before we break ground. Revise until it is perfect.'),
            array('icon' => 'fas fa-hammer', 'title' => 'Build', 'desc' => 'Our crew builds with precision. Weekly updates. On budget. On schedule.'),
            array('icon' => 'fas fa-glass-cheers', 'title' => 'Enjoy', 'desc' => 'Your backyard is ready. Time to make memories that last a lifetime.'),
        ),
        'process_enabled' => 1,

        // FAQ
        'faq_label' => 'FAQ',
        'faq_headline' => 'Got Questions? We Have Answers.',
        'faq_gold_word' => 'Answers.',
        'faq_items' => array(
            array('q' => 'How much does a backyard remodel cost?', 'a' => 'It depends on size and scope. A basic patio with landscaping starts around $25K-$40K. A full backyard with pool, kitchen, and hardscape runs $80K-$150K+. Our free calculator above gives a ballpark, and our in-home consultation gives you an exact number.'),
            array('q' => 'How long does a project take?', 'a' => 'Most projects take 6-14 weeks from groundbreaking to completion. A simple patio might be 4-6 weeks, while a full backyard with pool can be 12-16 weeks. We give you a detailed timeline before we start and stick to it.'),
            array('q' => 'Do you handle permits?', 'a' => 'Yes. We handle all permits and inspections for every project. Pools, structures, electrical, plumbing - we manage the entire process with the city so you do not have to.'),
            array('q' => 'What areas do you serve?', 'a' => 'We serve Los Angeles and the San Fernando Valley, including Encino, Sherman Oaks, Woodland Hills, Studio City, Tarzana, Calabasas, Brentwood, Bel Air, Pacific Palisades, and surrounding areas.'),
            array('q' => 'What is included in the free 3D design?', 'a' => 'After our initial consultation, we create a photorealistic 3D rendering of your proposed backyard. You can see exactly what your space will look like - materials, colors, layout, lighting - before any work begins. Revisions are included until you love it.'),
            array('q' => 'Do you offer financing?', 'a' => 'Yes, we work with several financing partners to offer flexible payment plans. Many homeowners finance their backyard projects with terms that work for their budget. Ask us during your consultation for current options.'),
            array('q' => 'What if I only want part of my backyard done?', 'a' => 'Absolutely. Many clients start with one area - like a patio or fire pit - and add features over time. We design with future phases in mind so everything integrates seamlessly when you are ready to expand.'),
        ),
        'faq_enabled' => 1,

        // Final CTA
        'final_badge' => 'Limited: Free 3D Rendering ($2,500 Value)',
        'final_headline' => 'Your Backyard. Reimagined.',
        'final_gold_word' => 'Reimagined.',
        'final_subheadline' => 'Take the first step. Tell us about your space and we will create a free 3D design - so you can see the possibilities before committing to anything.',
        'final_form_button' => 'Book My Free In-Home Consultation',
        'final_heard_options' => array('Google Search', 'Instagram / Facebook', 'Neighbor / Friend Referral', 'Yelp', 'Drove By a Project', 'Other'),
        'final_enabled' => 1,

        // Popup
        'popup_headline' => 'Get Your Free Backyard Plan',
        'popup_subtitle' => 'Tell us a little about your project. We respond within 2 hours.',
        'popup_button' => 'Get My Free Plan',
    );
}

// Migrate visualizer options to DB-backed mode toggle
function ace_blp_migrate_viz_options() {
    // Set active mode based on current state
    if (!get_option('ace_active_mode')) {
        $mode = get_option('ace_visualizer_enabled') ? 'visualizer' : 'calculator';
        update_option('ace_active_mode', $mode);
    }

    // Seed viz sizes if not yet present
    if (!get_option('ace_viz_sizes')) {
        update_option('ace_viz_sizes', ace_blp_get_viz_size_defaults());
    }

    // Seed viz features if not yet present
    if (!get_option('ace_viz_features')) {
        update_option('ace_viz_features', ace_blp_get_viz_feature_defaults());
    }

    // Seed viz finishes if not yet present
    if (!get_option('ace_viz_finishes')) {
        update_option('ace_viz_finishes', ace_blp_get_viz_finish_defaults());
    }
}

function ace_blp_get_viz_size_defaults() {
    return [
        ['key' => 'small',  'name' => 'Small',       'desc' => 'Under 500 sq ft',     'icon' => 'fas fa-seedling',      'multiplier' => '0.6', 'prompt_detail' => 'Small backyard under 500 sq ft. Compact, efficient design maximizing every inch.'],
        ['key' => 'medium', 'name' => 'Medium',      'desc' => '500 - 1,000 sq ft',   'icon' => 'fas fa-tree',          'multiplier' => '1.0', 'prompt_detail' => 'Medium backyard 500-1,000 sq ft. Standard-sized features with comfortable spacing.'],
        ['key' => 'large',  'name' => 'Large',       'desc' => '1,000 - 1,500 sq ft', 'icon' => 'fas fa-tree',          'multiplier' => '1.4', 'prompt_detail' => 'Large backyard 1,000-1,500 sq ft. Generous features with distinct zones.'],
        ['key' => 'xl',     'name' => 'Extra Large',  'desc' => 'Over 1,500 sq ft',    'icon' => 'fas fa-mountain-sun',  'multiplier' => '1.8', 'prompt_detail' => 'Extra-large backyard over 1,500 sq ft. Resort-scale with multiple entertainment zones.'],
    ];
}

function ace_blp_get_viz_feature_defaults() {
    return [
        ['key' => 'patio',       'name' => 'Patio / Hardscape', 'icon' => 'fas fa-border-all',        'detail' => 'Beautiful paver patio with clean installation and defined edges.', 'exclude' => 'No new patio, hardscape, or paver area', 'price_std' => 8000,  'price_prem' => 17000, 'price_lux' => 30000],
        ['key' => 'pool',        'name' => 'Swimming Pool',     'icon' => 'fas fa-person-swimming',    'detail' => 'Swimming pool with clear turquoise water, coping, and underwater LED lighting. Add raised spa with spillover if Premium or Luxury.', 'exclude' => 'No swimming pool, spa, hot tub, or standalone water feature', 'price_std' => 38000, 'price_prem' => 60000, 'price_lux' => 90000],
        ['key' => 'kitchen',     'name' => 'Outdoor Kitchen',   'icon' => 'fas fa-utensils',           'detail' => 'Outdoor kitchen island with stone veneer base, granite countertop, built-in grill. Add pizza oven if Luxury.', 'exclude' => 'No outdoor kitchen, BBQ island, or cooking area', 'price_std' => 12000, 'price_prem' => 30000, 'price_lux' => 60000],
        ['key' => 'fire',        'name' => 'Fire Feature',      'icon' => 'fas fa-fire',               'detail' => 'Fire pit with stacked stone surround and fire glass, surrounded by lounge seating. Linear fire table if Luxury.', 'exclude' => 'No fire pit, fireplace, or fire table', 'price_std' => 1500,  'price_prem' => 6000,  'price_lux' => 12000],
        ['key' => 'landscaping', 'name' => 'Landscaping',       'icon' => 'fas fa-leaf',               'detail' => 'Professional drought-tolerant landscape design with mature plantings, uplighting on trees, clean mulch beds.', 'exclude' => 'Keep existing landscaping unless a selected feature requires changes to the planting in that specific area', 'price_std' => 4000,  'price_prem' => 12000, 'price_lux' => 23000],
        ['key' => 'cover',       'name' => 'Patio Cover',       'icon' => 'fas fa-tent-arrows-down',   'detail' => 'Patio cover or pergola with integrated lighting. Solid roof if Premium/Luxury.', 'exclude' => 'No patio cover, pergola, cabana, or shade structure', 'price_std' => 4000,  'price_prem' => 9000,  'price_lux' => 15000],
        ['key' => 'lighting',    'name' => 'Lighting',          'icon' => 'fas fa-lightbulb',          'detail' => 'Comprehensive landscape lighting - path lights, uplights on trees, LED strips, string lights.', 'exclude' => 'No added landscape lighting or accent lighting fixtures', 'price_std' => 1500,  'price_prem' => 5000,  'price_lux' => 8000],
        ['key' => 'turf',        'name' => 'Artificial Turf',   'icon' => 'fas fa-seedling',           'detail' => 'Premium artificial turf section, lush green, with clean edge transitions to hardscape.', 'exclude' => 'No artificial turf replacement', 'price_std' => 2500,  'price_prem' => 6000,  'price_lux' => 12000],
    ];
}

function ace_blp_get_viz_finish_defaults() {
    return [
        ['key' => 'standard', 'name' => 'Standard', 'desc' => 'Quality materials, clean design. Professional results without the premium price tag.', 'badge' => '',            'prompt_detail' => 'Quality but standard materials. Concrete or basic pavers, prefab elements, standard landscaping. Clean and professional, not extravagant.'],
        ['key' => 'premium',  'name' => 'Premium',  'desc' => 'Upgraded materials throughout - travertine, granite, mature landscaping, LED lighting package.', 'badge' => 'Most Popular', 'prompt_detail' => 'Upgraded materials throughout. Travertine pavers, custom stone veneer, granite countertops, professional landscape with mature plants, LED lighting package. Polished and impressive.'],
        ['key' => 'luxury',   'name' => 'Luxury',   'desc' => 'No-expense-spared. Natural stone, infinity edge, full kitchen pavilion, architectural lighting. Magazine-quality.', 'badge' => '',            'prompt_detail' => 'No-expense-spared materials and custom features. Premium natural stone, infinity/vanishing-edge pool, full kitchen pavilion, architectural lighting design, specimen landscaping. Magazine-quality.'],
    ];
}

// Register rewrite rules
add_action('init', 'ace_blp_register_rewrite');
function ace_blp_register_rewrite() {
    $settings = get_option('ace_backyard_lp_settings', array());
    $slug = !empty($settings['slug']) ? sanitize_title($settings['slug']) : 'backyard';
    add_rewrite_rule('^' . $slug . '/?$', 'index.php?ace_backyard_lp=1', 'top');
    add_rewrite_tag('%ace_backyard_lp%', '1');
}

// Template include
add_filter('template_include', 'ace_blp_template_include');
function ace_blp_template_include($template) {
    if (get_query_var('ace_backyard_lp')) {
        $settings = get_option('ace_backyard_lp_settings', array());
        if (empty($settings['enabled'])) {
            return $template;
        }
        return ACE_BLP_PATH . 'templates/landing-page.php';
    }
    return $template;
}

// Helper function to check if on LP page
function is_ace_backyard_lp() {
    return (bool) get_query_var('ace_backyard_lp');
}

// One-time cleanup: strip accumulated backslash escaping from all ace_* options
function ace_blp_fix_slashed_options() {
    global $wpdb;

    // 1. Fix the main settings array (ace_backyard_lp_settings)
    $settings = get_option('ace_backyard_lp_settings');
    if (is_array($settings)) {
        $cleaned = ace_blp_deep_unslash($settings);
        if ($cleaned !== $settings) {
            update_option('ace_backyard_lp_settings', $cleaned);
        }
    }

    // 2. Fix all individual ace_viz_* and ace_visualizer_* text options
    $option_names = $wpdb->get_col(
        "SELECT option_name FROM {$wpdb->options}
         WHERE option_name LIKE 'ace_viz_%'
            OR option_name LIKE 'ace_visualizer_%'
            OR option_name LIKE 'ace_style_%'"
    );

    foreach ($option_names as $name) {
        $val = get_option($name);
        if (is_string($val) && strpos($val, '\\') !== false) {
            $clean = ace_blp_strip_deep_slashes($val);
            if ($clean !== $val) {
                update_option($name, $clean);
            }
        } elseif (is_array($val)) {
            $clean = ace_blp_deep_unslash($val);
            if ($clean !== $val) {
                update_option($name, $clean);
            }
        }
    }
}

// Recursively strip slashes from strings in nested arrays
function ace_blp_deep_unslash($data) {
    if (is_array($data)) {
        return array_map('ace_blp_deep_unslash', $data);
    }
    if (is_string($data)) {
        return ace_blp_strip_deep_slashes($data);
    }
    return $data;
}

// Strip multiple layers of backslash escaping until stable
function ace_blp_strip_deep_slashes($str) {
    $prev = '';
    $current = $str;
    // Keep stripping until no more changes (handles \\\\' → \\' → \' → ')
    while ($current !== $prev) {
        $prev = $current;
        $current = wp_unslash($current);
    }
    return $current;
}

// Initialize classes
add_action('init', function() {
    new ACE_BLP_Admin();
    new ACE_BLP_Frontend();
    new ACE_BLP_Leads();

    // Run DB upgrade check once
    if (!get_option('ace_blp_db_version') || get_option('ace_blp_db_version') < 2) {
        ace_blp_maybe_upgrade_db();
        update_option('ace_blp_db_version', 2);
    }

    // Run content upgrade - force-overwrite projects/reviews from defaults
    if (!get_option('ace_blp_content_version') || get_option('ace_blp_content_version') < 3) {
        ace_blp_upgrade_content();
        update_option('ace_blp_content_version', 3);
    }

    // Migrate to mode toggle + DB-backed viz options (v4)
    if (!get_option('ace_blp_content_version') || get_option('ace_blp_content_version') < 4) {
        ace_blp_migrate_viz_options();
        update_option('ace_blp_content_version', 4);
    }

    // Fix turf icon: fa-grass is FA6 Pro-only, replace with fa-seedling (v5)
    if (get_option('ace_blp_content_version') < 5) {
        $viz_features = get_option('ace_viz_features', []);
        if (is_array($viz_features)) {
            foreach ($viz_features as &$f) {
                if (($f['key'] ?? '') === 'turf' && ($f['icon'] ?? '') === 'fas fa-grass') {
                    $f['icon'] = 'fas fa-seedling';
                }
            }
            unset($f);
            update_option('ace_viz_features', $viz_features);
        }
        update_option('ace_blp_content_version', 5);
    }

    // One-time cleanup: fix backslash-corrupted option values (v6)
    if (get_option('ace_blp_content_version') < 6) {
        ace_blp_fix_slashed_options();
        update_option('ace_blp_content_version', 6);
    }
});
