/**
 * ACE Backyard Landing Page - Admin Script
 */
(function($) {
    'use strict';

    // ==================== Tab Navigation ====================
    document.addEventListener('DOMContentLoaded', function() {
        var tabs = document.querySelectorAll('.tab-btn');
        var panels = document.querySelectorAll('.tab-panel');

        // Check URL hash for active tab
        var hash = window.location.hash.replace('#', '');
        if (hash) {
            activateTab(hash);
        }

        tabs.forEach(function(btn) {
            btn.addEventListener('click', function() {
                activateTab(this.dataset.tab);
                window.location.hash = this.dataset.tab;
            });
        });

        function activateTab(tabId) {
            tabs.forEach(function(t) { t.classList.remove('active'); });
            panels.forEach(function(p) { p.classList.remove('active'); });
            var btn = document.querySelector('.tab-btn[data-tab="' + tabId + '"]');
            var panel = document.getElementById('tab-' + tabId);
            if (btn) btn.classList.add('active');
            if (panel) panel.classList.add('active');
        }
    });

    // ==================== Mode Toggle (Calculator / Visualizer) ====================
    $(document).on('change', 'input[name="ace_active_mode"]', function() {
        var mode = $(this).val();
        $('.ace-mode-option').removeClass('active');
        $(this).closest('.ace-mode-option').addClass('active');

        if (mode === 'calculator') {
            $('#calc-settings-panel').show();
            $('#viz-settings-panel').hide();
        } else {
            $('#calc-settings-panel').hide();
            $('#viz-settings-panel').show();
            // Auto-check the hidden viz-enabled checkbox
            $('#viz-enabled').prop('checked', true);
        }
    });

    // ==================== Media Uploader ====================
    $(document).on('click', '.upload-image-btn', function(e) {
        e.preventDefault();
        var button = $(this);
        var field = button.closest('.media-field');
        var frame = wp.media({
            title: 'Select Image',
            multiple: false,
            library: { type: 'image' }
        });
        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            field.find('input[type="hidden"]').val(attachment.url);
            field.find('.media-preview').html('<img src="' + attachment.url + '" style="max-width:200px;">');
            field.find('.remove-image-btn').show();
        });
        frame.open();
    });

    $(document).on('click', '.remove-image-btn', function(e) {
        e.preventDefault();
        var field = $(this).closest('.media-field');
        field.find('input[type="hidden"]').val('');
        field.find('.media-preview').html('');
        $(this).hide();
    });

    // ==================== Repeater Fields ====================

    // Add item - clone last item and clear values
    $(document).on('click', '.repeater-add', function() {
        var repeater = $(this).closest('.repeater');
        var items = repeater.find('.repeater-items');
        var lastItem = items.find('.repeater-item').last();
        var newIndex = items.find('.repeater-item').length;

        var clone = lastItem.clone();

        // Update all input names to new index
        clone.find('input, textarea, select').each(function() {
            var name = $(this).attr('name');
            if (name) {
                name = name.replace(/\[\d+\]/, '[' + newIndex + ']');
                $(this).attr('name', name);
            }
            // Clear values for text inputs
            if ($(this).is('input[type="text"], input[type="number"], input[type="email"], textarea')) {
                $(this).val('');
            }
            if ($(this).is('input[type="hidden"]')) {
                $(this).val('');
            }
            // Reset checkboxes
            if ($(this).is('input[type="checkbox"]')) {
                $(this).prop('checked', false);
            }
        });

        // Clear media preview
        clone.find('.media-preview').html('');
        clone.find('.remove-image-btn').hide();

        clone.attr('data-index', newIndex);
        items.append(clone);
    });

    // Remove item
    $(document).on('click', '.repeater-remove', function() {
        var repeater = $(this).closest('.repeater');
        var items = repeater.find('.repeater-item');
        if (items.length <= 1) {
            alert('You must keep at least one item.');
            return;
        }
        if (confirm('Remove this item?')) {
            $(this).closest('.repeater-item').remove();
            reindexRepeater(repeater);
        }
    });

    // Re-index repeater items after add/remove/sort
    function reindexRepeater(repeater) {
        repeater.find('.repeater-item').each(function(index) {
            $(this).attr('data-index', index);
            $(this).find('input, textarea, select').each(function() {
                var name = $(this).attr('name');
                if (name) {
                    name = name.replace(/\[\d+\]/, '[' + index + ']');
                    $(this).attr('name', name);
                }
            });
        });
    }

    // Make repeaters sortable via jQuery UI
    $(document).ready(function() {
        $('.repeater-items, .viz-repeater-items').sortable({
            handle: '.repeater-handle',
            placeholder: 'ui-sortable-placeholder',
            update: function() {
                var container = $(this).closest('.repeater, .viz-repeater');
                if (container.hasClass('repeater')) {
                    reindexRepeater(container);
                } else {
                    reindexVizRepeater(container);
                }
            }
        });
    });

    // ==================== Viz Repeater CRUD ====================

    // Add viz repeater item
    $(document).on('click', '.viz-repeater-add', function() {
        var repeater = $(this).closest('.viz-repeater');
        var items = repeater.find('.viz-repeater-items');
        var lastItem = items.find('.viz-repeater-item').last();
        var newIndex = items.find('.viz-repeater-item').length;
        var clone = lastItem.clone();

        clone.find('input, textarea').each(function() {
            if ($(this).is('input[type="text"], input[type="number"], textarea')) {
                $(this).val('');
            }
        });

        clone.attr('data-index', newIndex);
        items.append(clone);
    });

    // Remove viz repeater item
    $(document).on('click', '.viz-repeater-remove', function() {
        var repeater = $(this).closest('.viz-repeater');
        var items = repeater.find('.viz-repeater-item');
        if (items.length <= 1) {
            alert('You must keep at least one item.');
            return;
        }
        if (confirm('Remove this item?')) {
            $(this).closest('.viz-repeater-item').remove();
            reindexVizRepeater(repeater);
        }
    });

    // Re-index viz repeater items
    function reindexVizRepeater(repeater) {
        repeater.find('.viz-repeater-item').each(function(index) {
            $(this).attr('data-index', index);
        });
    }

    // Collect viz repeater data as JSON array
    function collectVizRepeater(type) {
        var items = [];
        $('.viz-repeater[data-type="' + type + '"] .viz-repeater-item').each(function() {
            var item = {};
            $(this).find('[data-field]').each(function() {
                var field = $(this).data('field');
                var val = $(this).val();
                // Preserve numbers as numbers for price/multiplier fields
                if ($(this).attr('type') === 'number') {
                    val = parseFloat(val) || 0;
                }
                item[field] = val;
            });
            items.push(item);
        });
        return JSON.stringify(items);
    }

    // ==================== Leads Management ====================

    // Load leads data via AJAX
    function loadLeads() {
        var source = $('#leads-filter-source').val() || '';
        var dateFrom = $('#leads-filter-from').val() || '';
        var dateTo = $('#leads-filter-to').val() || '';

        $.get(ajaxurl, {
            action: 'ace_blp_get_leads',
            source: source,
            date_from: dateFrom,
            date_to: dateTo
        }, function(response) {
            if (response.success) {
                renderLeadsTable(response.data);
                updateLeadsStats(response.data);
            }
        });
    }

    // Render leads into the table
    function renderLeadsTable(leads) {
        var tbody = $('#leads-table-body');
        tbody.empty();

        if (leads.length === 0) {
            tbody.append('<tr><td colspan="9" style="text-align:center;padding:20px;">No leads found.</td></tr>');
            return;
        }

        leads.forEach(function(lead) {
            var calcDisplay = '';
            if (lead.calculator_data) {
                try {
                    var calc = JSON.parse(lead.calculator_data);
                    calcDisplay = (calc.size || '') + ' | ' + (calc.features ? calc.features.join(', ') : '') + ' | ' + (calc.finish || '') + ' | ' + (calc.price || '');
                } catch(e) {
                    calcDisplay = lead.calculator_data;
                }
            }

            var row = '<tr>' +
                '<td>' + escHtml(lead.created_at) + '</td>' +
                '<td><strong>' + escHtml(lead.name) + '</strong></td>' +
                '<td>' + escHtml(lead.phone) + '</td>' +
                '<td>' + escHtml(lead.email || '-') + '</td>' +
                '<td>' + escHtml(lead.source) + '</td>' +
                '<td class="calc-data">' + escHtml(calcDisplay || '-') + '</td>' +
                '<td>' + escHtml(lead.message || '-') + '</td>' +
                '<td><input type="checkbox" class="contacted-toggle" data-id="' + lead.id + '"' + (lead.contacted == 1 ? ' checked' : '') + '></td>' +
                '<td><button type="button" class="button button-small delete-lead" data-id="' + lead.id + '">Delete</button></td>' +
                '</tr>';
            tbody.append(row);
        });
    }

    // Update stats counters
    function updateLeadsStats(leads) {
        $('#stat-total').text(leads.length);

        var now = new Date();
        var weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
        var monthAgo = new Date(now.getFullYear(), now.getMonth(), 1);

        var thisWeek = leads.filter(function(l) { return new Date(l.created_at) >= weekAgo; }).length;
        var thisMonth = leads.filter(function(l) { return new Date(l.created_at) >= monthAgo; }).length;

        $('#stat-week').text(thisWeek);
        $('#stat-month').text(thisMonth);
    }

    // Escape HTML for safe rendering
    function escHtml(str) {
        if (!str) return '';
        var div = document.createElement('div');
        div.textContent = str;
        return div.innerHTML;
    }

    // Filter button click
    $(document).on('click', '#leads-filter-btn', function() {
        loadLeads();
    });

    // Delete lead
    $(document).on('click', '.delete-lead', function() {
        if (!confirm('Delete this lead?')) return;
        var id = $(this).data('id');
        $.post(ajaxurl, {
            action: 'ace_blp_delete_lead',
            lead_id: id,
            _nonce: aceBlpAdmin.nonce
        }, function(response) {
            if (response.success) loadLeads();
        });
    });

    // Toggle contacted status
    $(document).on('change', '.contacted-toggle', function() {
        var id = $(this).data('id');
        var contacted = $(this).is(':checked') ? 1 : 0;
        $.post(ajaxurl, {
            action: 'ace_blp_toggle_contacted',
            lead_id: id,
            contacted: contacted,
            _nonce: aceBlpAdmin.nonce
        });
    });

    // Load leads when Leads tab is activated
    $(document).on('click', '.tab-btn[data-tab="leads"]', function() {
        loadLeads();
    });

    // Also load leads if the page loads directly on leads tab
    $(document).ready(function() {
        if (window.location.hash === '#leads') {
            loadLeads();
        }
    });

    // ==================== AI Visualizer Settings ====================

    // Save visualizer settings via AJAX
    $(document).on('click', '#viz-save-settings', function() {
        var btn = $(this);
        var result = $('#viz-save-result');
        btn.prop('disabled', true).text('Saving...');
        result.text('');

        var apiKey = $('#viz-api-key').val();
        // Don't send masked placeholder
        if (apiKey && apiKey.indexOf('****') === 0) {
            apiKey = '';
        }

        $.post(ajaxurl, {
            action: 'ace_save_visualizer_settings',
            _nonce: aceBlpAdmin.nonce,
            active_mode: $('input[name="ace_active_mode"]:checked').val() || 'calculator',
            enabled: $('#viz-enabled').is(':checked') ? 1 : 0,
            api_key: apiKey,
            model: $('#viz-model').val(),
            base_prompt: $('#viz-base-prompt').val(),
            max_retries: $('#viz-max-retries').val(),
            daily_limit: $('#viz-daily-limit').val(),
            lead_gate: $('#viz-lead-gate').val(),
            style_modern_image: $('#viz-style-modern-image').val(),
            style_mediterranean_image: $('#viz-style-mediterranean-image').val(),
            style_tropical_image: $('#viz-style-tropical-image').val(),
            style_classic_image: $('#viz-style-classic-image').val(),
            admin_email: $('#viz-admin-email').val(),
            phone: $('#viz-phone').val(),
            promo_popup: $('#viz-promo-popup').is(':checked') ? 1 : 0,
            promo_scroll_pct: $('#viz-promo-scroll-pct').val(),
            // Viz repeater arrays (JSON)
            viz_sizes: collectVizRepeater('viz_sizes'),
            viz_features: collectVizRepeater('viz_features'),
            viz_finishes: collectVizRepeater('viz_finishes'),
            // Text fields
            promo_title: $('#viz-promo-title').val(),
            promo_text: $('#viz-promo-text').val(),
            promo_bullet1: $('#viz-promo-bullet1').val(),
            promo_bullet2: $('#viz-promo-bullet2').val(),
            promo_cta: $('#viz-promo-cta').val(),
            promo_dismiss: $('#viz-promo-dismiss').val(),
            promo_fine: $('#viz-promo-fine').val(),
            hero_btn: $('#viz-hero-btn').val(),
            hero_sub: $('#viz-hero-sub').val(),
            step1_title: $('#viz-step1-title').val(),
            step1_subtitle: $('#viz-step1-subtitle').val(),
            step2_title: $('#viz-step2-title').val(),
            step2_subtitle: $('#viz-step2-subtitle').val(),
            step3_title: $('#viz-step3-title').val(),
            step3_subtitle: $('#viz-step3-subtitle').val(),
            step4_title: $('#viz-step4-title').val(),
            step4_subtitle: $('#viz-step4-subtitle').val(),
            step5_title: $('#viz-step5-title').val(),
            step5_subtitle: $('#viz-step5-subtitle').val(),
            upload_text: $('#viz-upload-text').val(),
            upload_btn: $('#viz-upload-btn').val(),
            upload_tip: $('#viz-upload-tip').val(),
            success_msg: $('#viz-success-msg').val(),
            start_btn: $('#viz-start-btn').val(),
            gate_title: $('#viz-gate-title').val(),
            gate_subtitle: $('#viz-gate-subtitle').val(),
            gate_btn: $('#viz-gate-btn').val(),
            result_title: $('#viz-result-title').val(),
            price_label: $('#viz-price-label').val(),
            price_disclaimer: $('#viz-price-disclaimer').val(),
            cta_btn: $('#viz-cta-btn').val(),
            retry_text: $('#viz-retry-text').val(),
            label_before: $('#viz-label-before').val(),
            label_after: $('#viz-label-after').val(),
            loading_messages: $('#viz-loading-messages').val(),
            loading_tips: $('#viz-loading-tips').val()
        }, function(response) {
            btn.prop('disabled', false).text('Save Visualizer Settings');
            if (response.success) {
                result.html('<span style="color:#46b450;">Settings saved.</span>');
                // If key was just saved, update the masked display
                if (apiKey) {
                    $('#viz-api-key').val('************************************');
                }
            } else {
                result.html('<span style="color:#dc3232;">Error: ' + escHtml(response.data) + '</span>');
            }
            setTimeout(function() { result.text(''); }, 5000);
        }).fail(function() {
            btn.prop('disabled', false).text('Save Visualizer Settings');
            result.html('<span style="color:#dc3232;">Request failed. Please try again.</span>');
        });
    });

    // Test API connection
    $(document).on('click', '#viz-test-connection', function() {
        var btn = $(this);
        var result = $('#viz-test-result');
        btn.prop('disabled', true).text('Testing...');
        result.text('');

        $.post(ajaxurl, {
            action: 'ace_test_api_connection',
            _nonce: aceBlpAdmin.nonce
        }, function(response) {
            btn.prop('disabled', false).text('Test Connection');
            if (response.success) {
                result.html('<span style="color:#46b450;">' + escHtml(response.data) + '</span>');
            } else {
                result.html('<span style="color:#dc3232;">' + escHtml(response.data) + '</span>');
            }
        }).fail(function() {
            btn.prop('disabled', false).text('Test Connection');
            result.html('<span style="color:#dc3232;">Request failed.</span>');
        });
    });

    // Reset prompt to default
    $(document).on('click', '#viz-reset-prompt', function() {
        if (!confirm('Reset the prompt to the default template?')) return;
        var defaultPrompt = 'You are editing a real backyard photograph into a professionally remodeled outdoor living space. The output must be an ultra-photorealistic photograph - indistinguishable from a real image shot by an architectural photographer on-site. Not a render, not a 3D visualization, not AI-generated looking. A real photograph of a real completed project.\n\nABSOLUTELY PROHIBITED - REJECT IMMEDIATELY IF PRESENT:\n- No text, logos, watermarks, labels, or UI elements anywhere\n- No people, pets, or animals\n- No floating objects or impossible shadows\n- No oversaturated/HDR colors - natural but appealing\n- No plastic-looking grass or materials - organic variation required\n- No perfectly symmetrical layouts - real design has intentional asymmetry\n- No cartoon, illustrated, or painterly style - this is a PHOTOGRAPH\n- No lens flare or artificial bokeh unless present in original\n\nPERSPECTIVE & STRUCTURE - NON-NEGOTIABLE:\nThis is a photo edit, NOT a new scene. The camera has NOT moved.\n- Every permanent structure stays PIXEL-PERFECT: house walls, roofline, windows, doors, garage, fences, property boundaries, neighboring buildings, sky beyond property.\n- Anchor points: TOP of fence/wall stays at exact same height and position. BOTTOM of house wall meets ground at the same point. HORIZON LINE does not shift. Roof angles unchanged.\n- Ground plane perspective must match original exactly - objects recede at the same rate.\n- If original shows wide-angle lens distortion, preserve that exact distortion in new elements.\n- Do NOT alter: house exterior, paint color, windows, doors, roof, neighbors\' property, utility infrastructure (AC units, electrical panels, gas meters, power lines, water spigots). These stay untouched.\n\n{ORIENTATION_INSTRUCTION}\n\nLIGHTING:\nRender in golden hour Southern California light - warm directional sunlight from the west, long soft shadows, warm color temperature (4500K). Apply consistently to ALL elements including existing structures. Specular highlights on stone, water, and metal must reflect this light source position. Detail preserved in both shadows and highlights - no blown-out sky.\n\nReplace overcast or dull sky with attractive blue sky and light wispy clouds.\n\nTRANSITION ZONES - WHERE NEW MEETS OLD:\nThis is where AI fails most. Get this right:\n- New landscaping meets existing fence: natural soil/mulch buffer, no hard geometric cuts\n- New patio meets house wall: realistic expansion joint or natural stone-to-wall connection, no floating edges\n- New grass meets existing concrete: organic edge with slight imperfection, not laser-cut\n- Ground level CONSISTENT throughout - new surfaces cannot float above or sink below existing ground plane\n- Existing drainage, slopes, or grade changes in original must be maintained\n\nMATERIAL REALISM:\n- Stone grain visible at correct scale, wood plank width realistic, concrete has subtle surface variation\n- Plants are Southern California appropriate: drought-tolerant, Mediterranean species, correct leaf size, natural growth patterns, NOT symmetrically arranged. Mix mature and younger plants.\n- NO brand-new showroom look. Subtle signs of well-maintained but real: slightly uneven stone joint, one plant slightly larger than others, natural color variation in pavers\n- Water: realistic surface with reflection of surroundings and sky, subtle caustic light patterns\n- Furniture: real shadows underneath, legs touching ground, correct material texture\n\nDESIGN SPECIFICATIONS:\n- Backyard size: {SIZE_DESCRIPTION}\n- Features to include: {FEATURES_LIST}\n- Quality level: {FINISH_DESCRIPTION}\n- Design style: {STYLE_NAME}\n\nSTYLE DETAILS:\n{STYLE_DETAILS}\n\nFEATURE DETAILS:\n{FEATURE_DETAILS}\n\nWHAT CAN BE REDESIGNED (part of the remodel):\n- All ground surfaces: lawn, patio, concrete, gravel, pavers, decking\n- Fences/railings: material, style, color, height (stay on same property boundary line)\n- Ground slope: regrade, level, terrace, retaining walls\n- Entry stairs to house: new material, width, railing, shape\n- All landscaping within property: plants, trees, beds, planters\n- Add structures: pergola, gazebo, shade sail, outdoor kitchen\n- Add pool, spa, fire pit, water features where space allows\n- All outdoor furniture, lighting, decorative elements\n\nCLEANUP - IMPROVE THE SHOT:\n- Remove power lines and utility poles from SKY\n- Screen AC units/gas meters behind tasteful landscaping (hide, don\'t remove)\n- Remove visible trash, clutter, old furniture, debris\n- Make trees/vegetation beyond fence look lush and healthy\n\nCAMERA: Canon EOS R5, 24-70mm f/2.8 at f/8. Full depth-of-field. Match original aspect ratio and resolution. Do not crop or extend the frame.\n\nThe transformation must be dramatic and aspirational - the viewer immediately wants this for their own backyard. But it must look REAL, like a contractor actually built this and a photographer came to shoot it.';
        $('#viz-base-prompt').val(defaultPrompt);
    });

})(jQuery);
