(function() {
    'use strict';

    // =============================================
    // COST CALCULATOR
    // =============================================

    // These functions are called via onclick in the HTML
    // They need to be on window scope

    window.selectSize = function(el) {
        document.querySelectorAll('.size-opt').forEach(function(opt) {
            opt.classList.remove('active');
        });
        el.classList.add('active');
        updateCalc();
    };

    window.toggleFeature = function(el) {
        el.classList.toggle('active');
        updateCalc();
    };

    window.selectFinish = function(el) {
        document.querySelectorAll('.finish-opt').forEach(function(opt) {
            opt.classList.remove('active');
        });
        el.classList.add('active');
        updateCalc();
    };

    function updateCalc() {
        // Get active size multiplier
        var activeSize = document.querySelector('.size-opt.active');
        var multiplier = activeSize ? parseFloat(activeSize.getAttribute('data-multiplier')) : 1.0;

        // Get active finish key
        var activeFinish = document.querySelector('.finish-opt.active');
        var finishKey = activeFinish ? activeFinish.getAttribute('data-key') : 'prem';

        // Sum active features for the selected finish level
        var total = 0;
        var activeFeatures = document.querySelectorAll('.calc-feature.active');
        activeFeatures.forEach(function(feat) {
            var price = parseFloat(feat.getAttribute('data-' + finishKey)) || 0;
            total += price;
        });

        // Apply size multiplier
        total = total * multiplier;

        // Calculate range (0.85 low, 1.15 high, rounded to nearest 1000)
        var low = Math.round(total * 0.85 / 1000) * 1000;
        var high = Math.round(total * 1.15 / 1000) * 1000;

        // Format as currency
        var priceEl = document.getElementById('calcPrice');
        if (priceEl) {
            if (low === 0 && high === 0) {
                priceEl.textContent = 'Select features above';
            } else {
                priceEl.textContent = '$' + low.toLocaleString() + ' - $' + high.toLocaleString();
            }
        }

        // Update hidden calculator_data field
        updateCalcData(activeSize, activeFeatures, activeFinish, low, high);
    }

    function updateCalcData(activeSize, activeFeatures, activeFinish, low, high) {
        var calcDataEl = document.getElementById('calcData');
        if (!calcDataEl) return;

        var data = {
            size: activeSize ? activeSize.querySelector('.size-name').textContent : '',
            features: [],
            finish: activeFinish ? activeFinish.querySelector('.finish-name').textContent : '',
            price: '$' + low.toLocaleString() + ' - $' + high.toLocaleString()
        };

        activeFeatures.forEach(function(feat) {
            var nameEl = feat.querySelector('.calc-feature-name');
            if (nameEl) data.features.push(nameEl.textContent);
        });

        calcDataEl.value = JSON.stringify(data);
    }

    // Initialize calculator on page load
    document.addEventListener('DOMContentLoaded', function() {
        // Run initial calculation with defaults
        if (document.querySelector('.calc-card')) {
            updateCalc();
        }
    });

    // =============================================
    // BEFORE/AFTER SLIDER
    // =============================================

    (function() {
        var slider = document.getElementById('baSlider');
        var handle = document.getElementById('baHandle');
        var after = document.getElementById('baAfter');

        if (!slider || !handle || !after) return;

        var isDragging = false;

        function setPosition(x) {
            var rect = slider.getBoundingClientRect();
            var pos = (x - rect.left) / rect.width;
            pos = Math.max(0, Math.min(1, pos));

            var percent = pos * 100;
            handle.style.left = percent + '%';
            after.style.clipPath = 'inset(0 0 0 ' + percent + '%)';
        }

        // Mouse events
        handle.addEventListener('mousedown', function(e) {
            e.preventDefault();
            isDragging = true;
        });

        document.addEventListener('mousemove', function(e) {
            if (!isDragging) return;
            setPosition(e.clientX);
        });

        document.addEventListener('mouseup', function() {
            isDragging = false;
        });

        // Touch events
        handle.addEventListener('touchstart', function(e) {
            e.preventDefault();
            isDragging = true;
        });

        document.addEventListener('touchmove', function(e) {
            if (!isDragging) return;
            setPosition(e.touches[0].clientX);
        });

        document.addEventListener('touchend', function() {
            isDragging = false;
        });

        // Click on container to jump
        slider.addEventListener('click', function(e) {
            if (e.target === handle || handle.contains(e.target)) return;
            setPosition(e.clientX);
        });
    })();

    // =============================================
    // FAQ ACCORDION
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var faqItems = document.querySelectorAll('.faq-item');

        faqItems.forEach(function(item) {
            var question = item.querySelector('.faq-q');
            if (!question) return;

            question.addEventListener('click', function() {
                var isOpen = item.classList.contains('open');

                // Close all other items
                faqItems.forEach(function(other) {
                    other.classList.remove('open');
                });

                // Toggle clicked item
                if (!isOpen) {
                    item.classList.add('open');
                }
            });
        });
    });

    // =============================================
    // FORM SUBMISSIONS (AJAX)
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var forms = document.querySelectorAll('.lp-form');

        forms.forEach(function(form) {
            form.addEventListener('submit', function(e) {
                e.preventDefault();

                var btn = form.querySelector('button[type="submit"]');
                if (btn.classList.contains('btn-loading')) return;

                // Validate required fields
                var requiredFields = form.querySelectorAll('[required]');
                var valid = true;
                requiredFields.forEach(function(field) {
                    if (!field.value.trim()) {
                        field.style.borderColor = '#e74c3c';
                        valid = false;
                    } else {
                        field.style.borderColor = '';
                    }
                });

                if (!valid) return;

                // Collect form data
                var formData = new FormData();
                formData.append('action', 'ace_backyard_submit_lead');

                // Add all form fields
                var inputs = form.querySelectorAll('input, textarea, select');
                inputs.forEach(function(input) {
                    if (input.name) {
                        formData.append(input.name, input.value);
                    }
                });

                // Show loading state
                btn.classList.add('btn-loading');

                // Submit via fetch
                fetch(ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(function(response) { return response.json(); })
                .then(function(data) {
                    btn.classList.remove('btn-loading');

                    if (data.success) {
                        // Show success message
                        var successEl = form.querySelector('.form-success');
                        if (successEl) {
                            successEl.style.display = 'flex';

                            // Hide after 3 seconds and reset form
                            setTimeout(function() {
                                successEl.style.display = 'none';
                                form.reset();

                                // Close popup if in popup
                                var popup = form.closest('.popup-overlay');
                                if (popup) {
                                    popup.classList.remove('active');
                                    document.body.style.overflow = '';
                                }
                            }, 3000);
                        }
                    } else {
                        // Show error
                        var msg = data.data && data.data.message ? data.data.message : 'Something went wrong. Please try again.';
                        alert(msg);
                    }
                })
                .catch(function() {
                    btn.classList.remove('btn-loading');
                    alert('Something went wrong. Please try again.');
                });
            });
        });
    });

    // =============================================
    // POPUP MODAL
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var overlay = document.getElementById('popupOverlay');
        if (!overlay) return;

        var closeBtn = overlay.querySelector('.popup-close');

        // Open popup
        document.querySelectorAll('[data-popup="quote"]').forEach(function(trigger) {
            trigger.addEventListener('click', function(e) {
                e.preventDefault();
                overlay.classList.add('active');
                document.body.style.overflow = 'hidden';
            });
        });

        // Close popup - X button
        if (closeBtn) {
            closeBtn.addEventListener('click', function() {
                overlay.classList.remove('active');
                document.body.style.overflow = '';
            });
        }

        // Close popup - backdrop click
        overlay.addEventListener('click', function(e) {
            if (e.target === overlay) {
                overlay.classList.remove('active');
                document.body.style.overflow = '';
            }
        });

        // Close popup - Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && overlay.classList.contains('active')) {
                overlay.classList.remove('active');
                document.body.style.overflow = '';
            }
        });
    });

    // =============================================
    // SCROLL NAVIGATION (active section tracking)
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var scrollNav = document.getElementById('scrollNav');
        if (!scrollNav) return;

        var sections = document.querySelectorAll('section[id]');
        var navLinks = document.querySelectorAll('.topbar-nav-link[href^="#"]');

        var sectionObserver = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    navLinks.forEach(function(link) {
                        link.classList.remove('active');
                    });
                    var activeLink = document.querySelector('.topbar-nav-link[href="#' + entry.target.id + '"]');
                    if (activeLink) {
                        activeLink.classList.add('active');
                    }
                }
            });
        }, { rootMargin: '-64px 0px -50% 0px' });

        sections.forEach(function(s) {
            sectionObserver.observe(s);
        });

        // Smooth scroll on click with offset
        navLinks.forEach(function(link) {
            link.addEventListener('click', function(e) {
                e.preventDefault();

                // Close visualizer flow popup if open (e.g. clicking "Calculator" nav link)
                var vizWasOpen = false;
                var flowPopup = document.getElementById('vizFlowPopup');
                if (flowPopup && flowPopup.classList.contains('visible') && typeof window.closeVisualizerPopup === 'function') {
                    window.closeVisualizerPopup();
                    vizWasOpen = true;
                }

                var targetId = link.getAttribute('href');
                var target = document.querySelector(targetId);
                if (target) {
                    var offset = 64; // single header row
                    // Delay scroll if popup was closing so animation finishes first
                    setTimeout(function() {
                        var top = target.getBoundingClientRect().top + window.pageYOffset - offset;
                        window.scrollTo({ top: top, behavior: 'smooth' });
                    }, vizWasOpen ? 350 : 0);
                }
            });
        });
    });

    // =============================================
    // SMOOTH SCROLL FOR ALL ANCHOR LINKS
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('a[href^="#"]:not(.topbar-nav-link)').forEach(function(link) {
            link.addEventListener('click', function(e) {
                var targetId = link.getAttribute('href');
                if (targetId === '#') return;

                var target = document.querySelector(targetId);
                if (target) {
                    e.preventDefault();
                    var offset = 64; // single header row
                    var top = target.getBoundingClientRect().top + window.pageYOffset - offset;
                    window.scrollTo({ top: top, behavior: 'smooth' });
                }
            });
        });
    });

    // =============================================
    // HAMBURGER MENU (mobile)
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var hamburger = document.getElementById('hamburgerBtn');
        var overlay = document.getElementById('mobileMenuOverlay');
        var closeBtn = document.getElementById('mobileMenuClose');
        if (!hamburger || !overlay) return;

        function openMenu() {
            overlay.classList.add('active');
            hamburger.classList.add('active');
            hamburger.setAttribute('aria-expanded', 'true');
            document.body.style.overflow = 'hidden';
        }
        function closeMenu() {
            overlay.classList.remove('active');
            hamburger.classList.remove('active');
            hamburger.setAttribute('aria-expanded', 'false');
            document.body.style.overflow = '';
        }

        hamburger.addEventListener('click', function() {
            if (overlay.classList.contains('active')) {
                closeMenu();
            } else {
                openMenu();
            }
        });

        if (closeBtn) closeBtn.addEventListener('click', closeMenu);

        // Close on backdrop click
        overlay.addEventListener('click', function(e) {
            if (e.target === overlay) closeMenu();
        });

        // Close on Escape
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && overlay.classList.contains('active')) closeMenu();
        });

        // Close on nav link click + smooth scroll
        overlay.querySelectorAll('.mobile-menu-link').forEach(function(link) {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                closeMenu();
                var targetId = link.getAttribute('href');
                var target = document.querySelector(targetId);
                if (target) {
                    setTimeout(function() {
                        var offset = 64;
                        var top = target.getBoundingClientRect().top + window.pageYOffset - offset;
                        window.scrollTo({ top: top, behavior: 'smooth' });
                    }, 300);
                }
            });
        });
    });

    // =============================================
    // PROOF STRIP - Pause on hover
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var track = document.querySelector('.proof-track');
        if (!track) return;

        track.addEventListener('mouseenter', function() {
            track.style.animationPlayState = 'paused';
        });
        track.addEventListener('mouseleave', function() {
            track.style.animationPlayState = 'running';
        });
    });

    // =============================================
    // GENERIC CAROUSEL ENGINE (Infinite Loop)
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var carousels = document.querySelectorAll('.carousel-wrap');

        carousels.forEach(function(wrap) {
            var track = wrap.querySelector('.carousel-track');
            var originalSlides = Array.prototype.slice.call(wrap.querySelectorAll('.carousel-slide'));
            var prevBtn = wrap.querySelector('.carousel-prev');
            var nextBtn = wrap.querySelector('.carousel-next');
            var dotsContainer = wrap.querySelector('.carousel-dots');

            if (!track || originalSlides.length === 0) return;

            var autoplayDelay = parseInt(wrap.getAttribute('data-autoplay')) || 5000;
            var desktopVisible = parseInt(wrap.getAttribute('data-desktop')) || 3;
            var tabletVisible = parseInt(wrap.getAttribute('data-tablet')) || 2;
            var mobileVisible = parseInt(wrap.getAttribute('data-mobile')) || 1;
            var totalOriginal = originalSlides.length;

            var currentIndex = 0; // index into original slides
            var autoplayTimer = null;
            var isHovered = false;
            var isTransitioning = false;
            var clonesBefore = 0;
            var allSlides = [];

            function getVisibleCount() {
                var w = window.innerWidth;
                if (w <= 767) return mobileVisible;
                if (w <= 1023) return tabletVisible;
                return desktopVisible;
            }

            function getMaxIndex() {
                var visible = getVisibleCount();
                return Math.max(0, totalOriginal - visible);
            }

            // Build clones for infinite loop
            function buildClones() {
                // Remove old clones
                var oldClones = track.querySelectorAll('.carousel-clone');
                for (var c = 0; c < oldClones.length; c++) {
                    track.removeChild(oldClones[c]);
                }

                var visible = getVisibleCount();
                clonesBefore = visible;

                // Clone last N slides and prepend
                for (var i = 0; i < visible; i++) {
                    var idx = totalOriginal - visible + i;
                    if (idx < 0) idx = 0;
                    var clone = originalSlides[idx].cloneNode(true);
                    clone.classList.add('carousel-clone');
                    clone.setAttribute('aria-hidden', 'true');
                    track.insertBefore(clone, track.firstChild);
                }

                // Clone first N slides and append
                for (var j = 0; j < visible; j++) {
                    var idx2 = j % totalOriginal;
                    var clone2 = originalSlides[idx2].cloneNode(true);
                    clone2.classList.add('carousel-clone');
                    clone2.setAttribute('aria-hidden', 'true');
                    track.appendChild(clone2);
                }

                allSlides = Array.prototype.slice.call(track.querySelectorAll('.carousel-slide'));
            }

            function updateSlideWidths() {
                var visible = getVisibleCount();
                var pct = 100 / visible;
                allSlides.forEach(function(slide) {
                    slide.style.width = pct + '%';
                });
            }

            function jumpTo(index, animate) {
                var visible = getVisibleCount();
                var trackIndex = index + clonesBefore;
                var offset = -(trackIndex * (100 / visible));
                if (!animate) {
                    track.style.transition = 'none';
                }
                track.style.transform = 'translateX(' + offset + '%)';
                if (!animate) {
                    // Force reflow
                    track.offsetHeight; // eslint-disable-line no-unused-expressions
                    track.style.transition = '';
                }
            }

            function goTo(index, animate) {
                if (isTransitioning) return;
                if (typeof animate === 'undefined') animate = true;
                currentIndex = index;

                var visible = getVisibleCount();
                var trackIndex = currentIndex + clonesBefore;
                var offset = -(trackIndex * (100 / visible));

                if (animate) {
                    isTransitioning = true;
                    track.style.transition = 'transform 0.5s ease';
                } else {
                    track.style.transition = 'none';
                }
                track.style.transform = 'translateX(' + offset + '%)';

                if (!animate) {
                    track.offsetHeight; // eslint-disable-line no-unused-expressions
                    track.style.transition = '';
                }

                updateDots();
            }

            // After transition ends, silently reposition if we're on a clone
            track.addEventListener('transitionend', function() {
                isTransitioning = false;
                var maxIdx = getMaxIndex();
                if (currentIndex > maxIdx) {
                    currentIndex = 0;
                    jumpTo(0, false);
                    updateDots();
                } else if (currentIndex < 0) {
                    currentIndex = maxIdx;
                    jumpTo(maxIdx, false);
                    updateDots();
                }
            });

            function next() {
                var maxIdx = getMaxIndex();
                if (currentIndex >= maxIdx) {
                    // Animate to clone, then snap back
                    currentIndex = maxIdx + 1;
                    goTo(currentIndex);
                } else {
                    goTo(currentIndex + 1);
                }
            }

            function prev() {
                if (currentIndex <= 0) {
                    currentIndex = -1;
                    goTo(currentIndex);
                } else {
                    goTo(currentIndex - 1);
                }
            }

            function buildDots() {
                if (!dotsContainer) return;
                dotsContainer.innerHTML = '';
                var totalDots = getMaxIndex() + 1;
                for (var i = 0; i < totalDots; i++) {
                    var dot = document.createElement('button');
                    dot.className = 'carousel-dot' + (i === currentIndex ? ' active' : '');
                    dot.type = 'button';
                    dot.setAttribute('aria-label', 'Go to slide ' + (i + 1));
                    dot.setAttribute('data-index', i);
                    dot.addEventListener('click', function() {
                        goTo(parseInt(this.getAttribute('data-index')));
                        resetAutoplay();
                    });
                    dotsContainer.appendChild(dot);
                }
            }

            function updateDots() {
                if (!dotsContainer) return;
                var realIndex = currentIndex;
                var maxIdx = getMaxIndex();
                if (realIndex < 0) realIndex = maxIdx;
                if (realIndex > maxIdx) realIndex = 0;
                var dots = dotsContainer.querySelectorAll('.carousel-dot');
                dots.forEach(function(dot, i) {
                    dot.classList.toggle('active', i === realIndex);
                });
            }

            function startAutoplay() {
                stopAutoplay();
                autoplayTimer = setInterval(function() {
                    if (!isHovered) {
                        next();
                    }
                }, autoplayDelay);
            }

            function stopAutoplay() {
                if (autoplayTimer) {
                    clearInterval(autoplayTimer);
                    autoplayTimer = null;
                }
            }

            function resetAutoplay() {
                stopAutoplay();
                startAutoplay();
            }

            // Events
            if (prevBtn) {
                prevBtn.addEventListener('click', function() {
                    prev();
                    resetAutoplay();
                });
            }
            if (nextBtn) {
                nextBtn.addEventListener('click', function() {
                    next();
                    resetAutoplay();
                });
            }

            wrap.addEventListener('mouseenter', function() { isHovered = true; });
            wrap.addEventListener('mouseleave', function() { isHovered = false; });

            // Touch swipe support
            var touchStartX = 0;
            var touchEndX = 0;
            wrap.addEventListener('touchstart', function(e) {
                touchStartX = e.changedTouches[0].screenX;
            }, { passive: true });
            wrap.addEventListener('touchend', function(e) {
                touchEndX = e.changedTouches[0].screenX;
                var diff = touchStartX - touchEndX;
                if (Math.abs(diff) > 50) {
                    if (diff > 0) {
                        next();
                    } else {
                        prev();
                    }
                    resetAutoplay();
                }
            }, { passive: true });

            // Init
            function init() {
                buildClones();
                updateSlideWidths();
                jumpTo(0, false);
                buildDots();
                startAutoplay();
            }

            init();

            // Recalculate on resize
            var resizeTimeout;
            window.addEventListener('resize', function() {
                clearTimeout(resizeTimeout);
                resizeTimeout = setTimeout(function() {
                    buildClones();
                    updateSlideWidths();
                    if (currentIndex > getMaxIndex()) currentIndex = getMaxIndex();
                    jumpTo(currentIndex, false);
                    buildDots();
                }, 200);
            });
        });
    });

    // =============================================
    // PROBLEM CARDS – staggered fade-in on scroll
    // =============================================

    document.addEventListener('DOMContentLoaded', function() {
        var cards = document.querySelectorAll('.problem-card');
        if (!cards.length) return;

        var observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    var card = entry.target;
                    var idx = Array.prototype.indexOf.call(cards, card);
                    setTimeout(function() {
                        card.classList.add('visible');
                    }, idx * 200);
                    observer.unobserve(card);
                }
            });
        }, { threshold: 0.2 });

        cards.forEach(function(card) {
            observer.observe(card);
        });
    });

})();
