<?php
/**
 * ACE Backyard AI Visualizer - Backend Logic
 * Phase 1: Admin + Backend Infrastructure
 *
 * Handles: API key encryption, image security, prompt building,
 * Gemini API calls, price calculation, rate limiting, lead management,
 * email notifications, cron cleanup, DB migration, asset enqueue.
 */

if (!defined('ABSPATH')) exit;

// ============================================================
// PARAMETER MAPS (copied exactly from workplan section 3.2)
// ============================================================

function ace_viz_get_size_map() {
    $db_sizes = get_option('ace_viz_sizes', []);
    if (!empty($db_sizes) && is_array($db_sizes)) {
        $map = [];
        foreach ($db_sizes as $s) {
            if (!empty($s['key'])) {
                $map[$s['key']] = $s['prompt_detail'] ?? '';
            }
        }
        if (!empty($map)) return $map;
    }
    // Fallback
    return [
        'small'  => 'Small backyard under 500 sq ft. Compact, efficient design maximizing every inch.',
        'medium' => 'Medium backyard 500-1,000 sq ft. Standard-sized features with comfortable spacing.',
        'large'  => 'Large backyard 1,000-1,500 sq ft. Generous features with distinct zones.',
        'xl'     => 'Extra-large backyard over 1,500 sq ft. Resort-scale with multiple entertainment zones.'
    ];
}

function ace_viz_get_feature_map() {
    $db_features = get_option('ace_viz_features', []);
    if (!empty($db_features) && is_array($db_features)) {
        $map = [];
        foreach ($db_features as $f) {
            if (!empty($f['key'])) {
                $map[$f['key']] = [
                    'name'    => $f['name'] ?? '',
                    'detail'  => $f['detail'] ?? '',
                    'exclude' => $f['exclude'] ?? '',
                    'prices'  => [
                        'standard' => intval($f['price_std'] ?? 0),
                        'premium'  => intval($f['price_prem'] ?? 0),
                        'luxury'   => intval($f['price_lux'] ?? 0),
                    ]
                ];
            }
        }
        if (!empty($map)) return $map;
    }
    // Fallback
    return [
        'patio' => [
            'name'    => 'Patio / Hardscape',
            'detail'  => 'Beautiful paver patio with clean installation and defined edges.',
            'exclude' => 'No new patio, hardscape, or paver area',
            'prices'  => ['standard' => 8000, 'premium' => 17000, 'luxury' => 30000]
        ],
        'pool' => [
            'name'    => 'Swimming Pool',
            'detail'  => 'Swimming pool with clear turquoise water, coping, and underwater LED lighting. Add raised spa with spillover if Premium or Luxury.',
            'exclude' => 'No swimming pool, spa, hot tub, or standalone water feature',
            'prices'  => ['standard' => 38000, 'premium' => 60000, 'luxury' => 90000]
        ],
        'kitchen' => [
            'name'    => 'Outdoor Kitchen',
            'detail'  => 'Outdoor kitchen island with stone veneer base, granite countertop, built-in grill. Add pizza oven if Luxury.',
            'exclude' => 'No outdoor kitchen, BBQ island, or cooking area',
            'prices'  => ['standard' => 12000, 'premium' => 30000, 'luxury' => 60000]
        ],
        'fire' => [
            'name'    => 'Fire Feature',
            'detail'  => 'Fire pit with stacked stone surround and fire glass, surrounded by lounge seating. Linear fire table if Luxury.',
            'exclude' => 'No fire pit, fireplace, or fire table',
            'prices'  => ['standard' => 1500, 'premium' => 6000, 'luxury' => 12000]
        ],
        'landscaping' => [
            'name'    => 'Landscaping',
            'detail'  => 'Professional drought-tolerant landscape design with mature plantings, uplighting on trees, clean mulch beds.',
            'exclude' => 'Keep existing landscaping unless a selected feature requires changes to the planting in that specific area',
            'prices'  => ['standard' => 4000, 'premium' => 12000, 'luxury' => 23000]
        ],
        'cover' => [
            'name'    => 'Patio Cover',
            'detail'  => 'Patio cover or pergola with integrated lighting. Solid roof if Premium/Luxury.',
            'exclude' => 'No patio cover, pergola, cabana, or shade structure',
            'prices'  => ['standard' => 4000, 'premium' => 9000, 'luxury' => 15000]
        ],
        'lighting' => [
            'name'    => 'Lighting',
            'detail'  => 'Comprehensive landscape lighting - path lights, uplights on trees, LED strips, string lights.',
            'exclude' => 'No added landscape lighting or accent lighting fixtures',
            'prices'  => ['standard' => 1500, 'premium' => 5000, 'luxury' => 8000]
        ],
        'turf' => [
            'name'    => 'Artificial Turf',
            'detail'  => 'Premium artificial turf section, lush green, with clean edge transitions to hardscape.',
            'exclude' => 'No artificial turf replacement',
            'prices'  => ['standard' => 2500, 'premium' => 6000, 'luxury' => 12000]
        ]
    ];
}

function ace_viz_get_finish_map() {
    $db_finishes = get_option('ace_viz_finishes', []);
    if (!empty($db_finishes) && is_array($db_finishes)) {
        $map = [];
        foreach ($db_finishes as $f) {
            if (!empty($f['key'])) {
                $map[$f['key']] = [
                    'name'   => ($f['name'] ?? '') . (!empty($f['desc']) ? ' - ' . $f['desc'] : ''),
                    'detail' => $f['prompt_detail'] ?? ''
                ];
            }
        }
        if (!empty($map)) return $map;
    }
    // Fallback
    return [
        'standard' => [
            'name'   => 'Standard - Quality materials, clean design',
            'detail' => 'Quality but standard materials. Concrete or basic pavers, prefab elements, standard landscaping. Clean and professional, not extravagant.'
        ],
        'premium' => [
            'name'   => 'Premium - Upgraded materials & features',
            'detail' => 'Upgraded materials throughout. Travertine pavers, custom stone veneer, granite countertops, professional landscape with mature plants, LED lighting package. Polished and impressive.'
        ],
        'luxury' => [
            'name'   => 'Luxury - High-end, custom everything',
            'detail' => 'No-expense-spared materials and custom features. Premium natural stone, infinity/vanishing-edge pool, full kitchen pavilion, architectural lighting design, specimen landscaping. Magazine-quality.'
        ]
    ];
}

function ace_viz_get_style_map() {
    return [
        'modern' => [
            'name'   => 'Modern Contemporary',
            'detail' => 'Clean geometric lines, dark porcelain or concrete pavers, horizontal wood slat fencing, minimal palette (charcoal, white, warm wood), architectural planting with agaves and grasses, matte-black accents. Rectangular pool shape.'
        ],
        'mediterranean' => [
            'name'   => 'Mediterranean / Tuscan',
            'detail' => 'Warm tumbled travertine, stucco columns, terracotta accents, wrought-iron details, lavender, olive trees, bougainvillea, cypress, wood pergola with climbing vines. Soft organic pool curves.'
        ],
        'tropical' => [
            'name'   => 'Tropical Resort',
            'detail' => 'Natural stone, freeform pool with rock waterfall, palms, bird of paradise, monstera, plumeria, banana plants, thatched palapa, tiki-torch lighting. Dense layered vegetation.'
        ],
        'classic' => [
            'name'   => 'Classic Traditional',
            'detail' => 'Cream travertine French pattern, traditional furniture in dark wicker with cream cushions, string lights, mature shade trees, hedges, roses, hydrangeas, cedar fence. Warm family-friendly.'
        ]
    ];
}

function ace_viz_get_default_prompt() {
    return 'You are editing a real backyard photograph into a professionally remodeled outdoor living space. The output must be an ultra-photorealistic photograph - indistinguishable from a real image shot by an architectural photographer on-site. Not a render, not a 3D visualization, not AI-generated looking. A real photograph of a real completed project.

ABSOLUTELY PROHIBITED - REJECT IMMEDIATELY IF PRESENT:
- No text, logos, watermarks, labels, or UI elements anywhere
- No people, pets, or animals
- No floating objects or impossible shadows
- No oversaturated/HDR colors - natural but appealing
- No plastic-looking grass or materials - organic variation required
- No perfectly symmetrical layouts - real design has intentional asymmetry
- No cartoon, illustrated, or painterly style - this is a PHOTOGRAPH
- No lens flare or artificial bokeh unless present in original

PERSPECTIVE & STRUCTURE - NON-NEGOTIABLE:
This is a photo edit, NOT a new scene. The camera has NOT moved.
- Every permanent structure stays PIXEL-PERFECT: house walls, roofline, windows, doors, garage, fences, property boundaries, neighboring buildings, sky beyond property.
- Anchor points: TOP of fence/wall stays at exact same height and position. BOTTOM of house wall meets ground at the same point. HORIZON LINE does not shift. Roof angles unchanged.
- Ground plane perspective must match original exactly - objects recede at the same rate.
- If original shows wide-angle lens distortion, preserve that exact distortion in new elements.
- Do NOT alter: house exterior, paint color, windows, doors, roof, neighbors\' property, utility infrastructure (AC units, electrical panels, gas meters, power lines, water spigots). These stay untouched.

{ORIENTATION_INSTRUCTION}

LIGHTING:
Render in golden hour Southern California light - warm directional sunlight from the west, long soft shadows, warm color temperature (4500K). Apply consistently to ALL elements including existing structures. Specular highlights on stone, water, and metal must reflect this light source position. Detail preserved in both shadows and highlights - no blown-out sky.

Replace overcast or dull sky with attractive blue sky and light wispy clouds.

TRANSITION ZONES - WHERE NEW MEETS OLD:
This is where AI fails most. Get this right:
- New landscaping meets existing fence: natural soil/mulch buffer, no hard geometric cuts
- New patio meets house wall: realistic expansion joint or natural stone-to-wall connection, no floating edges
- New grass meets existing concrete: organic edge with slight imperfection, not laser-cut
- Ground level CONSISTENT throughout - new surfaces cannot float above or sink below existing ground plane
- Existing drainage, slopes, or grade changes in original must be maintained

MATERIAL REALISM:
- Stone grain visible at correct scale, wood plank width realistic, concrete has subtle surface variation
- Plants are Southern California appropriate: drought-tolerant, Mediterranean species, correct leaf size, natural growth patterns, NOT symmetrically arranged. Mix mature and younger plants.
- NO brand-new showroom look. Subtle signs of well-maintained but real: slightly uneven stone joint, one plant slightly larger than others, natural color variation in pavers
- Water: realistic surface with reflection of surroundings and sky, subtle caustic light patterns
- Furniture: real shadows underneath, legs touching ground, correct material texture

DESIGN SPECIFICATIONS:
- Backyard size: {SIZE_DESCRIPTION}
- Features to include: {FEATURES_LIST}
- Quality level: {FINISH_DESCRIPTION}
- Design style: {STYLE_NAME}

STYLE DETAILS:
{STYLE_DETAILS}

FEATURE DETAILS:
{FEATURE_DETAILS}

EXCLUDED FEATURES - CLIENT DID NOT REQUEST THESE:
{EXCLUDED_FEATURES}

WHAT CAN BE REDESIGNED (part of the remodel):
- All ground surfaces: lawn, patio, concrete, gravel, pavers, decking
- Fences/railings: material, style, color, height (stay on same property boundary line)
- Ground slope: regrade, level, terrace, retaining walls
- Entry stairs to house: new material, width, railing, shape
- All landscaping within property: plants, trees, beds, planters
- Add structures: pergola, gazebo, shade sail, outdoor kitchen
- Add pool, spa, fire pit, water features where space allows
- All outdoor furniture, lighting, decorative elements

CLEANUP - IMPROVE THE SHOT:
- Remove power lines and utility poles from SKY
- Screen AC units/gas meters behind tasteful landscaping (hide, don\'t remove)
- Remove visible trash, clutter, old furniture, debris
- Make trees/vegetation beyond fence look lush and healthy

CAMERA: Canon EOS R5, 24-70mm f/2.8 at f/8. Full depth-of-field. Match original aspect ratio and resolution. Do not crop or extend the frame.

The transformation must be dramatic and aspirational - the viewer immediately wants this for their own backyard. But it must look REAL, like a contractor actually built this and a photographer came to shoot it.';
}


// ============================================================
// API KEY ENCRYPTION / DECRYPTION (AES-256-CBC)
// ============================================================

function ace_save_api_key($raw_key) {
    if (empty($raw_key)) return;

    // Validate format (Gemini keys start with AIza)
    if (strpos($raw_key, 'AIza') !== 0 || strlen($raw_key) < 30) {
        return new WP_Error('invalid_key', 'Invalid API key format');
    }

    $iv = substr(AUTH_SALT, 0, 16);
    $encrypted = openssl_encrypt($raw_key, 'AES-256-CBC', AUTH_KEY, 0, $iv);
    update_option('ace_gemini_api_key_enc', $encrypted);
    wp_cache_delete('ace_gemini_api_key_enc', 'options');
}

function ace_get_api_key() {
    $encrypted = get_option('ace_gemini_api_key_enc');
    if (!$encrypted) return false;

    $iv = substr(AUTH_SALT, 0, 16);
    $decrypted = openssl_decrypt($encrypted, 'AES-256-CBC', AUTH_KEY, 0, $iv);
    return $decrypted ?: false;
}


// ============================================================
// IMAGE UPLOAD SECURITY (7-Layer Validation)
// ============================================================

function ace_secure_image_upload($file) {
    // Layer 1: Extension whitelist
    $allowed_ext = ['jpg', 'jpeg', 'png', 'webp'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, $allowed_ext, true)) {
        return ['error' => 'Only JPG, PNG, and WebP files are allowed.'];
    }

    // Layer 2: MIME from file content (not headers - headers can be spoofed)
    if (function_exists('finfo_open')) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $real_mime = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
    } elseif (function_exists('mime_content_type')) {
        $real_mime = mime_content_type($file['tmp_name']);
    } else {
        $img_info = @getimagesize($file['tmp_name']);
        $real_mime = $img_info ? $img_info['mime'] : '';
    }
    
    if (!in_array($real_mime, ['image/jpeg', 'image/png', 'image/webp'], true)) {
        return ['error' => 'Invalid file type detected.'];
    }

    // Layer 3: Binary image verification
    $image_info = @getimagesize($file['tmp_name']);
    if ($image_info === false) {
        return ['error' => 'File is not a valid image.'];
    }

    // Layer 4: Dimension limits
    if ($image_info[0] < 400 || $image_info[1] < 300) {
        return ['error' => 'Image too small. Minimum 400x300 pixels.'];
    }
    if ($image_info[0] > 8000 || $image_info[1] > 8000) {
        return ['error' => 'Image too large. Maximum 8000x8000 pixels.'];
    }

    // Layer 5: File size
    if ($file['size'] > 10 * 1024 * 1024) {
        return ['error' => 'File too large. Maximum 10MB.'];
    }

    // Layer 6: Re-encode (strips ALL embedded payloads, EXIF, scripts)
    $src = @imagecreatefromstring(file_get_contents($file['tmp_name']));
    if (!$src) {
        return ['error' => 'Cannot process image.'];
    }
    $clean_path = wp_tempnam('ace_clean_') . '.jpg';
    imagejpeg($src, $clean_path, 90);
    imagedestroy($src);

    // Layer 7: Verify re-encoded output
    if (!@getimagesize($clean_path)) {
        @unlink($clean_path);
        return ['error' => 'Image processing failed.'];
    }

    return ['success' => true, 'path' => $clean_path, 'mime' => 'image/jpeg'];
}


// ============================================================
// IMAGE NORMALIZATION (Orientation Handling)
// ============================================================

function ace_normalize_image($file_path) {
    $info = getimagesize($file_path);
    $w = $info[0];
    $h = $info[1];
    $ratio = $w / $h;
    $target = 1024;

    $src = imagecreatefromstring(file_get_contents($file_path));

    if ($ratio < 0.8) {
        // Portrait -> crop center-bottom to square
        $crop_size = $w;
        $crop_y = round(($h - $crop_size) * 0.6);
        $dst = imagecreatetruecolor($target, $target);
        imagecopyresampled($dst, $src, 0, 0, 0, $crop_y, $target, $target, $crop_size, $crop_size);
        $orientation = 'portrait-cropped';
    } elseif ($ratio >= 0.8 && $ratio <= 1.2) {
        // Square -> resize
        $dst = imagecreatetruecolor($target, $target);
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $target, $target, $w, $h);
        $orientation = 'square';
    } else {
        // Landscape -> resize keeping ratio
        $new_w = $target;
        $new_h = round($target / $ratio);
        $dst = imagecreatetruecolor($new_w, $new_h);
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $new_w, $new_h, $w, $h);
        $orientation = 'landscape';
    }

    $out_path = wp_tempnam('ace_norm_') . '.jpg';
    imagejpeg($dst, $out_path, 90);
    imagedestroy($src);
    imagedestroy($dst);

    return ['path' => $out_path, 'orientation' => $orientation];
}

function ace_get_orientation_instruction($orientation) {
    $map = [
        'landscape'        => 'Photo is landscape. Maintain wide horizontal composition.',
        'portrait-cropped' => 'Photo was cropped to square. Fill the entire square frame.',
        'square'           => 'Photo is square. Maintain square composition.'
    ];
    return $map[$orientation] ?? $map['landscape'];
}


// ============================================================
// INPUT VALIDATION (Whitelist-Based, Strict Comparison)
// ============================================================

function ace_validate_quiz_answers($data) {
    $valid_sizes    = array_keys(ace_viz_get_size_map());
    $valid_features = array_keys(ace_viz_get_feature_map());
    $valid_finishes = array_keys(ace_viz_get_finish_map());
    $valid_styles   = array_keys(ace_viz_get_style_map());

    $size = sanitize_text_field($data['size'] ?? '');
    if (!in_array($size, $valid_sizes, true)) {
        return new WP_Error('invalid_size', 'Invalid yard size');
    }

    $features_raw = is_array($data['features'] ?? null)
        ? $data['features']
        : json_decode($data['features'] ?? '[]', true);
    if (!is_array($features_raw) || empty($features_raw)) {
        return new WP_Error('invalid_features', 'Select at least one feature');
    }
    $features = array_values(array_intersect($features_raw, $valid_features));
    if (empty($features)) {
        return new WP_Error('invalid_features', 'Invalid feature selection');
    }

    $finish = sanitize_text_field($data['finish'] ?? '');
    if (!in_array($finish, $valid_finishes, true)) {
        return new WP_Error('invalid_finish', 'Invalid finish level');
    }

    $style = sanitize_text_field($data['style'] ?? '');
    if (!in_array($style, $valid_styles, true)) {
        return new WP_Error('invalid_style', 'Invalid style');
    }

    return [
        'size'     => $size,
        'features' => $features,
        'finish'   => $finish,
        'style'    => $style
    ];
}


// ============================================================
// PROMPT BUILDER
// ============================================================

function ace_build_prompt($answers, $orientation) {
    $size_map    = ace_viz_get_size_map();
    $feature_map = ace_viz_get_feature_map();
    $finish_map  = ace_viz_get_finish_map();
    $style_map   = ace_viz_get_style_map();

    $base = get_option('ace_visualizer_base_prompt', ace_viz_get_default_prompt());
    if (empty(trim($base))) {
        $base = ace_viz_get_default_prompt();
    }

    $size_desc = $size_map[$answers['size']] ?? $size_map['medium'];

    $selected        = array_flip($answers['features']);
    $feature_names   = [];
    $feature_details = [];
    $excluded_lines  = [];
    foreach ($feature_map as $key => $feat) {
        if (isset($selected[$key])) {
            $feature_names[]   = $feat['name'];
            $feature_details[] = $feat['detail'];
        } else {
            $excluded_lines[] = '- ' . $feat['exclude'];
        }
    }

    // Build exclusion block (empty string if user selected all 8)
    $excluded_text = '';
    if (!empty($excluded_lines)) {
        $excluded_text = "The client did NOT request the following. Do NOT add these as standalone features. However, if a SELECTED feature naturally requires minor adjustments to these areas (e.g., adding a pool requires removing some existing plants), that is acceptable.\n"
            . implode("\n", $excluded_lines);
    }

    $finish = $finish_map[$answers['finish']] ?? $finish_map['standard'];
    $style  = $style_map[$answers['style']]   ?? $style_map['modern'];
    $orient = ace_get_orientation_instruction($orientation);

    return str_replace(
        [
            '{ORIENTATION_INSTRUCTION}',
            '{SIZE_DESCRIPTION}',
            '{FEATURES_LIST}',
            '{FINISH_DESCRIPTION}',
            '{STYLE_NAME}',
            '{STYLE_DETAILS}',
            '{FEATURE_DETAILS}',
            '{EXCLUDED_FEATURES}'
        ],
        [
            $orient,
            $size_desc,
            implode(', ', $feature_names),
            $finish['detail'],
            $style['name'],
            $style['detail'],
            implode("\n", $feature_details),
            $excluded_text
        ],
        $base
    );
}


// ============================================================
// GEMINI API CALLER
// ============================================================

function ace_generate_image($image_base64, $prompt) {
    $api_key = ace_get_api_key();
    if (!$api_key) {
        error_log('[ACE Visualizer] API key not configured');
        return ['error' => 'Service not configured. Please try again later.'];
    }

    $model = get_option('ace_visualizer_model', 'gemini-2.5-flash-image');
    $url   = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$api_key}";

    $body = [
        'contents' => [[
            'parts' => [
                ['inlineData' => ['mimeType' => 'image/jpeg', 'data' => $image_base64]],
                ['text' => $prompt]
            ]
        ]],
        'generationConfig' => [
            'responseModalities' => ['TEXT', 'IMAGE']
        ]
    ];

    $response = wp_remote_post($url, [
        'headers' => ['Content-Type' => 'application/json'],
        'body'    => json_encode($body),
        'timeout' => 60
    ]);

    if (is_wp_error($response)) {
        error_log('[ACE Visualizer] Network error: ' . $response->get_error_message());
        return ['error' => 'Connection error. Please try again.'];
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $data = json_decode(wp_remote_retrieve_body($response), true);

    if ($status_code !== 200) {
        error_log('[ACE Visualizer] API error (' . $status_code . '): ' . ($data['error']['message'] ?? 'Unknown'));
        if ($status_code === 429) {
            return ['error' => 'Service is busy. Please try again in a minute.'];
        }
        return ['error' => 'Something went wrong. Please try again.'];
    }

    // Extract generated image from response
    foreach ($data['candidates'][0]['content']['parts'] ?? [] as $part) {
        if (isset($part['inlineData'])) {
            return [
                'success'      => true,
                'image_base64' => $part['inlineData']['data'],
                'mime_type'    => $part['inlineData']['mimeType']
            ];
        }
    }

    error_log('[ACE Visualizer] No image in API response');
    return ['error' => 'Could not generate design. Please try again.'];
}


// ============================================================
// GENERATED IMAGE STORAGE (Secure Directory + Optimized Size)
// ============================================================

function ace_save_generated_image($base64_data) {
    $upload_dir = wp_upload_dir();
    $viz_dir    = $upload_dir['basedir'] . '/ace-visualizer/';

    // Create protected directory on first use
    if (!file_exists($viz_dir)) {
        wp_mkdir_p($viz_dir);

        // .htaccess: block PHP execution, allow only image files
        file_put_contents(
            $viz_dir . '.htaccess',
            "Options -Indexes\n" .
            "<FilesMatch \"\.php$\">\n    Deny from all\n</FilesMatch>\n" .
            "<FilesMatch \"\.(jpg|jpeg|png|webp)$\">\n    Allow from all\n</FilesMatch>"
        );
        file_put_contents($viz_dir . 'index.php', '<?php // Silence is golden');
    }

    $image_data = base64_decode($base64_data);
    if ($image_data === false) {
        return ['error' => 'Invalid image data'];
    }

    // Optimize: resize to max 1600px, quality 85
    $src = imagecreatefromstring($image_data);
    if (!$src) {
        return ['error' => 'Cannot process generated image'];
    }

    $w = imagesx($src);
    $h = imagesy($src);
    if ($w > 1600) {
        $new_h = round($h * (1600 / $w));
        $dst = imagecreatetruecolor(1600, $new_h);
        imagecopyresampled($dst, $src, 0, 0, 0, 0, 1600, $new_h, $w, $h);
        imagedestroy($src);
        $src = $dst;
    }

    $filename = wp_generate_uuid4() . '.jpg'; // Non-predictable name
    $filepath = $viz_dir . $filename;
    imagejpeg($src, $filepath, 85);
    imagedestroy($src);

    if (!@getimagesize($filepath)) {
        @unlink($filepath);
        return ['error' => 'Generated image is corrupted'];
    }

    return [
        'success' => true,
        'url'     => $upload_dir['baseurl'] . '/ace-visualizer/' . $filename,
        'path'    => $filepath
    ];
}


// ============================================================
// PRICE CALCULATOR
// ============================================================

function ace_calculate_price($size, $features, $finish) {
    // Build size multipliers from DB
    $size_multipliers = ['small' => 0.6, 'medium' => 1.0, 'large' => 1.4, 'xl' => 1.8]; // fallback
    $db_sizes = get_option('ace_viz_sizes', []);
    if (!empty($db_sizes) && is_array($db_sizes)) {
        $size_multipliers = [];
        foreach ($db_sizes as $s) {
            if (!empty($s['key'])) {
                $size_multipliers[$s['key']] = floatval($s['multiplier'] ?? 1.0);
            }
        }
    }
    $feature_map = ace_viz_get_feature_map();

    $multiplier = $size_multipliers[$size] ?? 1.0;
    $total = 0;

    foreach ($features as $feature_key) {
        if (isset($feature_map[$feature_key]['prices'][$finish])) {
            $total += $feature_map[$feature_key]['prices'][$finish];
        }
    }

    $total *= $multiplier;

    $low  = (int) (round($total * 0.85 / 1000) * 1000);
    $high = (int) (round($total * 1.15 / 1000) * 1000);

    return ['low' => max($low, 1000), 'high' => max($high, 2000)];
}


// ============================================================
// RATE LIMITING (Double Layer: per-minute + per-day)
// ============================================================

function ace_check_rate_limit() {
    $daily_limit  = (int) get_option('ace_visualizer_daily_limit', 5);
    $minute_limit = 3;

    $ip      = ace_get_client_ip();
    $ip_hash = md5($ip . AUTH_SALT); // Hash IPs, don't store raw

    // Per-minute check
    $min_key   = 'ace_viz_min_' . $ip_hash;
    $min_count = (int) get_transient($min_key);
    if ($min_count >= $minute_limit) return false;
    set_transient($min_key, $min_count + 1, 60);

    // Per-day check
    $day_key   = 'ace_viz_day_' . $ip_hash;
    $day_count = (int) get_transient($day_key);
    if ($day_count >= $daily_limit) return false;
    set_transient($day_key, $day_count + 1, DAY_IN_SECONDS);

    return true;
}

function ace_get_client_ip() {
    foreach (['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'] as $h) {
        if (!empty($_SERVER[$h])) {
            $ip = trim(explode(',', $_SERVER[$h])[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                return $ip;
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}


// ============================================================
// LEAD SAVER (Sanitized + Prepared Statements)
// ============================================================

function ace_save_lead_secure($data) {
    global $wpdb;

    $name  = sanitize_text_field(substr($data['name'], 0, 100));
    $phone = preg_replace('/[^0-9+\-() ]/', '', substr($data['phone'], 0, 20));
    $email = sanitize_email($data['email'] ?? '');

    $digits_only = preg_replace('/\D/', '', $phone);
    if (strlen($digits_only) < 7) {
        return new WP_Error('invalid_phone', 'Invalid phone number');
    }

    $result = $wpdb->insert(
        $wpdb->prefix . 'ace_backyard_leads',
        [
            'name'                => $name,
            'phone'               => $phone,
            'email'               => $email,
            'source'              => 'visualizer',
            'visualizer_data'     => wp_json_encode($data['quiz_data'] ?? []),
            'original_image_url'  => esc_url_raw($data['original_url'] ?? ''),
            'generated_image_url' => esc_url_raw($data['generated_url'] ?? ''),
            'created_at'          => current_time('mysql')
        ],
        ['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s']
    );

    if ($result === false) {
        error_log('[ACE Visualizer] Failed to save lead: ' . $wpdb->last_error);
        return new WP_Error('db_error', 'Failed to save lead');
    }

    return $wpdb->insert_id;
}


// ============================================================
// EMAIL NOTIFICATIONS
// ============================================================

function ace_notify_admin_visualizer($data) {
    $settings   = get_option('ace_backyard_lp_settings', []);
    $viz_email  = get_option('ace_visualizer_admin_email', '');
    $to         = !empty($viz_email) ? $viz_email : (!empty($settings['email']) ? $settings['email'] : get_option('admin_email'));

    $name  = esc_html($data['name'] ?? '');
    $phone = esc_html($data['phone'] ?? '');
    $email = esc_html($data['email'] ?? '');
    $quiz  = $data['quiz_data'] ?? [];

    $size_map    = ace_viz_get_size_map();
    $feature_map = ace_viz_get_feature_map();
    $finish_map  = ace_viz_get_finish_map();
    $style_map   = ace_viz_get_style_map();

    $size_label    = $size_map[$quiz['size'] ?? ''] ?? 'Unknown';
    $finish_label  = isset($finish_map[$quiz['finish'] ?? '']) ? $finish_map[$quiz['finish']]['name'] : 'Unknown';
    $style_label   = isset($style_map[$quiz['style'] ?? '']) ? $style_map[$quiz['style']]['name'] : 'Unknown';

    $feature_labels = [];
    foreach ($quiz['features'] ?? [] as $f) {
        if (isset($feature_map[$f])) {
            $feature_labels[] = $feature_map[$f]['name'];
        }
    }

    $price_low  = isset($data['price_low'])  ? '$' . number_format($data['price_low'])  : '';
    $price_high = isset($data['price_high']) ? '$' . number_format($data['price_high']) : '';

    $subject = 'New AI Visualizer Lead: ' . $name;

    $body = '<html><body>';
    $body .= '<h2 style="color:#C6A355;">New AI Visualizer Lead</h2>';
    $body .= '<table style="border-collapse:collapse;width:100%;max-width:600px;">';
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Name</td><td style="padding:8px;border:1px solid #ddd;">' . $name . '</td></tr>';
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Phone</td><td style="padding:8px;border:1px solid #ddd;">' . $phone . '</td></tr>';
    if ($email) {
        $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Email</td><td style="padding:8px;border:1px solid #ddd;">' . $email . '</td></tr>';
    }
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;" colspan="2"><strong>Design Choices:</strong></td></tr>';
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Size</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($size_label) . '</td></tr>';
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Features</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html(implode(', ', $feature_labels)) . '</td></tr>';
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Finish</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($finish_label) . '</td></tr>';
    $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Style</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($style_label) . '</td></tr>';
    if ($price_low && $price_high) {
        $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Estimated Budget</td><td style="padding:8px;border:1px solid #ddd;">' . $price_low . ' - ' . $price_high . '</td></tr>';
    }
    $body .= '</table></body></html>';

    $headers = ['Content-Type: text/html; charset=UTF-8'];
    if ($email) {
        $headers[] = 'Reply-To: ' . sanitize_email($data['email']);
    }

    wp_mail($to, $subject, $body, $headers);
}

function ace_notify_visitor_visualizer($data) {
    $email = sanitize_email($data['email'] ?? '');
    if (empty($email)) return; // No email provided

    $name     = esc_html($data['name'] ?? 'there');
    $settings = get_option('ace_backyard_lp_settings', []);
    $phone    = get_option('ace_visualizer_phone', $settings['phone'] ?? '(310) 438-6866');

    $subject = 'Your ACE Backyard Design is Ready!';

    $body = '<html><body>';
    $body .= '<h2 style="color:#C6A355;">Your Custom Backyard Design</h2>';
    $body .= '<p>Hi ' . $name . ',</p>';
    $body .= '<p>Thanks for using our AI Backyard Visualizer! Your custom design is ready.</p>';
    if (!empty($data['generated_url'])) {
        $body .= '<p><a href="' . esc_url($data['generated_url']) . '" style="color:#C6A355;">View Your Design</a></p>';
    }
    $body .= '<p>Ready to make it real?</p>';
    $body .= '<p><strong>Book your free in-home consultation</strong> where our designer will create multiple custom concepts tailored to your space.</p>';
    $body .= '<p>Or call us: <a href="tel:' . preg_replace('/[^0-9+]/', '', $phone) . '">' . esc_html($phone) . '</a></p>';
    $body .= '<br><p style="color:#999;font-size:12px;">- ACE Design & Build</p>';
    $body .= '</body></html>';

    $headers = ['Content-Type: text/html; charset=UTF-8'];

    wp_mail($email, $subject, $body, $headers);
}


// ============================================================
// SECURE ERROR RESPONSE
// ============================================================

function ace_error_response($user_message, $log_message = '') {
    if ($log_message) {
        error_log('[ACE Visualizer] ' . $log_message);
    }
    wp_send_json_error(['message' => $user_message]);
}


// ============================================================
// FRONTEND AJAX: Generate Design
// ============================================================

add_action('wp_ajax_ace_generate_design', 'ace_handle_generate');
add_action('wp_ajax_nopriv_ace_generate_design', 'ace_handle_generate');

function ace_handle_generate() {
    // 1. Nonce verification
    if (!check_ajax_referer('ace_visualizer_nonce', 'nonce', false)) {
        wp_send_json_error(['message' => 'Security check failed. Please refresh.'], 403);
    }

    // WordPress adds magic quotes to $_POST — strip them
    $_POST = wp_unslash($_POST);

    // 2. Rate limit (3/min + X/day)
    if (!ace_check_rate_limit()) {
        $settings = get_option('ace_backyard_lp_settings', []);
        $phone = get_option('ace_visualizer_phone', $settings['phone'] ?? '(310) 438-6866');
        wp_send_json_error(['message' => 'Daily limit reached. Call us: ' . $phone], 429);
    }

    // 3. Honeypot
    if (!empty($_POST['website_url'])) {
        wp_send_json_error(['message' => 'Something went wrong.'], 400);
    }

    // 4. Validate quiz answers (whitelist + strict comparison)
    $answers = ace_validate_quiz_answers($_POST);
    if (is_wp_error($answers)) {
        wp_send_json_error(['message' => $answers->get_error_message()], 400);
    }

    // 5. Validate image (7 layers)
    if (empty($_FILES['image'])) {
        wp_send_json_error(['message' => 'No image provided.'], 400);
    }
    $image = ace_secure_image_upload($_FILES['image']);
    if (isset($image['error'])) {
        wp_send_json_error(['message' => $image['error']], 400);
    }

    // 6. Normalize image (orientation handling, resize to 1024px)
    $normalized = ace_normalize_image($image['path']);
    @unlink($image['path']); // Clean temp from step 5

    // 7. Build prompt from quiz answers + base template
    $prompt = ace_build_prompt($answers, $normalized['orientation']);

    // 8. Convert to base64 for API
    $image_base64 = base64_encode(file_get_contents($normalized['path']));
    @unlink($normalized['path']); // Clean temp from step 6

    // 9. Call Gemini API
    $start_time = microtime(true);
    $result = ace_generate_image($image_base64, $prompt);
    $generation_time = round((microtime(true) - $start_time) * 1000);

    if (isset($result['error'])) {
        ace_error_response($result['error']);
        return;
    }

    // 10. Save optimized image (max 1600px, quality 85, secure dir)
    $saved = ace_save_generated_image($result['image_base64']);
    if (isset($saved['error'])) {
        ace_error_response('Could not save design.', $saved['error']);
        return;
    }

    // 11. Calculate price
    $price = ace_calculate_price($answers['size'], $answers['features'], $answers['finish']);

    // 12. Return URL only - NEVER base64
    wp_send_json_success([
        'image_url'       => $saved['url'],
        'price_low'       => $price['low'],
        'price_high'      => $price['high'],
        'generation_time' => $generation_time
    ]);
}


// ============================================================
// FRONTEND AJAX: Save Visualizer Lead
// ============================================================

add_action('wp_ajax_ace_save_visualizer_lead', 'ace_handle_save_lead');
add_action('wp_ajax_nopriv_ace_save_visualizer_lead', 'ace_handle_save_lead');

function ace_handle_save_lead() {
    // Nonce verification
    if (!check_ajax_referer('ace_visualizer_nonce', 'nonce', false)) {
        wp_send_json_error(['message' => 'Security check failed. Please refresh.'], 403);
    }

    // WordPress adds magic quotes to $_POST — strip them
    $_POST = wp_unslash($_POST);

    // Honeypot
    if (!empty($_POST['website_url'])) {
        wp_send_json_error(['message' => 'Something went wrong.'], 400);
    }

    $name  = sanitize_text_field($_POST['name'] ?? '');
    $phone = sanitize_text_field($_POST['phone'] ?? '');

    if (empty($name) || empty($phone)) {
        wp_send_json_error(['message' => 'Name and phone are required.'], 400);
    }

    $quiz_data_raw = json_decode($_POST['quiz_data'] ?? '{}', true);

    $lead_data = [
        'name'          => $name,
        'phone'         => $phone,
        'email'         => sanitize_email($_POST['email'] ?? ''),
        'quiz_data'     => is_array($quiz_data_raw) ? $quiz_data_raw : [],
        'original_url'  => esc_url_raw($_POST['original_url'] ?? ''),
        'generated_url' => esc_url_raw($_POST['generated_url'] ?? ''),
        'price_low'     => intval($_POST['price_low'] ?? 0),
        'price_high'    => intval($_POST['price_high'] ?? 0),
    ];

    $result = ace_save_lead_secure($lead_data);
    if (is_wp_error($result)) {
        wp_send_json_error(['message' => 'Could not save your information. Please try again.'], 500);
    }

    // Send email notifications
    ace_notify_admin_visualizer($lead_data);
    ace_notify_visitor_visualizer($lead_data);

    wp_send_json_success(['message' => 'Thank you! Your design is unlocked.']);
}


// ============================================================
// ADMIN AJAX: Save Visualizer Settings
// ============================================================

add_action('wp_ajax_ace_save_visualizer_settings', 'ace_handle_save_visualizer_settings');

function ace_handle_save_visualizer_settings() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    if (!check_ajax_referer('ace_blp_admin_nonce', '_nonce', false)) {
        wp_send_json_error('Invalid nonce');
    }

    // WordPress adds magic quotes to $_POST — strip them once up front
    $_POST = wp_unslash($_POST);

    // Active mode (calculator / visualizer)
    $valid_modes = ['calculator', 'visualizer'];
    $mode = sanitize_text_field($_POST['active_mode'] ?? '');
    if (in_array($mode, $valid_modes, true)) {
        update_option('ace_active_mode', $mode);
        // Keep ace_visualizer_enabled in sync for backward compat
        update_option('ace_visualizer_enabled', $mode === 'visualizer' ? 1 : 0);
    }

    // Enable/Disable (legacy - only if no active_mode sent, since mode toggle handles it)
    if (empty($mode)) {
        update_option('ace_visualizer_enabled', intval($_POST['enabled'] ?? 0));
    }

    // API Key (encrypt if provided and not masked placeholder)
    $raw_key = sanitize_text_field($_POST['api_key'] ?? '');
    if (!empty($raw_key) && strpos($raw_key, '****') === false) {
        $save_result = ace_save_api_key($raw_key);
        if (is_wp_error($save_result)) {
            wp_send_json_error($save_result->get_error_message());
        }
    }

    // Model
    $valid_models = ['gemini-2.5-flash-image', 'gemini-3-pro-image-preview'];
    $model = sanitize_text_field($_POST['model'] ?? 'gemini-2.5-flash-image');
    if (in_array($model, $valid_models, true)) {
        update_option('ace_visualizer_model', $model);
    }

    // Base Prompt
    $base_prompt = wp_kses_post($_POST['base_prompt'] ?? '');
    if (!empty(trim($base_prompt))) {
        update_option('ace_visualizer_base_prompt', $base_prompt);
    }

    // Max Retries (0-3)
    $retries = max(0, min(3, intval($_POST['max_retries'] ?? 1)));
    update_option('ace_visualizer_max_retries', $retries);

    // Daily Limit (1-20)
    $limit = max(1, min(20, intval($_POST['daily_limit'] ?? 5)));
    update_option('ace_visualizer_daily_limit', $limit);

    // Lead Gate
    $valid_gates = ['blur', 'watermark', 'none'];
    $gate = sanitize_text_field($_POST['lead_gate'] ?? 'blur');
    if (in_array($gate, $valid_gates, true)) {
        update_option('ace_visualizer_lead_gate', $gate);
    }

    // Style Images (4 URLs)
    $styles = ['modern', 'mediterranean', 'tropical', 'classic'];
    foreach ($styles as $style) {
        $url = esc_url_raw($_POST['style_' . $style . '_image'] ?? '');
        update_option('ace_style_' . $style . '_image', $url);
    }

    // Admin Email
    $admin_email = sanitize_email($_POST['admin_email'] ?? '');
    update_option('ace_visualizer_admin_email', $admin_email);

    // Phone Number
    $phone = sanitize_text_field($_POST['phone'] ?? '');
    update_option('ace_visualizer_phone', $phone);

    // Promo Popup
    update_option('ace_visualizer_promo_popup', intval($_POST['promo_popup'] ?? 1));
    $promo_pct = max(5, min(90, intval($_POST['promo_scroll_pct'] ?? 15)));
    update_option('ace_visualizer_promo_scroll_pct', $promo_pct);

    // Text fields (sanitize and save each)
    $text_fields = [
        'promo_title', 'promo_text', 'promo_bullet1', 'promo_bullet2',
        'promo_cta', 'promo_dismiss', 'promo_fine',
        'hero_btn', 'hero_sub',
        'step1_title', 'step1_subtitle', 'step2_title', 'step2_subtitle',
        'step3_title', 'step3_subtitle', 'step4_title', 'step4_subtitle',
        'step5_title', 'step5_subtitle',
        'upload_text', 'upload_btn', 'upload_tip', 'success_msg', 'start_btn',
        'gate_title', 'gate_subtitle', 'gate_btn',
        'result_title', 'price_label', 'price_disclaimer',
        'cta_btn', 'retry_text', 'label_before', 'label_after',
    ];
    foreach ($text_fields as $tf) {
        if (isset($_POST[$tf])) {
            // Allow <em> tags in step titles
            $val = wp_kses($_POST[$tf], ['em' => []]);
            update_option('ace_viz_' . $tf, $val);
        }
    }

    // Loading messages/tips (multiline text)
    if (isset($_POST['loading_messages'])) {
        update_option('ace_viz_loading_messages', sanitize_textarea_field($_POST['loading_messages']));
    }
    if (isset($_POST['loading_tips'])) {
        update_option('ace_viz_loading_tips', sanitize_textarea_field($_POST['loading_tips']));
    }

    // Viz repeater arrays (JSON-encoded from admin JS)
    if (isset($_POST['viz_sizes'])) {
        $sizes = json_decode($_POST['viz_sizes'], true);
        if (is_array($sizes)) {
            $clean = [];
            foreach ($sizes as $s) {
                $clean[] = [
                    'key'           => sanitize_key($s['key'] ?? ''),
                    'name'          => sanitize_text_field($s['name'] ?? ''),
                    'desc'          => sanitize_text_field($s['desc'] ?? ''),
                    'icon'          => sanitize_text_field($s['icon'] ?? ''),
                    'multiplier'    => floatval($s['multiplier'] ?? 1.0),
                    'prompt_detail' => sanitize_textarea_field($s['prompt_detail'] ?? ''),
                ];
            }
            update_option('ace_viz_sizes', $clean);
        }
    }

    if (isset($_POST['viz_features'])) {
        $features = json_decode($_POST['viz_features'], true);
        if (is_array($features)) {
            $clean = [];
            foreach ($features as $f) {
                $clean[] = [
                    'key'        => sanitize_key($f['key'] ?? ''),
                    'name'       => sanitize_text_field($f['name'] ?? ''),
                    'icon'       => sanitize_text_field($f['icon'] ?? ''),
                    'detail'     => sanitize_textarea_field($f['detail'] ?? ''),
                    'exclude'    => sanitize_textarea_field($f['exclude'] ?? ''),
                    'price_std'  => intval($f['price_std'] ?? 0),
                    'price_prem' => intval($f['price_prem'] ?? 0),
                    'price_lux'  => intval($f['price_lux'] ?? 0),
                ];
            }
            update_option('ace_viz_features', $clean);
        }
    }

    if (isset($_POST['viz_finishes'])) {
        $finishes = json_decode($_POST['viz_finishes'], true);
        if (is_array($finishes)) {
            $clean = [];
            foreach ($finishes as $f) {
                $clean[] = [
                    'key'           => sanitize_key($f['key'] ?? ''),
                    'name'          => sanitize_text_field($f['name'] ?? ''),
                    'desc'          => sanitize_text_field($f['desc'] ?? ''),
                    'badge'         => sanitize_text_field($f['badge'] ?? ''),
                    'prompt_detail' => sanitize_textarea_field($f['prompt_detail'] ?? ''),
                ];
            }
            update_option('ace_viz_finishes', $clean);
        }
    }

    wp_send_json_success('Settings saved.');
}


// ============================================================
// ADMIN AJAX: Test API Connection
// ============================================================

add_action('wp_ajax_ace_test_api_connection', 'ace_handle_test_api_connection');

function ace_handle_test_api_connection() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    if (!check_ajax_referer('ace_blp_admin_nonce', '_nonce', false)) {
        wp_send_json_error('Invalid nonce');
    }

    $api_key = ace_get_api_key();
    if (!$api_key) {
        wp_send_json_error('No API key saved. Save settings first.');
    }

    $model = get_option('ace_visualizer_model', 'gemini-2.5-flash-image');

    // Text-only test (no image, minimal cost)
    $url  = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$api_key}";
    $body = [
        'contents' => [[
            'parts' => [['text' => 'Say hello in 3 words']]
        ]]
    ];

    $response = wp_remote_post($url, [
        'headers' => ['Content-Type' => 'application/json'],
        'body'    => json_encode($body),
        'timeout' => 15
    ]);

    if (is_wp_error($response)) {
        wp_send_json_error('Network error: ' . $response->get_error_message());
    }

    $status = wp_remote_retrieve_response_code($response);
    $data   = json_decode(wp_remote_retrieve_body($response), true);

    if ($status === 200) {
        $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? 'OK';
        wp_send_json_success('Connected! Model: ' . $model . '. Response: ' . substr($text, 0, 50));
    } else {
        $error = $data['error']['message'] ?? 'Unknown error (HTTP ' . $status . ')';
        wp_send_json_error('API error: ' . $error);
    }
}


// ============================================================
// SECURITY HEADERS (on Landing Page only)
// ============================================================

add_action('send_headers', 'ace_add_security_headers');

function ace_add_security_headers() {
    if (!function_exists('is_ace_backyard_lp') || !is_ace_backyard_lp()) return;

    header('X-Frame-Options: SAMEORIGIN');
    header('X-Content-Type-Options: nosniff');
    header('X-XSS-Protection: 1; mode=block');
    header('Referrer-Policy: strict-origin-when-cross-origin');
}


// ============================================================
// IMAGE CLEANUP CRON (7-day auto-delete)
// ============================================================

add_action('ace_cleanup_images_cron', 'ace_cleanup_old_images');

function ace_cleanup_old_images() {
    $viz_dir = wp_upload_dir()['basedir'] . '/ace-visualizer/';
    if (!is_dir($viz_dir)) return;

    $files     = glob($viz_dir . '*.{jpg,jpeg,png,webp}', GLOB_BRACE);
    $threshold = time() - (7 * DAY_IN_SECONDS);

    if (!is_array($files)) return;

    foreach ($files as $file) {
        if (filemtime($file) < $threshold) {
            @unlink($file);
        }
    }
}

function ace_schedule_cleanup_cron() {
    if (!wp_next_scheduled('ace_cleanup_images_cron')) {
        wp_schedule_event(time(), 'daily', 'ace_cleanup_images_cron');
    }
}
add_action('init', 'ace_schedule_cleanup_cron');


// ============================================================
// DATABASE MIGRATION (add visualizer columns)
// ============================================================

function ace_visualizer_db_upgrade() {
    if ((int) get_option('ace_visualizer_db_version', 0) >= 1) return;

    global $wpdb;
    $table   = $wpdb->prefix . 'ace_backyard_leads';
    $columns = $wpdb->get_col("SHOW COLUMNS FROM $table");

    if (!in_array('visualizer_data', $columns)) {
        $wpdb->query("ALTER TABLE $table ADD COLUMN visualizer_data LONGTEXT DEFAULT NULL");
    }
    if (!in_array('original_image_url', $columns)) {
        $wpdb->query("ALTER TABLE $table ADD COLUMN original_image_url VARCHAR(500) DEFAULT NULL");
    }
    if (!in_array('generated_image_url', $columns)) {
        $wpdb->query("ALTER TABLE $table ADD COLUMN generated_image_url VARCHAR(500) DEFAULT NULL");
    }

    update_option('ace_visualizer_db_version', 1);
}
add_action('admin_init', 'ace_visualizer_db_upgrade');


// ============================================================
// ASSET ENQUEUE (Frontend - LP only, with defer)
// ============================================================

add_action('wp_enqueue_scripts', 'ace_enqueue_visualizer_assets');

function ace_enqueue_visualizer_assets() {
    if (!function_exists('is_ace_backyard_lp') || !is_ace_backyard_lp()) return;
    if (get_option('ace_active_mode', 'calculator') !== 'visualizer') return;

    wp_enqueue_style(
        'ace-visualizer',
        plugin_dir_url(dirname(__FILE__)) . 'assets/css/visualizer.css',
        [],
        ACE_BLP_VERSION
    );

    wp_enqueue_script(
        'ace-visualizer',
        plugin_dir_url(dirname(__FILE__)) . 'assets/js/visualizer.js',
        [],
        ACE_BLP_VERSION,
        true
    );

    // Add defer attribute for non-blocking load
    add_filter('script_loader_tag', function($tag, $handle) {
        if ($handle === 'ace-visualizer') {
            return str_replace(' src', ' defer src', $tag);
        }
        return $tag;
    }, 10, 2);

    // PUBLIC data only. API key is NEVER here.
    // Loading messages from admin (one per line) with defaults
    $loading_msgs_raw = get_option('ace_viz_loading_messages', "Analyzing your backyard layout...\nMapping property boundaries...\nSelecting your design elements...\nApplying your style preferences...\nRendering hardscape and structures...\nAdding landscaping and greenery...\nFine-tuning lighting and shadows...\nAlmost ready...");
    $loading_tips_raw = get_option('ace_viz_loading_tips', "A swimming pool can add 5-8% to your home value\nProfessional landscaping returns 100-200% ROI at resale\nOutdoor kitchens are the #1 requested feature in LA\nA well-designed patio can add 300-500 sq ft of living space\nLED landscape lighting uses 80% less energy than traditional");
    $loading_msgs = array_values(array_filter(array_map('trim', explode("\n", $loading_msgs_raw))));
    $loading_tips = array_values(array_filter(array_map('trim', explode("\n", $loading_tips_raw))));

    wp_localize_script('ace-visualizer', 'aceViz', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('ace_visualizer_nonce'),
        'settings' => [
            'maxRetries'     => (int) get_option('ace_visualizer_max_retries', 1),
            'leadGate'       => get_option('ace_visualizer_lead_gate', 'blur'),
            'dailyLimit'     => (int) get_option('ace_visualizer_daily_limit', 5),
            'promoPopup'     => (int) get_option('ace_visualizer_promo_popup', 1),
            'promoScrollPct' => (int) get_option('ace_visualizer_promo_scroll_pct', 15),
        ],
        'styleImages' => [
            'modern'        => get_option('ace_style_modern_image', ''),
            'mediterranean' => get_option('ace_style_mediterranean_image', ''),
            'tropical'      => get_option('ace_style_tropical_image', ''),
            'classic'       => get_option('ace_style_classic_image', ''),
        ],
        'text' => [
            'loadingMessages' => $loading_msgs,
            'loadingTips'     => $loading_tips,
        ]
    ]);
}


// ============================================================
// DEACTIVATION CLEANUP (unschedule cron)
// ============================================================

function ace_visualizer_deactivate_cleanup() {
    $timestamp = wp_next_scheduled('ace_cleanup_images_cron');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'ace_cleanup_images_cron');
    }
}
