<?php
/**
 * ACE Backyard Landing Page - Main Template
 *
 * Renders independently from the WordPress theme.
 * Outputs its own DOCTYPE html and loads plugin settings.
 *
 * @package ACE_Backyard_LP
 */

if (!defined('ABSPATH')) exit;

// Load settings
$s = get_option('ace_backyard_lp_settings', array());
if (empty($s)) {
    if (function_exists('ace_blp_get_defaults')) {
        $s = ace_blp_get_defaults();
    }
}


// Helper function
function blp($s, $key, $default = '') {
    return isset($s[$key]) ? $s[$key] : $default;
}

$phone = blp($s, 'phone', '(310) 438-6866');
$phone_link = 'tel:' . preg_replace('/[^0-9+]/', '', $phone);
?>
<!DOCTYPE html>
<html <?php language_attributes(); ?>>
<head>
<meta charset="<?php bloginfo('charset'); ?>">
<meta name="viewport" content="width=device-width, initial-scale=1.0">

<?php wp_head(); ?>

<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
<link rel="stylesheet" href="<?php echo ACE_BLP_URL; ?>assets/css/landing-page.css?v=<?php echo ACE_BLP_VERSION; ?>">
</head>
<body <?php body_class('ace-backyard-lp'); ?>>
<?php wp_body_open(); ?>

<?php $nonce = wp_create_nonce('ace_blp_form_nonce'); ?>

<!-- ============================================
     Section 0: Unified Header Bar
     ============================================ -->
<?php
$logo_url    = !empty($s['logo']) ? $s['logo'] : '';
$logo_height = intval( blp($s, 'logo_height', 50) );
$logo_link   = blp($s, 'logo_link', '#hero');
$logo_scroll = ( strpos($logo_link, '#') === 0 );
$nav_labels  = (!empty($s['scroll_nav_enabled']) && !empty($s['scroll_nav_labels'])) ? $s['scroll_nav_labels'] : array();
?>
<header class="topbar" id="siteHeader">
    <div class="topbar-inner">
        <!-- Logo -->
        <div class="topbar-logo">
            <?php if (!empty($logo_url)) : ?>
                <a href="<?php echo esc_attr($logo_link); ?>"<?php if ($logo_scroll) echo ' data-scroll="true"'; ?>>
                    <img src="<?php echo esc_url($logo_url); ?>" alt="ACE Design &amp; Build" class="topbar-logo-img" style="height: <?php echo $logo_height; ?>px; width: auto;">
                </a>
            <?php else : ?>
                <a href="<?php echo esc_attr($logo_link); ?>" class="topbar-logo-text"<?php if ($logo_scroll) echo ' data-scroll="true"'; ?>>ACE Design &amp; Build</a>
            <?php endif; ?>
        </div>

        <!-- Nav links (desktop) -->
        <?php if (!empty($nav_labels)): ?>
        <nav class="topbar-nav" id="scrollNav">
            <?php foreach ($nav_labels as $i => $item): ?>
                <?php
                $target = esc_attr($item['target']);
                $label = esc_html($item['label']);
                $is_cta = ($target === '#final');
                ?>
                <a href="<?php echo $target; ?>" class="topbar-nav-link<?php echo $is_cta ? ' nav-cta' : ''; ?>"><?php echo $label; ?></a>
            <?php endforeach; ?>
        </nav>
        <?php endif; ?>

        <!-- Right side: phone + CTA -->
        <div class="topbar-right">
            <a href="<?php echo esc_attr($phone_link); ?>" class="topbar-phone">
                <i class="fas fa-phone"></i> <?php echo esc_html($phone); ?>
            </a>
            <a href="#final" class="btn-gold btn-sm topbar-cta">Free Quote</a>
        </div>

        <!-- Hamburger (mobile/tablet) -->
        <button class="topbar-hamburger" id="hamburgerBtn" type="button" aria-label="Open menu" aria-expanded="false">
            <span></span><span></span><span></span>
        </button>
    </div>
</header>

<!-- Mobile slide-out menu -->
<div class="mobile-menu-overlay" id="mobileMenuOverlay">
    <div class="mobile-menu" id="mobileMenu">
        <div class="mobile-menu-header">
            <span class="mobile-menu-title">Menu</span>
            <button class="mobile-menu-close" id="mobileMenuClose" type="button" aria-label="Close menu">&times;</button>
        </div>
        <nav class="mobile-menu-nav">
            <?php foreach ($nav_labels as $item): ?>
                <?php
                $target = esc_attr($item['target']);
                $label = esc_html($item['label']);
                ?>
                <a href="<?php echo $target; ?>" class="mobile-menu-link"><?php echo $label; ?></a>
            <?php endforeach; ?>
        </nav>
        <div class="mobile-menu-footer">
            <a href="<?php echo esc_attr($phone_link); ?>" class="mobile-menu-phone">
                <i class="fas fa-phone"></i> <?php echo esc_html($phone); ?>
            </a>
            <a href="#final" class="btn-gold btn-full mobile-menu-cta">Free Quote</a>
        </div>
    </div>
</div>

<!-- ============================================
     Section 1: Hero
     ============================================ -->
<?php if (!empty($s['hero_enabled'])): ?>
<?php
$hero_bg = blp($s, 'hero_bg_image', 'https://images.unsplash.com/photo-1600596542815-ffad4c1539a8?w=1920');
$hero_headline = esc_html(blp($s, 'hero_headline', "Your Backyard's Best Days Are Ahead"));
$hero_gold_word = esc_html(blp($s, 'hero_gold_word', 'Ahead'));
if ($hero_gold_word) {
    $hero_headline = str_replace($hero_gold_word, '<span class="gold">' . $hero_gold_word . '</span>', $hero_headline);
}
?>
<section class="section hero" id="hero">
    <div class="hero-bg">
        <img src="<?php echo esc_url($hero_bg); ?>" alt="Luxury Backyard" loading="eager">
    </div>
    <div class="hero-inner">
        <div class="hero-content">
            <span class="hero-badge"><?php echo esc_html(blp($s, 'hero_badge', 'Serving Los Angeles & San Fernando Valley')); ?></span>
            <h1 class="hero-title"><?php echo $hero_headline; ?></h1>
            <p class="hero-sub"><?php echo esc_html(blp($s, 'hero_subheadline', '')); ?></p>

            <div class="trust-row">
                <div class="trust-badge" style="animation-delay: 0ms">
                    <i class="fas fa-star trust-badge-icon"></i>
                    <div class="trust-badge-text">
                        <span class="trust-badge-value">4.9</span>
                        <span class="trust-badge-label">Google (120+ Reviews)</span>
                    </div>
                </div>
                <div class="trust-badge" style="animation-delay: 200ms">
                    <i class="fas fa-trophy trust-badge-icon"></i>
                    <div class="trust-badge-text">
                        <span class="trust-badge-value">15+</span>
                        <span class="trust-badge-label">Years Experience</span>
                    </div>
                </div>
                <div class="trust-badge" style="animation-delay: 400ms">
                    <i class="fas fa-shield-halved trust-badge-icon"></i>
                    <div class="trust-badge-text">
                        <span class="trust-badge-value">Licensed</span>
                        <span class="trust-badge-label">Bonded &amp; Insured</span>
                    </div>
                </div>
                <div class="trust-badge" style="animation-delay: 600ms">
                    <i class="fas fa-wand-magic-sparkles trust-badge-icon"></i>
                    <div class="trust-badge-text">
                        <span class="trust-badge-value">Free</span>
                        <span class="trust-badge-label">3D Design</span>
                    </div>
                </div>
            </div>

            <?php if (get_option('ace_active_mode', 'calculator') === 'visualizer') : ?>
            <?php
            $viz_hero_btn = get_option('ace_viz_hero_btn', 'See Your Backyard Transformed');
            $viz_hero_sub = get_option('ace_viz_hero_sub', 'Upload a photo. Get an AI design in 60 seconds — free.');
            ?>
            <div class="hero-viz-cta">
                <a href="#" class="hero-viz-btn" id="heroVizBtn">
                    <i class="fas fa-wand-magic-sparkles"></i> <?php echo esc_html($viz_hero_btn); ?>
                </a>
                <p class="hero-viz-sub"><?php echo esc_html($viz_hero_sub); ?></p>
            </div>
            <?php endif; ?>
        </div>

        <div class="hero-form-wrap">
            <form class="lp-form hero-form" data-source="hero">
                <input type="hidden" name="_nonce" value="<?php echo $nonce; ?>">
                <input type="hidden" name="source" value="hero">
                <h3 class="form-title"><?php echo esc_html(blp($s, 'hero_form_title', 'Get Your Free Backyard Plan')); ?></h3>
                <p class="form-subtitle"><?php echo esc_html(blp($s, 'hero_form_subtitle', 'Tell us a little about your project. We respond within 2 hours.')); ?></p>
                <input type="text" name="name" placeholder="Your Name" required>
                <input type="tel" name="phone" placeholder="Phone Number" required>
                <input type="email" name="email" placeholder="Email Address">
                <input type="text" name="zip_code" placeholder="Project Zip Code">
                <textarea name="message" placeholder="Tell Us About Your Project" rows="3"></textarea>
                <button type="submit" class="btn-gold btn-full"><?php echo esc_html(blp($s, 'hero_form_button', 'Get My Free Plan')); ?></button>
                <div class="form-success" style="display:none;">
                    <i class="fas fa-check-circle"></i>
                    <p>Thank you! We will be in touch within 2 hours.</p>
                </div>
            </form>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 2: Visual Proof Strip
     ============================================ -->
<?php if (!empty($s['proof_enabled'])): ?>
<section class="proof-strip">
    <div class="proof-track" style="animation-duration: <?php echo intval(blp($s, 'proof_speed', 30)); ?>s;">
        <?php
        $images = !empty($s['proof_images']) ? array_filter($s['proof_images']) : array();
        // Output images twice for seamless loop
        for ($loop = 0; $loop < 2; $loop++):
            foreach ($images as $img): ?>
            <img src="<?php echo esc_url($img); ?>" alt="ACE Project" loading="lazy">
            <?php endforeach;
        endfor;
        // If no images, show placeholder divs
        if (empty($images)):
            for ($i = 0; $i < 16; $i++): ?>
            <div class="proof-placeholder"></div>
            <?php endfor;
        endif; ?>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 3: Transformation (Before/After)
     ============================================ -->
<?php if (!empty($s['transform_enabled'])): ?>
<?php
$transform_headline = esc_html(blp($s, 'transform_headline', 'Stop Dreaming. Start Living.'));
$transform_gold = esc_html(blp($s, 'transform_gold_word', 'Living.'));
if ($transform_gold) {
    $transform_headline = str_replace($transform_gold, '<span class="gold">' . $transform_gold . '</span>', $transform_headline);
}
?>
<section class="section transform-section" id="transform">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'transform_label', 'The Transformation')); ?></span>
        <h2 class="section-title"><?php echo $transform_headline; ?></h2>
        <p class="section-sub"><?php echo esc_html(blp($s, 'transform_subheadline', '')); ?></p>

        <div class="transform-grid">
            <div class="transform-slider">
                <div class="ba-container" id="baSlider">
                    <img src="<?php echo esc_url(blp($s, 'transform_before_image', '')); ?>" alt="Before" class="ba-before">
                    <img src="<?php echo esc_url(blp($s, 'transform_after_image', '')); ?>" alt="After" class="ba-after" id="baAfter">
                    <div class="ba-handle" id="baHandle">
                        <div class="ba-handle-circle"><i class="fas fa-arrows-left-right"></i></div>
                    </div>
                    <span class="ba-label ba-label-before">BEFORE</span>
                    <span class="ba-label ba-label-after">AFTER</span>
                </div>
            </div>

        </div>

        <div class="problems-block">
            <h3 class="problems-title">Sound Familiar?</h3>
            <div class="problems-grid">
                <div class="problem-card">
                    <div class="problem-icon-wrap">
                        <i class="fas fa-eye-slash"></i>
                    </div>
                    <h4 class="problem-headline">Wasted Space</h4>
                    <p class="problem-desc">Your backyard sits empty while you're paying the mortgage on every square foot.</p>
                </div>
                <div class="problem-card">
                    <div class="problem-icon-wrap">
                        <i class="fas fa-arrow-trend-down"></i>
                    </div>
                    <h4 class="problem-headline">Falling Behind</h4>
                    <p class="problem-desc">Your neighbors are upgrading. Your home value isn't keeping up.</p>
                </div>
                <div class="problem-card">
                    <div class="problem-icon-wrap">
                        <i class="fas fa-money-bill-wave"></i>
                    </div>
                    <h4 class="problem-headline">DIY Money Pit</h4>
                    <p class="problem-desc">Weekend projects that drag on for months and end up costing more than hiring a pro.</p>
                </div>
            </div>
            <p class="problems-cta">There's a better way.</p>
            <div class="problems-divider"></div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 4: AI Visualizer / Calculator
     ============================================ -->
<?php if (get_option('ace_active_mode', 'calculator') === 'visualizer') : ?>
<?php
// Visualizer phone with fallback
$viz_phone = get_option('ace_visualizer_phone', '');
$viz_phone = !empty($viz_phone) ? $viz_phone : $phone;
$viz_phone_link = 'tel:' . preg_replace('/[^0-9+]/', '', $viz_phone);

// Visualizer text options (admin-configurable)
$vt = [
    'step1_title'      => get_option('ace_viz_step1_title', 'See Your Backyard <em>Transformed</em>'),
    'step1_subtitle'   => get_option('ace_viz_step1_subtitle', 'Upload a photo. Pick your dream features. Our AI does the rest.'),
    'step2_title'      => get_option('ace_viz_step2_title', 'How Big Is Your Backyard?'),
    'step2_subtitle'   => get_option('ace_viz_step2_subtitle', 'This helps us scale the design to your space.'),
    'step3_title'      => get_option('ace_viz_step3_title', 'What Do You Want In Your Backyard?'),
    'step3_subtitle'   => get_option('ace_viz_step3_subtitle', 'Select all that interest you. You can always adjust later.'),
    'step4_title'      => get_option('ace_viz_step4_title', 'Select Your Finish Level'),
    'step4_subtitle'   => get_option('ace_viz_step4_subtitle', 'This affects materials, details, and your estimate.'),
    'step5_title'      => get_option('ace_viz_step5_title', 'Pick Your Style'),
    'step5_subtitle'   => get_option('ace_viz_step5_subtitle', 'Choose what speaks to you. This is the final step before the magic happens.'),
    'upload_text'      => get_option('ace_viz_upload_text', 'Drag & drop your backyard photo here'),
    'upload_btn'       => get_option('ace_viz_upload_btn', 'Choose Photo'),
    'upload_tip'       => get_option('ace_viz_upload_tip', 'For best results, use a landscape photo taken from your back door'),
    'success_msg'      => get_option('ace_viz_success_msg', 'Beautiful! Now let\'s design your dream backyard.'),
    'start_btn'        => get_option('ace_viz_start_btn', 'Start Designing'),
    'gate_title'       => get_option('ace_viz_gate_title', 'Your Design Is Ready!'),
    'gate_subtitle'    => get_option('ace_viz_gate_subtitle', 'Enter your details to unlock the full rendering'),
    'gate_btn'         => get_option('ace_viz_gate_btn', 'Unlock My Design'),
    'result_title'     => get_option('ace_viz_result_title', 'Here\'s Your New Backyard'),
    'price_label'      => get_option('ace_viz_price_label', 'YOUR ESTIMATED INVESTMENT'),
    'price_disclaimer' => get_option('ace_viz_price_disclaimer', 'Estimate based on typical projects. Final pricing determined during your free in-home consultation.'),
    'cta_btn'          => get_option('ace_viz_cta_btn', 'Book Your Free In-Home Consultation'),
    'retry_text'       => get_option('ace_viz_retry_text', 'Not quite right? Try different options'),
    'label_before'     => get_option('ace_viz_label_before', 'YOUR BACKYARD'),
    'label_after'      => get_option('ace_viz_label_after', 'ACE DESIGN'),
];
?>
<section id="calculator" class="viz-section">
    <div id="vizSlot"></div>
    <div class="viz-container">

        <!-- Progress Bar -->
        <div class="viz-progress" id="vizProgress">
            <div class="viz-progress-bar" id="vizProgressBar"></div>
            <div class="viz-progress-steps">
                <span class="viz-progress-step active" data-step="1">Upload</span>
                <span class="viz-progress-step" data-step="2">Size</span>
                <span class="viz-progress-step" data-step="3">Features</span>
                <span class="viz-progress-step" data-step="4">Finish</span>
                <span class="viz-progress-step" data-step="5">Style</span>
            </div>
        </div>

        <!-- STEP 1: Upload -->
        <div class="viz-step active" id="vizStep1" data-step="1">
            <div class="viz-step-header">
                <h2 class="viz-title"><?php echo wp_kses($vt['step1_title'], ['em' => []]); ?></h2>
                <p class="viz-subtitle"><?php echo esc_html($vt['step1_subtitle']); ?></p>
            </div>

            <div class="viz-upload-zone" id="vizDropZone">
                <div class="viz-upload-content" id="vizUploadContent">
                    <div class="viz-upload-icon"><i class="fas fa-cloud-arrow-up"></i></div>
                    <p class="viz-upload-text"><?php echo esc_html($vt['upload_text']); ?></p>
                    <p class="viz-upload-or">or</p>
                    <label class="viz-upload-btn">
                        <span><?php echo esc_html($vt['upload_btn']); ?></span>
                        <input type="file" id="vizFileInput" accept="image/jpeg,image/png,image/webp" hidden>
                    </label>
                    <p class="viz-upload-tip">
                        <i class="fas fa-lightbulb"></i>
                        <?php echo esc_html($vt['upload_tip']); ?>
                    </p>
                </div>

                <div class="viz-upload-preview" id="vizPreview" style="display:none;">
                    <img id="vizPreviewImg" src="" alt="Your backyard">
                    <button type="button" class="viz-change-photo" id="vizChangePhoto">
                        <i class="fas fa-camera"></i> Change Photo
                    </button>
                </div>
            </div>

            <div class="viz-upload-success" id="vizUploadSuccess" style="display:none;">
                <p class="viz-success-text">
                    <i class="fas fa-check-circle"></i> <?php echo esc_html($vt['success_msg']); ?>
                </p>
                <button type="button" class="viz-btn-primary" id="vizStartDesign">
                    <?php echo esc_html($vt['start_btn']); ?> <i class="fas fa-arrow-right"></i>
                </button>
            </div>

            <div class="viz-upload-error" id="vizUploadError" style="display:none;">
                <p id="vizErrorText"></p>
            </div>
        </div>

        <!-- STEP 2: Yard Size -->
        <div class="viz-step" id="vizStep2" data-step="2">
            <div class="viz-step-header">
                <button type="button" class="viz-back-btn" data-back="1"><i class="fas fa-arrow-left"></i></button>
                <h2 class="viz-title"><?php echo wp_kses($vt['step2_title'], ['em' => []]); ?></h2>
                <p class="viz-subtitle"><?php echo esc_html($vt['step2_subtitle']); ?></p>
            </div>
            <div class="viz-options viz-options-size">
                <?php
                $viz_db_sizes = get_option('ace_viz_sizes', []);
                if (empty($viz_db_sizes) && function_exists('ace_blp_get_viz_size_defaults')) {
                    $viz_db_sizes = ace_blp_get_viz_size_defaults();
                }
                foreach ($viz_db_sizes as $vsize): ?>
                <button type="button" class="viz-option" data-value="<?php echo esc_attr($vsize['key']); ?>">
                    <span class="viz-option-icon"><i class="<?php echo esc_attr($vsize['icon'] ?? 'fas fa-tree'); ?>"></i></span>
                    <span class="viz-option-label"><?php echo esc_html($vsize['name']); ?></span>
                    <span class="viz-option-detail"><?php echo esc_html($vsize['desc']); ?></span>
                </button>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- STEP 3: Features (multi-select) -->
        <div class="viz-step" id="vizStep3" data-step="3">
            <div class="viz-step-header">
                <button type="button" class="viz-back-btn" data-back="2"><i class="fas fa-arrow-left"></i></button>
                <h2 class="viz-title"><?php echo wp_kses($vt['step3_title'], ['em' => []]); ?></h2>
                <p class="viz-subtitle"><?php echo esc_html($vt['step3_subtitle']); ?></p>
            </div>
            <div class="viz-options viz-options-features viz-multi">
                <?php
                $viz_db_features = get_option('ace_viz_features', []);
                if (empty($viz_db_features) && function_exists('ace_blp_get_viz_feature_defaults')) {
                    $viz_db_features = ace_blp_get_viz_feature_defaults();
                }
                foreach ($viz_db_features as $vfeat): ?>
                <button type="button" class="viz-option" data-value="<?php echo esc_attr($vfeat['key']); ?>">
                    <span class="viz-option-icon"><i class="<?php echo esc_attr($vfeat['icon'] ?? 'fas fa-star'); ?>"></i></span>
                    <span class="viz-option-label"><?php echo esc_html($vfeat['name']); ?></span>
                </button>
                <?php endforeach; ?>
            </div>
            <button type="button" class="viz-btn-primary viz-continue-btn" id="vizFeatContinue" disabled>
                Continue <i class="fas fa-arrow-right"></i>
            </button>
        </div>

        <!-- STEP 4: Finish Level -->
        <div class="viz-step" id="vizStep4" data-step="4">
            <div class="viz-step-header">
                <button type="button" class="viz-back-btn" data-back="3"><i class="fas fa-arrow-left"></i></button>
                <h2 class="viz-title"><?php echo wp_kses($vt['step4_title'], ['em' => []]); ?></h2>
                <p class="viz-subtitle"><?php echo esc_html($vt['step4_subtitle']); ?></p>
            </div>
            <div class="viz-options viz-options-finish">
                <?php
                $viz_db_finishes = get_option('ace_viz_finishes', []);
                if (empty($viz_db_finishes) && function_exists('ace_blp_get_viz_finish_defaults')) {
                    $viz_db_finishes = ace_blp_get_viz_finish_defaults();
                }
                foreach ($viz_db_finishes as $vfin): ?>
                <button type="button" class="viz-option viz-option-wide" data-value="<?php echo esc_attr($vfin['key']); ?>">
                    <?php if (!empty($vfin['badge'])): ?>
                    <span class="viz-option-badge"><?php echo esc_html($vfin['badge']); ?></span>
                    <?php endif; ?>
                    <span class="viz-option-label"><?php echo esc_html($vfin['name']); ?></span>
                    <span class="viz-option-detail"><?php echo esc_html($vfin['desc']); ?></span>
                </button>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- STEP 5: Style -->
        <div class="viz-step" id="vizStep5" data-step="5">
            <div class="viz-step-header">
                <button type="button" class="viz-back-btn" data-back="4"><i class="fas fa-arrow-left"></i></button>
                <h2 class="viz-title"><?php echo wp_kses($vt['step5_title'], ['em' => []]); ?></h2>
                <p class="viz-subtitle"><?php echo esc_html($vt['step5_subtitle']); ?></p>
            </div>
            <div class="viz-options viz-options-style">
                <?php
                $viz_styles = [
                    'modern'        => ['name' => 'Modern Contemporary',  'option' => 'ace_style_modern_image'],
                    'mediterranean' => ['name' => 'Mediterranean',        'option' => 'ace_style_mediterranean_image'],
                    'tropical'      => ['name' => 'Tropical Resort',      'option' => 'ace_style_tropical_image'],
                    'classic'       => ['name' => 'Classic Traditional',   'option' => 'ace_style_classic_image'],
                ];
                foreach ($viz_styles as $viz_key => $viz_style) :
                    $viz_img = get_option($viz_style['option'], '');
                ?>
                <button type="button" class="viz-option viz-option-style" data-value="<?php echo esc_attr($viz_key); ?>">
                    <?php if ($viz_img) : ?>
                        <img src="<?php echo esc_url($viz_img); ?>"
                             alt="<?php echo esc_attr($viz_style['name']); ?>"
                             class="viz-style-img"
                             loading="lazy"
                             decoding="async"
                             width="400" height="300">
                    <?php else : ?>
                        <div class="viz-style-placeholder"></div>
                    <?php endif; ?>
                    <span class="viz-option-label"><?php echo esc_html($viz_style['name']); ?></span>
                </button>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- STEP 6: Loading -->
        <div class="viz-step" id="vizStep6" data-step="6">
            <div class="viz-loading">
                <div class="viz-loading-spinner"></div>
                <p class="viz-loading-message" id="vizLoadMsg">Analyzing your backyard layout...</p>
                <div class="viz-loading-bar">
                    <div class="viz-loading-bar-fill" id="vizLoadBar"></div>
                </div>
                <p class="viz-loading-percent" id="vizLoadPercent">0%</p>
                <p class="viz-loading-tip" id="vizLoadTip">A swimming pool can add 5-8% to your home value</p>
            </div>
        </div>

    </div>

    <!-- Result Popup Modal (covers lead gate + result) -->
    <div class="viz-popup" id="vizPopup">
        <div class="viz-popup-backdrop"></div>
        <div class="viz-popup-scroll">
            <div class="viz-popup-content">
                <button type="button" class="viz-popup-close" id="vizPopupClose" aria-label="Close">&times;</button>

                <!-- Gate view: blurred image + form overlay (shown when lead gate is active) -->
                <div class="viz-popup-gate" id="vizPopupGate">
                    <div class="viz-popup-gate-img" id="vizGateImgWrap">
                        <img id="vizGateImg" src="" alt="Your design preview">
                        <div class="viz-popup-gate-darken"></div>
                        <div class="viz-popup-gate-form" id="vizGateFormWrap">
                            <form id="vizGateForm">
                                <!-- HONEYPOT: hidden from humans, bots auto-fill it, server rejects if filled -->
                                <div style="position:absolute;left:-9999px;top:-9999px;" aria-hidden="true">
                                    <input type="text" name="website_url" id="vizHoneypot" tabindex="-1" autocomplete="off">
                                </div>
                                <h3 class="viz-popup-gate-title"><?php echo esc_html($vt['gate_title']); ?></h3>
                                <p class="viz-popup-gate-sub"><?php echo esc_html($vt['gate_subtitle']); ?></p>
                                <div class="viz-form-group">
                                    <input type="text" id="vizLeadName" placeholder="Your Name *" required autocomplete="name">
                                </div>
                                <div class="viz-form-group">
                                    <input type="tel" id="vizLeadPhone" placeholder="Phone Number *" required inputmode="tel" autocomplete="tel">
                                </div>
                                <div class="viz-form-group">
                                    <input type="email" id="vizLeadEmail" placeholder="Email (optional)" autocomplete="email">
                                </div>
                                <button type="submit" class="viz-btn-primary viz-btn-unlock">
                                    <i class="fas fa-lock-open"></i> <?php echo esc_html($vt['gate_btn']); ?>
                                </button>
                            </form>
                            <div class="viz-gate-email-alt" id="vizEmailAlt" style="display:none;">
                                <p>Prefer email? We'll send the full design to your inbox.</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Result view: before/after slider + price + CTA (shown after gate or directly) -->
                <div class="viz-popup-result" id="vizPopupResult" style="display:none;">
                    <h2 class="viz-title"><?php echo esc_html($vt['result_title']); ?></h2>

                    <div class="viz-ba-container" id="vizBA">
                        <div class="viz-ba-wrapper">
                            <img class="viz-ba-after" id="vizAfter" src="" alt="Your AI-designed backyard">
                            <img class="viz-ba-before" id="vizBefore" src="" alt="Your current backyard">
                            <div class="viz-ba-divider" id="vizDivider">
                                <div class="viz-ba-handle"><i class="fas fa-arrows-left-right"></i></div>
                            </div>
                        </div>
                        <span class="viz-ba-label viz-ba-label-before"><?php echo esc_html($vt['label_before']); ?></span>
                        <span class="viz-ba-label viz-ba-label-after"><?php echo esc_html($vt['label_after']); ?></span>
                    </div>

                    <div class="viz-price-result">
                        <p class="viz-price-label"><?php echo esc_html($vt['price_label']); ?></p>
                        <p class="viz-price-value" id="vizPrice"></p>
                        <p class="viz-price-disclaimer"><?php echo esc_html($vt['price_disclaimer']); ?></p>
                    </div>

                    <div class="viz-cta">
                        <a href="<?php echo esc_attr($viz_phone_link); ?>" class="viz-btn-primary viz-cta-main" id="vizCtaConsult">
                            <i class="fas fa-phone"></i> Call for Your Free In-Home Quote
                        </a>
                        <span class="viz-cta-phone-number"><?php echo esc_html($viz_phone); ?></span>
                    </div>

                    <div class="viz-retry">
                        <button type="button" class="viz-retry-btn" id="vizRetry">
                            <i class="fas fa-rotate"></i> <?php echo esc_html($vt['retry_text']); ?>
                        </button>
                    </div>
                </div>

            </div>
        </div>
    </div>
</section>

<!-- Visualizer Flow Popup (full wizard in modal - opened by hero CTA / promo popup) -->
<div class="viz-flow-popup" id="vizFlowPopup">
    <div class="viz-flow-popup-backdrop"></div>
    <div class="viz-flow-popup-scroll" id="vizFlowScroll">
        <div class="viz-flow-popup-inner" id="vizFlowInner">
            <button type="button" class="viz-popup-close viz-flow-close" id="vizFlowClose" aria-label="Close">&times;</button>
            <div class="viz-flow-body" id="vizFlowBody"></div>
        </div>
    </div>
</div>

<?php elseif (!empty($s['calc_enabled'])) : ?>
<!-- Original Calculator (fallback when visualizer is OFF) -->
<section class="section calc-section" id="calculator">
    <div class="container">
        <span class="section-label">Calculator</span>
        <h2 class="section-title"><?php echo esc_html(blp($s, 'calc_headline', 'What Does a Backyard Remodel Cost?')); ?></h2>
        <p class="section-sub"><?php echo esc_html(blp($s, 'calc_subheadline', '')); ?></p>

        <div class="calc-card">
            <!-- Step 1: Size -->
            <div class="calc-step">
                <h4 class="calc-step-title"><span class="step-num">1</span> Select Your Yard Size</h4>
                <div class="size-options">
                    <?php
                    $sizes = !empty($s['calc_sizes']) ? $s['calc_sizes'] : array();
                    $default_size = intval(blp($s, 'calc_default_size', 1));
                    foreach ($sizes as $i => $size): ?>
                    <div class="size-opt <?php echo ($i === $default_size) ? 'active' : ''; ?>" data-multiplier="<?php echo esc_attr($size['multiplier']); ?>" onclick="selectSize(this)">
                        <span class="size-name"><?php echo esc_html($size['name']); ?></span>
                        <span class="size-desc"><?php echo esc_html($size['desc']); ?></span>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Step 2: Features -->
            <div class="calc-step">
                <h4 class="calc-step-title"><span class="step-num">2</span> Choose Your Features</h4>
                <div class="feature-grid">
                    <?php
                    $features = !empty($s['calc_features']) ? $s['calc_features'] : array();
                    $default_features = !empty($s['calc_default_features']) ? $s['calc_default_features'] : array(0, 4);
                    foreach ($features as $i => $feat):
                        $is_active = in_array($i, $default_features);
                    ?>
                    <div class="calc-feature <?php echo $is_active ? 'active' : ''; ?>"
                         data-std="<?php echo esc_attr($feat['std']); ?>"
                         data-prem="<?php echo esc_attr($feat['prem']); ?>"
                         data-lux="<?php echo esc_attr($feat['lux']); ?>"
                         onclick="toggleFeature(this)">
                        <div class="calc-feature-icon">
                            <i class="<?php echo esc_attr($feat['icon']); ?>"></i>
                        </div>
                        <span class="calc-feature-name"><?php echo esc_html($feat['name']); ?></span>
                        <div class="calc-feature-check">
                            <i class="fas fa-check"></i>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Step 3: Finish Level -->
            <div class="calc-step">
                <h4 class="calc-step-title"><span class="step-num">3</span> Select Finish Level</h4>
                <div class="finish-options">
                    <?php
                    $finishes = !empty($s['calc_finishes']) ? $s['calc_finishes'] : array();
                    $default_finish = intval(blp($s, 'calc_default_finish', 1));
                    foreach ($finishes as $i => $fin): ?>
                    <div class="finish-opt <?php echo ($i === $default_finish) ? 'active' : ''; ?>" data-key="<?php echo esc_attr($fin['key']); ?>" onclick="selectFinish(this)">
                        <span class="finish-name"><?php echo esc_html($fin['name']); ?></span>
                        <span class="finish-desc"><?php echo esc_html($fin['desc']); ?></span>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Result -->
            <div class="calc-result">
                <p class="calc-result-label">Your Estimated Investment</p>
                <div class="calc-result-price" id="calcPrice">$25,000 - $34,000</div>
                <p class="calc-disclaimer">Estimate based on typical projects. Final pricing determined during your free in-home consultation.</p>
                <p class="calc-result-note"><?php echo esc_html(blp($s, 'calc_result_note', '')); ?></p>

                <form class="lp-form calc-form" data-source="calculator">
                    <input type="hidden" name="_nonce" value="<?php echo $nonce; ?>">
                    <input type="hidden" name="source" value="calculator">
                    <input type="hidden" name="calculator_data" id="calcData" value="">
                    <div class="calc-result-form">
                        <input type="text" name="name" placeholder="Your Name" required>
                        <input type="tel" name="phone" placeholder="Phone Number" required>
                        <input type="email" name="email" placeholder="Email Address">
                        <input type="text" name="zip_code" placeholder="Project Zip Code">
                        <textarea name="message" placeholder="Tell Us About Your Project" class="calc-form-full"></textarea>
                    </div>
                    <button type="submit" class="btn-gold btn-full"><?php echo esc_html(blp($s, 'calc_form_button', 'Get My Exact Quote')); ?></button>
                    <div class="form-success" style="display:none;">
                        <i class="fas fa-check-circle"></i>
                        <p>Thank you! We will be in touch within 2 hours.</p>
                    </div>
                </form>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 5: What We Build (Services Grid)
     ============================================ -->
<?php if (!empty($s['services_enabled'])): ?>
<?php
$services_headline = esc_html(blp($s, 'services_headline', 'Everything Your Backyard Needs'));
$services_gold = esc_html(blp($s, 'services_gold_word', 'Needs'));
if ($services_gold) {
    $services_headline = str_replace($services_gold, '<span class="gold">' . $services_gold . '</span>', $services_headline);
}
?>
<section class="section services-section" id="services">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'services_label', 'What We Build')); ?></span>
        <h2 class="section-title"><?php echo $services_headline; ?></h2>

        <div class="services-grid">
            <?php
            $services = !empty($s['services_items']) ? $s['services_items'] : array();
            foreach ($services as $svc): ?>
            <div class="service-card">
                <div class="service-image">
                    <?php if (!empty($svc['image'])): ?>
                    <img src="<?php echo esc_url($svc['image']); ?>" alt="<?php echo esc_attr($svc['title']); ?>" loading="lazy">
                    <?php else: ?>
                    <div class="service-placeholder"><i class="<?php echo esc_attr($svc['icon']); ?>"></i></div>
                    <?php endif; ?>
                </div>
                <div class="service-body">
                    <div class="service-icon"><i class="<?php echo esc_attr($svc['icon']); ?>"></i></div>
                    <h4 class="service-title"><?php echo esc_html($svc['title']); ?></h4>
                    <p class="service-desc"><?php echo esc_html($svc['desc']); ?></p>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 6: Featured Projects
     ============================================ -->
<?php if (!empty($s['projects_enabled'])): ?>
<?php
$projects_headline = esc_html(blp($s, 'projects_headline', 'Real Backyards. Real Families. Real Results.'));
$projects_gold = esc_html(blp($s, 'projects_gold_word', 'Results.'));
if ($projects_gold) {
    $projects_headline = str_replace($projects_gold, '<span class="gold">' . $projects_gold . '</span>', $projects_headline);
}
?>
<section class="section projects-section" id="projects">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'projects_label', 'Featured Projects')); ?></span>
        <h2 class="section-title"><?php echo $projects_headline; ?></h2>
        <?php $projects_sub = blp($s, 'projects_subheadline', ''); ?>
        <?php if ($projects_sub): ?>
        <p class="section-sub"><?php echo esc_html($projects_sub); ?></p>
        <?php endif; ?>
    </div>

    <?php $projects = !empty($s['projects_items']) ? $s['projects_items'] : array(); ?>
    <div class="carousel-wrap" id="projectsCarousel" data-autoplay="4000" data-desktop="3" data-tablet="2" data-mobile="1">
        <button class="carousel-arrow carousel-prev" type="button" aria-label="Previous"><i class="fas fa-chevron-left"></i></button>
        <div class="carousel-viewport">
            <div class="carousel-track">
                <?php foreach ($projects as $proj): ?>
                <div class="carousel-slide">
                    <div class="project-card">
                        <div class="project-image">
                            <?php if (!empty($proj['image'])): ?>
                            <img src="<?php echo esc_url($proj['image']); ?>" alt="<?php echo esc_attr($proj['name']); ?>" loading="lazy">
                            <?php else: ?>
                            <div class="project-placeholder"><i class="fas fa-image"></i></div>
                            <?php endif; ?>
                        </div>
                        <div class="project-body">
                            <h4 class="project-name"><?php echo esc_html($proj['name']); ?></h4>
                            <p class="project-location"><i class="fas fa-map-marker-alt"></i> <?php echo esc_html($proj['location']); ?></p>
                            <p class="project-scope"><?php echo esc_html($proj['scope']); ?></p>
                            <div class="project-meta">
                                <div class="meta-item">
                                    <span class="meta-label">Timeline</span>
                                    <span class="meta-value"><?php echo esc_html($proj['timeline']); ?></span>
                                </div>
                                <div class="meta-item">
                                    <span class="meta-label">Area</span>
                                    <span class="meta-value"><?php echo esc_html($proj['area']); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <button class="carousel-arrow carousel-next" type="button" aria-label="Next"><i class="fas fa-chevron-right"></i></button>
        <div class="carousel-dots"></div>
    </div>
    <div class="projects-cta" style="text-align:center; margin-top:2rem;">
        <button class="btn-gold" data-popup="quote"><?php echo esc_html(blp($s, 'projects_cta_text', 'Your Backyard Could Be Next')); ?></button>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 7: CTA Bar
     ============================================ -->
<?php if (!empty($s['cta_enabled'])): ?>
<?php
$cta_text = esc_html(blp($s, 'cta_text', 'Ready to Love Your Backyard?'));
$cta_gold_text = esc_html(blp($s, 'cta_gold_text', 'It Starts with a Free Plan.'));
?>
<section class="section cta-bar">
    <div class="container cta-bar-inner">
        <p class="cta-bar-text"><?php echo $cta_text; ?> <span class="gold"><?php echo $cta_gold_text; ?></span></p>
        <button class="btn-gold" data-popup="quote"><?php echo esc_html(blp($s, 'cta_button', 'Get Started')); ?></button>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 8: Social Proof / Reviews
     ============================================ -->
<?php if (!empty($s['reviews_enabled'])): ?>
<?php
$reviews_headline = esc_html(blp($s, 'reviews_headline', "Don't Take Our Word For It"));
$reviews_gold = esc_html(blp($s, 'reviews_gold_word', ''));
if ($reviews_gold) {
    $reviews_headline = str_replace($reviews_gold, '<span class="gold">' . $reviews_gold . '</span>', $reviews_headline);
}
?>
<section class="section reviews-section" id="reviews">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'reviews_label', 'Reviews')); ?></span>
        <h2 class="section-title"><?php echo $reviews_headline; ?></h2>

        <div class="carousel-wrap" id="reviewsCarousel" data-autoplay="6000" data-desktop="3" data-tablet="2" data-mobile="1">
            <button class="carousel-arrow carousel-prev" type="button" aria-label="Previous"><i class="fas fa-chevron-left"></i></button>
            <div class="carousel-viewport">
                <div class="carousel-track">
                    <?php
                    $reviews = !empty($s['reviews_items']) ? $s['reviews_items'] : array();
                    foreach ($reviews as $review): ?>
                    <div class="carousel-slide">
                        <div class="review-card">
                            <div class="review-quote-mark">&ldquo;</div>
                            <div class="review-stars">
                                <?php
                                $stars = isset($review['stars']) ? intval($review['stars']) : 5;
                                for ($i = 0; $i < $stars; $i++): ?>
                                <i class="fas fa-star"></i>
                                <?php endfor; ?>
                            </div>
                            <p class="review-text"><em><?php echo esc_html($review['quote']); ?></em></p>
                            <div class="review-author">
                                <div class="review-avatar"><?php echo esc_html(mb_substr($review['name'], 0, 1)); ?></div>
                                <div class="review-info">
                                    <span class="review-name"><?php echo esc_html($review['name']); ?></span>
                                    <span class="review-location"><?php echo esc_html($review['location']); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <button class="carousel-arrow carousel-next" type="button" aria-label="Next"><i class="fas fa-chevron-right"></i></button>
            <div class="carousel-dots"></div>
        </div>

        <div class="google-badge">
            <span class="google-score"><?php echo esc_html(blp($s, 'reviews_google_score', '4.9')); ?></span>
            <div class="google-stars">
                <?php for ($i = 0; $i < 5; $i++): ?><i class="fas fa-star"></i><?php endfor; ?>
            </div>
            <span class="google-count"><?php echo esc_html(blp($s, 'reviews_google_count', '120+')); ?> Google Reviews</span>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 9: Why ACE
     ============================================ -->
<?php if (!empty($s['why_enabled'])): ?>
<?php
$why_headline = esc_html(blp($s, 'why_headline', 'Why 500+ LA Homeowners Chose ACE'));
$why_gold = esc_html(blp($s, 'why_gold_word', 'ACE'));
if ($why_gold) {
    $why_headline = str_replace($why_gold, '<span class="gold">' . $why_gold . '</span>', $why_headline);
}
?>
<section class="section why-section" id="why-ace">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'why_label', 'Why ACE')); ?></span>
        <h2 class="section-title"><?php echo $why_headline; ?></h2>
        <?php $why_sub = blp($s, 'why_subheadline', ''); ?>
        <?php if ($why_sub): ?>
        <p class="section-sub"><?php echo esc_html($why_sub); ?></p>
        <?php endif; ?>

        <div class="why-ace-grid">
            <?php
            $why_items = !empty($s['why_items']) ? $s['why_items'] : array();
            foreach ($why_items as $item): ?>
            <div class="why-ace-card">
                <div class="why-ace-icon"><i class="<?php echo esc_attr($item['icon']); ?>"></i></div>
                <h3><?php echo esc_html($item['title']); ?></h3>
                <p><?php echo esc_html($item['desc']); ?></p>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 10: Process
     ============================================ -->
<?php if (!empty($s['process_enabled'])): ?>
<?php
$process_headline = esc_html(blp($s, 'process_headline', 'From Vision to Backyard in 4 Steps'));
$process_gold = esc_html(blp($s, 'process_gold_word', '4 Steps'));
if ($process_gold) {
    $process_headline = str_replace($process_gold, '<span class="gold">' . $process_gold . '</span>', $process_headline);
}
?>
<section class="section process-section" id="process">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'process_label', 'Our Process')); ?></span>
        <h2 class="section-title"><?php echo $process_headline; ?></h2>

        <div class="process-steps">
            <?php
            $process_items = !empty($s['process_items']) ? $s['process_items'] : array();
            foreach ($process_items as $idx => $step):
                $step_num = str_pad($idx + 1, 2, '0', STR_PAD_LEFT);
            ?>
            <div class="process-step">
                <span class="process-ghost"><?php echo $step_num; ?></span>
                <div class="process-icon"><i class="<?php echo esc_attr($step['icon']); ?>"></i></div>
                <h4 class="process-title"><?php echo esc_html($step['title']); ?></h4>
                <p class="process-desc"><?php echo esc_html($step['desc']); ?></p>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 11: FAQ
     ============================================ -->
<?php if (!empty($s['faq_enabled'])): ?>
<?php
$faq_headline = esc_html(blp($s, 'faq_headline', 'Got Questions? We Have Answers.'));
$faq_gold = esc_html(blp($s, 'faq_gold_word', 'Answers.'));
if ($faq_gold) {
    $faq_headline = str_replace($faq_gold, '<span class="gold">' . $faq_gold . '</span>', $faq_headline);
}
?>
<section class="section faq-section" id="faq">
    <div class="container">
        <span class="section-label"><?php echo esc_html(blp($s, 'faq_label', 'FAQ')); ?></span>
        <h2 class="section-title"><?php echo $faq_headline; ?></h2>

        <div class="faq-list">
            <?php
            $faqs = !empty($s['faq_items']) ? $s['faq_items'] : array();
            foreach ($faqs as $i => $faq): ?>
            <div class="faq-item <?php echo ($i === 0) ? 'open' : ''; ?>">
                <button class="faq-q" type="button">
                    <span><?php echo esc_html($faq['q']); ?></span>
                    <i class="fas fa-plus"></i>
                </button>
                <div class="faq-a">
                    <p><?php echo esc_html($faq['a']); ?></p>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 12: Final CTA
     ============================================ -->
<?php if (!empty($s['final_enabled'])): ?>
<?php
$final_headline = esc_html(blp($s, 'final_headline', 'Your Backyard. Reimagined.'));
$final_gold = esc_html(blp($s, 'final_gold_word', 'Reimagined.'));
if ($final_gold) {
    $final_headline = str_replace($final_gold, '<span class="gold">' . $final_gold . '</span>', $final_headline);
}
?>
<section class="section final-section" id="final">
    <div class="container">
        <span class="hero-badge"><?php echo esc_html(blp($s, 'final_badge', 'Limited: Free 3D Rendering ($2,500 Value)')); ?></span>
        <h2 class="section-title"><?php echo $final_headline; ?></h2>
        <p class="section-sub"><?php echo esc_html(blp($s, 'final_subheadline', '')); ?></p>

        <div class="final-form-wrap">
            <form class="lp-form final-form" data-source="final">
                <input type="hidden" name="_nonce" value="<?php echo $nonce; ?>">
                <input type="hidden" name="source" value="final">
                <input type="text" name="name" placeholder="Your Name" required>
                <input type="tel" name="phone" placeholder="Phone Number" required>
                <input type="email" name="email" placeholder="Email Address">
                <input type="text" name="zip_code" placeholder="Project Zip Code">
                <textarea name="message" placeholder="Tell Us About Your Project" class="full-width"></textarea>
                <select name="heard_about" class="full-width">
                    <option value="">How did you hear about us?</option>
                    <?php
                    $heard = !empty($s['final_heard_options']) ? $s['final_heard_options'] : array();
                    foreach ($heard as $opt): ?>
                    <option value="<?php echo esc_attr($opt); ?>"><?php echo esc_html($opt); ?></option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="btn-gold btn-full full-width"><?php echo esc_html(blp($s, 'final_form_button', 'Book My Free In-Home Consultation')); ?></button>
                <div class="form-success" style="display:none;">
                    <i class="fas fa-check-circle"></i>
                    <p>Thank you! We will be in touch within 2 hours.</p>
                </div>
            </form>
        </div>

        <p class="final-phone">Or call us directly: <a href="<?php echo esc_attr($phone_link); ?>"><?php echo esc_html($phone); ?></a></p>
    </div>
</section>
<?php endif; ?>

<!-- ============================================
     Section 13: Footer
     ============================================ -->
<footer class="lp-footer">
    <div class="container">
        <?php
        $footer_logo_url    = blp($s, 'logo_white', '');
        $footer_logo_height = intval( blp($s, 'logo_white_height', 40) );
        $footer_logo_link   = blp($s, 'logo_white_link', '#hero');
        $footer_logo_scroll = ( strpos($footer_logo_link, '#') === 0 );
        ?>
        <?php if ( !empty($footer_logo_url) ) : ?>
        <div class="lp-footer-logo">
            <a href="<?php echo esc_attr($footer_logo_link); ?>"<?php if ($footer_logo_scroll) echo ' data-scroll="true"'; ?> aria-label="ACE Design &amp; Build">
                <img src="<?php echo esc_url($footer_logo_url); ?>" alt="ACE Design &amp; Build" loading="lazy" style="height: <?php echo $footer_logo_height; ?>px; width: auto;">
            </a>
        </div>
        <?php endif; ?>
        <p>&copy; <?php echo date('Y'); ?> ACE Design &amp; Build. All rights reserved. Licensed &amp; Insured. CSLB #XXXXXX</p>
    </div>
</footer>

<!-- ============================================
     Popup Modal
     ============================================ -->
<div class="popup-overlay" id="popupOverlay">
    <div class="popup-card">
        <button class="popup-close" type="button"><i class="fas fa-times"></i></button>
        <h3><?php echo esc_html(blp($s, 'popup_headline', 'Get Your Free Backyard Plan')); ?></h3>
        <p><?php echo esc_html(blp($s, 'popup_subtitle', '')); ?></p>
        <form class="lp-form popup-form" data-source="popup">
            <input type="hidden" name="_nonce" value="<?php echo $nonce; ?>">
            <input type="hidden" name="source" value="popup">
            <input type="text" name="name" placeholder="Your Name" required>
            <input type="tel" name="phone" placeholder="Phone Number" required>
            <input type="email" name="email" placeholder="Email Address">
            <input type="text" name="zip_code" placeholder="Project Zip Code">
            <textarea name="message" placeholder="Tell Us About Your Project" rows="3"></textarea>
            <button type="submit" class="btn-gold btn-full"><?php echo esc_html(blp($s, 'popup_button', 'Get My Free Plan')); ?></button>
            <div class="form-success" style="display:none;">
                <i class="fas fa-check-circle"></i>
                <p>Thank you! We will be in touch within 2 hours.</p>
            </div>
        </form>
    </div>
</div>

<!-- Sticky Mobile CTA removed - page has multiple inline CTAs -->

<!-- ============================================
     Promo Popup (AI Visualizer invitation)
     ============================================ -->
<?php if (get_option('ace_visualizer_enabled') && !empty($s['calc_enabled']) && get_option('ace_visualizer_promo_popup', 1)) : ?>
<?php
$vp_title   = get_option('ace_viz_promo_title', 'See Your Dream Backyard in 60 Seconds');
$vp_text    = get_option('ace_viz_promo_text', 'Upload a photo of your yard, pick your favorite features, and our AI creates a photorealistic design — matched to your budget.');
$vp_bullet1 = get_option('ace_viz_promo_bullet1', 'Free — no credit card, no commitment');
$vp_bullet2 = get_option('ace_viz_promo_bullet2', 'See exactly what your backyard could look like');
$vp_cta     = get_option('ace_viz_promo_cta', 'Try It Now — It\'s Free');
$vp_dismiss = get_option('ace_viz_promo_dismiss', 'Maybe later');
$vp_fine    = get_option('ace_viz_promo_fine', 'Takes less than 60 seconds');
?>
<div class="viz-promo-popup" id="vizPromoPopup">
    <div class="viz-promo-popup-backdrop" id="vizPromoBackdrop"></div>
    <div class="viz-promo-popup-content">
        <button type="button" class="viz-promo-popup-close" id="vizPromoClose" aria-label="Close">&times;</button>
        <div class="viz-promo-popup-icon">
            <i class="fas fa-wand-magic-sparkles"></i>
        </div>
        <h2 class="viz-promo-popup-title"><?php echo esc_html($vp_title); ?></h2>
        <p class="viz-promo-popup-text"><?php echo esc_html($vp_text); ?></p>
        <ul class="viz-promo-popup-bullets">
            <li><i class="fas fa-check"></i> <?php echo esc_html($vp_bullet1); ?></li>
            <li><i class="fas fa-check"></i> <?php echo esc_html($vp_bullet2); ?></li>
        </ul>
        <button type="button" class="viz-promo-popup-cta" id="vizPromoCta"><?php echo esc_html($vp_cta); ?></button>
        <button type="button" class="viz-promo-popup-dismiss" id="vizPromoDismiss"><?php echo esc_html($vp_dismiss); ?></button>
        <p class="viz-promo-popup-fine"><?php echo esc_html($vp_fine); ?></p>
    </div>
</div>
<?php endif; ?>

<!-- ============================================
     Footer Scripts
     ============================================ -->
<?php wp_footer(); ?>
<script>var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';</script>
<script src="<?php echo ACE_BLP_URL; ?>assets/js/landing-page.js?v=<?php echo ACE_BLP_VERSION; ?>"></script>
</body>
</html>
