/**
 * ACE Theme Manager - Admin JavaScript
 *
 * Handles media uploads, repeater fields, tab navigation,
 * toggle switches, service editing, confirm dialogs, and
 * basic form validation for the ACE admin panel.
 *
 * @package ACE_Theme_Manager
 * @since   1.0.0
 */

(function ($) {
	'use strict';

	/* ======================================================================
	   1. Media Uploader
	   ====================================================================== */

	/**
	 * Open the WordPress media frame when an upload button is clicked.
	 * On selection set the hidden input value and show a preview.
	 */
	function initMediaUploader() {
		$(document).on('click', '.ace-upload-btn', function (e) {
			e.preventDefault();

			var $button   = $(this);
			var $wrapper  = $button.closest('.ace-upload-field');
			var $input    = $wrapper.find('input[type="hidden"], input.ace-image-url');
			var $preview  = $wrapper.find('.ace-upload-preview');
			var $remove   = $wrapper.find('.ace-remove-image');
			var frameTitle  = $button.data('title')  || 'Select Image';
			var frameBtnTxt = $button.data('button') || 'Use Image';

			// Create a new media frame.
			var frame = wp.media({
				title:    frameTitle,
				library:  { type: 'image' },
				button:   { text: frameBtnTxt },
				multiple: false
			});

			// When an image is selected, populate the field.
			frame.on('select', function () {
				var attachment = frame.state().get('selection').first().toJSON();
				var url        = attachment.url;

				// Prefer medium size if available.
				if (attachment.sizes && attachment.sizes.medium) {
					url = attachment.sizes.medium.url;
				}

				$input.val(attachment.url).trigger('change');

				// Show or create the preview.
				if ($preview.length) {
					$preview.html('<img src="' + url + '" alt="">').show();
				} else {
					$button.after('<div class="ace-upload-preview"><img src="' + url + '" alt=""></div>');
				}

				$remove.show();
			});

			frame.open();
		});

		// Remove image handler.
		$(document).on('click', '.ace-remove-image', function (e) {
			e.preventDefault();

			var $wrapper = $(this).closest('.ace-upload-field');
			var $input   = $wrapper.find('input[type="hidden"], input.ace-image-url');
			var $preview = $wrapper.find('.ace-upload-preview');

			$input.val('').trigger('change');
			$preview.html('').hide();
			$(this).hide();
		});
	}

	/* ======================================================================
	   2. Repeater Fields
	   ====================================================================== */

	/**
	 * Manage repeater field add, remove, and drag-reorder functionality.
	 */
	function initRepeaterFields() {
		// Add new repeater item.
		$(document).on('click', '.ace-repeater-add', function (e) {
			e.preventDefault();

			var $repeater = $(this).closest('.ace-repeater');
			var $list     = $repeater.find('.ace-repeater-list');
			var $template = $repeater.find('.ace-repeater-template');

			if (!$template.length) {
				return;
			}

			// Determine the next index.
			var count   = $list.find('.ace-repeater-item').length;
			var newHtml = $template.html();

			// Replace __INDEX__ placeholders with the new index.
			newHtml = newHtml.replace(/__INDEX__/g, count);

			var $newItem = $('<div class="ace-repeater-item">' + newHtml + '</div>');
			$list.append($newItem);

			// Animate the new item in.
			$newItem.hide().slideDown(200);

			// Trigger a custom event so other scripts can react.
			$repeater.trigger('ace:repeater:added', [$newItem, count]);
		});

		// Remove a repeater item.
		$(document).on('click', '.ace-repeater-remove', function (e) {
			e.preventDefault();

			if (!confirm('Are you sure you want to remove this item?')) {
				return;
			}

			var $item     = $(this).closest('.ace-repeater-item');
			var $repeater = $item.closest('.ace-repeater');

			$item.slideUp(200, function () {
				$item.remove();
				reindexRepeater($repeater);
				$repeater.trigger('ace:repeater:removed');
			});
		});

		// Initialise jQuery UI Sortable on repeater lists for drag reorder.
		$('.ace-repeater .ace-repeater-list').each(function () {
			initSortable($(this));
		});
	}

	/**
	 * Initialise sortable on a repeater list element.
	 *
	 * @param {jQuery} $list The .ace-repeater-list element.
	 */
	function initSortable($list) {
		if (!$.fn.sortable) {
			return;
		}

		$list.sortable({
			handle:      '.handle',
			items:       '> .ace-repeater-item',
			axis:        'y',
			cursor:      'grabbing',
			placeholder: 'ace-repeater-item ui-sortable-placeholder',
			tolerance:   'pointer',
			opacity:     0.8,
			update: function () {
				var $repeater = $list.closest('.ace-repeater');
				reindexRepeater($repeater);
				$repeater.trigger('ace:repeater:reordered');
			}
		});
	}

	/**
	 * Re-index all name attributes in a repeater so indices stay sequential.
	 *
	 * Expects name attributes in the form: fieldname[0][subfield].
	 *
	 * @param {jQuery} $repeater The .ace-repeater container.
	 */
	function reindexRepeater($repeater) {
		$repeater.find('.ace-repeater-list > .ace-repeater-item').each(function (index) {
			$(this).find('input, select, textarea').each(function () {
				var name = $(this).attr('name');
				if (name) {
					// Replace the first [N] occurrence with the current index.
					name = name.replace(/\[\d+\]/, '[' + index + ']');
					$(this).attr('name', name);
				}

				var id = $(this).attr('id');
				if (id) {
					id = id.replace(/_\d+_/, '_' + index + '_');
					$(this).attr('id', id);
				}
			});
		});
	}

	/* ======================================================================
	   3. Tab Navigation
	   ====================================================================== */

	/**
	 * Handle sub-tab switching within admin pages.
	 * Clicking a .ace-sub-tab link shows the corresponding .ace-tab-content panel.
	 */
	function initTabNavigation() {
		$(document).on('click', '.ace-sub-tab', function (e) {
			e.preventDefault();

			var $link  = $(this);
			var target = $link.attr('href') || $link.data('tab');

			if (!target) {
				return;
			}

			// Remove the leading # if present.
			var tabId = target.replace(/^#/, '');

			// Update active class on tab links.
			$link.closest('.ace-admin-tabs').find('a').removeClass('active');
			$link.addClass('active');

			// Show the target panel, hide siblings.
			var $container = $link.closest('.ace-admin-tabs').parent();
			$container.find('.ace-tab-content').removeClass('active').hide();

			var $targetPanel = $container.find('#' + tabId);
			$targetPanel.addClass('active').fadeIn(150);

			// Store the active tab in the URL hash for bookmarking.
			if (window.history && window.history.replaceState) {
				window.history.replaceState(null, '', '#' + tabId);
			}
		});

		// On page load, activate the tab referenced in the URL hash.
		var hash = window.location.hash;
		if (hash) {
			var $tabLink = $('.ace-sub-tab[href="' + hash + '"], .ace-sub-tab[data-tab="' + hash.replace('#', '') + '"]');
			if ($tabLink.length) {
				$tabLink.trigger('click');
			}
		}
	}

	/* ======================================================================
	   4. Toggle Switches
	   ====================================================================== */

	/**
	 * Bind change events on toggle inputs for optional real-time feedback.
	 */
	function initToggleSwitches() {
		$(document).on('change', '.ace-toggle input', function () {
			var $toggle = $(this);
			var isOn    = $toggle.is(':checked');
			var $label  = $toggle.closest('.ace-toggle-label').find('.toggle-text');

			if ($label.length) {
				$label.text(isOn ? ($label.data('on') || 'Enabled') : ($label.data('off') || 'Disabled'));
			}

			// Trigger a custom event for other scripts to listen to.
			$toggle.trigger('ace:toggle:changed', [isOn]);
		});
	}

	/* ======================================================================
	   5. Service Editor
	   ====================================================================== */

	/**
	 * Handle service list interactions - click to load editor, save via form.
	 */
	function initServiceEditor() {
		// Click on a service item to load its editor.
		$(document).on('click', '.ace-service-item .service-actions .edit-service', function (e) {
			e.preventDefault();

			var $link    = $(this);
			var slug     = $link.data('slug');
			var $editor  = $('#ace-service-editor');

			if (!$editor.length || !slug) {
				// Fall through to standard link behaviour if no AJAX editor exists.
				window.location.href = $link.attr('href');
				return;
			}

			// Mark the selected service.
			$('.ace-service-item').removeClass('selected');
			$link.closest('.ace-service-item').addClass('selected');

			// Load the editor content via AJAX.
			$editor.html('<div style="text-align:center;padding:40px;"><span class="ace-loading"></span></div>');
			$editor.show();

			$.ajax({
				url:  ajaxurl,
				type: 'GET',
				data: {
					action: 'ace_load_service_editor',
					slug:   slug,
					_wpnonce: $link.data('nonce') || ''
				},
				success: function (response) {
					if (response.success && response.data && response.data.html) {
						$editor.html(response.data.html);
					} else {
						$editor.html('<div class="ace-notice error">Failed to load service editor.</div>');
					}
				},
				error: function () {
					$editor.html('<div class="ace-notice error">An error occurred while loading the editor.</div>');
				}
			});
		});

		// Save service data via AJAX when inside the AJAX editor.
		$(document).on('submit', '#ace-service-editor form.ace-service-form', function (e) {
			e.preventDefault();

			var $form   = $(this);
			var $submit = $form.find('.ace-submit-btn');
			var origText = $submit.val();

			$submit.val('Saving...').prop('disabled', true);

			$.ajax({
				url:  ajaxurl,
				type: 'POST',
				data: $form.serialize() + '&action=ace_save_service',
				success: function (response) {
					if (response.success) {
						$form.before('<div class="ace-notice success">Service saved successfully.</div>');
						setTimeout(function () {
							$form.siblings('.ace-notice').fadeOut(300, function () {
								$(this).remove();
							});
						}, 3000);
					} else {
						var msg = (response.data && response.data.message) ? response.data.message : 'Save failed.';
						$form.before('<div class="ace-notice error">' + msg + '</div>');
					}
				},
				error: function () {
					$form.before('<div class="ace-notice error">A server error occurred.</div>');
				},
				complete: function () {
					$submit.val(origText).prop('disabled', false);
				}
			});
		});
	}

	/* ======================================================================
	   6. Confirm Dialogs
	   ====================================================================== */

	/**
	 * Attach confirmation prompts to links and buttons with the
	 * data-confirm attribute.
	 */
	function initConfirmDialogs() {
		$(document).on('click', '[data-confirm]', function (e) {
			var message = $(this).data('confirm') || 'Are you sure you want to proceed?';

			if (!confirm(message)) {
				e.preventDefault();
				e.stopImmediatePropagation();
				return false;
			}
		});

		// Specific handler for lead deletion.
		$(document).on('click', '.ace-delete-lead', function (e) {
			if (!$(this).data('confirm')) {
				if (!confirm('Are you sure you want to delete this lead? This action cannot be undone.')) {
					e.preventDefault();
					return false;
				}
			}
		});

		// Specific handler for service deletion.
		$(document).on('click', '.ace-delete-service', function (e) {
			if (!$(this).data('confirm')) {
				if (!confirm('Are you sure you want to delete this service? All associated content will be removed.')) {
					e.preventDefault();
					return false;
				}
			}
		});
	}

	/* ======================================================================
	   7. Form Validation
	   ====================================================================== */

	/**
	 * Basic client-side validation for ACE admin forms.
	 * Validates required fields, email formats, and URL formats
	 * before allowing the form to submit.
	 */
	function initFormValidation() {
		$(document).on('submit', 'form.ace-admin-form', function (e) {
			var $form    = $(this);
			var isValid  = true;
			var $firstError = null;

			// Clear previous validation errors.
			$form.find('.ace-field-error').remove();
			$form.find('.ace-invalid').removeClass('ace-invalid');

			// Validate required fields.
			$form.find('[required]').each(function () {
				var $field = $(this);
				var value  = $.trim($field.val());

				if (value === '') {
					markInvalid($field, 'This field is required.');
					isValid = false;
					if (!$firstError) {
						$firstError = $field;
					}
				}
			});

			// Validate email fields.
			$form.find('input[type="email"]').each(function () {
				var $field = $(this);
				var value  = $.trim($field.val());

				if (value !== '' && !isValidEmail(value)) {
					markInvalid($field, 'Please enter a valid email address.');
					isValid = false;
					if (!$firstError) {
						$firstError = $field;
					}
				}
			});

			// Validate URL fields.
			$form.find('input[type="url"]').each(function () {
				var $field = $(this);
				var value  = $.trim($field.val());

				if (value !== '' && !isValidUrl(value)) {
					markInvalid($field, 'Please enter a valid URL.');
					isValid = false;
					if (!$firstError) {
						$firstError = $field;
					}
				}
			});

			// Validate number fields respect min/max.
			$form.find('input[type="number"]').each(function () {
				var $field = $(this);
				var value  = $field.val();
				var min    = $field.attr('min');
				var max    = $field.attr('max');

				if (value !== '') {
					var num = parseFloat(value);

					if (isNaN(num)) {
						markInvalid($field, 'Please enter a valid number.');
						isValid = false;
					} else if (min !== undefined && num < parseFloat(min)) {
						markInvalid($field, 'Value must be at least ' + min + '.');
						isValid = false;
					} else if (max !== undefined && num > parseFloat(max)) {
						markInvalid($field, 'Value must be at most ' + max + '.');
						isValid = false;
					}

					if (!isValid && !$firstError) {
						$firstError = $field;
					}
				}
			});

			if (!isValid) {
				e.preventDefault();

				// Scroll to the first error.
				if ($firstError) {
					$('html, body').animate({
						scrollTop: $firstError.offset().top - 100
					}, 300);
					$firstError.focus();
				}

				return false;
			}
		});

		// Remove error styling when a field is changed.
		$(document).on('input change', '.ace-invalid', function () {
			$(this).removeClass('ace-invalid');
			$(this).next('.ace-field-error').remove();
		});
	}

	/**
	 * Mark a field as invalid and show an error message below it.
	 *
	 * @param {jQuery} $field  The input/select/textarea element.
	 * @param {string} message The validation error message.
	 */
	function markInvalid($field, message) {
		$field.addClass('ace-invalid');

		// Only add the message if one is not already present.
		if (!$field.next('.ace-field-error').length) {
			$field.after('<span class="ace-field-error" style="color:#dc2626;font-size:12px;display:block;margin-top:4px;">' + message + '</span>');
		}
	}

	/**
	 * Basic email validation.
	 *
	 * @param  {string}  email
	 * @return {boolean}
	 */
	function isValidEmail(email) {
		return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
	}

	/**
	 * Basic URL validation.
	 *
	 * @param  {string}  url
	 * @return {boolean}
	 */
	function isValidUrl(url) {
		try {
			new URL(url);
			return true;
		} catch (e) {
			return false;
		}
	}

	/* ======================================================================
	   8. Miscellaneous Helpers
	   ====================================================================== */

	/**
	 * Auto-dismiss admin notices after a timeout.
	 */
	function initNoticeDismiss() {
		$('.ace-notice[data-auto-dismiss]').each(function () {
			var $notice = $(this);
			var timeout = parseInt($notice.data('auto-dismiss'), 10) || 5000;

			setTimeout(function () {
				$notice.fadeOut(300, function () {
					$notice.remove();
				});
			}, timeout);
		});
	}

	/**
	 * Conditionally show/hide fields based on a toggle or select value.
	 * Uses data-ace-show-when="fieldname:value" on a container element.
	 */
	function initConditionalFields() {
		$('[data-ace-show-when]').each(function () {
			var $target = $(this);
			var rule    = $target.data('ace-show-when');

			if (!rule || rule.indexOf(':') === -1) {
				return;
			}

			var parts     = rule.split(':');
			var fieldName = parts[0];
			var expected  = parts[1];
			var $field    = $('[name="' + fieldName + '"]');

			if (!$field.length) {
				return;
			}

			// Evaluate on load and on change.
			function evaluate() {
				var value;

				if ($field.is(':checkbox')) {
					value = $field.is(':checked') ? '1' : '0';
				} else {
					value = $field.val();
				}

				if (value === expected) {
					$target.slideDown(150);
				} else {
					$target.slideUp(150);
				}
			}

			evaluate();
			$field.on('change input', evaluate);
		});
	}

	/* ======================================================================
	   DOM Ready - Initialise all modules
	   ====================================================================== */

	$(document).ready(function () {
		initMediaUploader();
		initRepeaterFields();
		initTabNavigation();
		initToggleSwitches();
		initServiceEditor();
		initConfirmDialogs();
		initFormValidation();
		initNoticeDismiss();
		initConditionalFields();
	});

})(jQuery);
