<?php
/**
 * ACE Admin Page
 * Handles all admin panel rendering and settings saving
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ACE_Admin_Page {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_pages' ) );
        add_action( 'admin_init', array( $this, 'handle_save' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
    }

    public function add_admin_pages() {
        add_menu_page(
            __( 'ACE Theme Manager', 'ace-theme-manager' ),
            __( 'ACE Theme', 'ace-theme-manager' ),
            'manage_options',
            'ace-theme-manager',
            array( $this, 'render_dashboard' ),
            'dashicons-building',
            3
        );

        $submenus = array(
            array( 'ace-theme-manager', __( 'Dashboard', 'ace-theme-manager' ), 'render_dashboard' ),
            array( 'ace-general', __( 'General Settings', 'ace-theme-manager' ), 'render_general' ),
            array( 'ace-homepage', __( 'Homepage', 'ace-theme-manager' ), 'render_homepage' ),
            array( 'ace-services', __( 'Services', 'ace-theme-manager' ), 'render_services' ),
            array( 'ace-about', __( 'About Page', 'ace-theme-manager' ), 'render_about' ),
            array( 'ace-testimonials', __( 'Testimonials', 'ace-theme-manager' ), 'render_testimonials' ),
            array( 'ace-leads', __( 'Leads', 'ace-theme-manager' ), 'render_leads' ),
            array( 'ace-lead-form', __( 'Form Settings', 'ace-theme-manager' ), 'render_form_settings' ),
            array( 'ace-tracking', __( 'Tracking & Scripts', 'ace-theme-manager' ), 'render_tracking' ),
            array( 'ace-visibility', __( 'Section Visibility', 'ace-theme-manager' ), 'render_visibility' ),
            array( 'ace-seo', __( 'SEO Settings', 'ace-theme-manager' ), 'render_seo' ),
            array( 'ace-import-export', __( 'Import / Export', 'ace-theme-manager' ), 'render_import_export' ),
        );

        foreach ( $submenus as $sub ) {
            add_submenu_page(
                'ace-theme-manager',
                $sub[1],
                $sub[1],
                'manage_options',
                $sub[0],
                array( $this, $sub[2] )
            );
        }
    }

    public function enqueue_admin_assets( $hook ) {
        if ( strpos( $hook, 'ace-' ) === false && strpos( $hook, 'ace_' ) === false ) {
            return;
        }

        wp_enqueue_media();
        wp_enqueue_style( 'ace-admin-css', ACE_PLUGIN_URL . 'assets/css/admin.css', array(), ACE_PLUGIN_VERSION );
        wp_enqueue_script( 'ace-admin-js', ACE_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery', 'jquery-ui-sortable' ), ACE_PLUGIN_VERSION, true );
        wp_localize_script( 'ace-admin-js', 'aceAdmin', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'ace_admin_nonce' ),
        ) );
    }

    private function render_header( $title = '' ) {
        ?>
        <div class="ace-admin-header">
            <h1><?php echo esc_html( $title ?: 'ACE Theme Manager' ); ?></h1>
            <span class="tagline"><?php esc_html_e( 'Create with Confidence', 'ace-theme-manager' ); ?></span>
        </div>
        <?php
        // Show save notice
        if ( isset( $_GET['saved'] ) && $_GET['saved'] === '1' ) {
            echo '<div class="ace-notice success"><p>' . esc_html__( 'Settings saved successfully.', 'ace-theme-manager' ) . '</p></div>';
        }
    }

    private function media_upload_field( $name, $value, $label = '' ) {
        ?>
        <div class="ace-media-field">
            <?php if ( $label ) : ?>
                <label><strong><?php echo esc_html( $label ); ?></strong></label>
            <?php endif; ?>
            <input type="hidden" name="<?php echo esc_attr( $name ); ?>" value="<?php echo esc_url( $value ); ?>" class="ace-media-input">
            <button type="button" class="ace-upload-btn"><?php esc_html_e( 'Upload Image', 'ace-theme-manager' ); ?></button>
            <?php if ( $value ) : ?>
                <div class="ace-upload-preview">
                    <img src="<?php echo esc_url( $value ); ?>" alt="">
                </div>
                <button type="button" class="ace-remove-image"><?php esc_html_e( 'Remove', 'ace-theme-manager' ); ?></button>
            <?php else : ?>
                <div class="ace-upload-preview" style="display:none;"><img src="" alt=""></div>
                <button type="button" class="ace-remove-image" style="display:none;"><?php esc_html_e( 'Remove', 'ace-theme-manager' ); ?></button>
            <?php endif; ?>
        </div>
        <?php
    }

    private function text_field( $name, $value, $label, $desc = '', $type = 'text' ) {
        ?>
        <tr>
            <th scope="row"><label for="<?php echo esc_attr( $name ); ?>"><?php echo esc_html( $label ); ?></label></th>
            <td>
                <input type="<?php echo esc_attr( $type ); ?>" id="<?php echo esc_attr( $name ); ?>" name="<?php echo esc_attr( $name ); ?>" value="<?php echo esc_attr( $value ); ?>" class="regular-text">
                <?php if ( $desc ) : ?>
                    <p class="description"><?php echo esc_html( $desc ); ?></p>
                <?php endif; ?>
            </td>
        </tr>
        <?php
    }

    private function textarea_field( $name, $value, $label, $desc = '', $rows = 4 ) {
        ?>
        <tr>
            <th scope="row"><label for="<?php echo esc_attr( $name ); ?>"><?php echo esc_html( $label ); ?></label></th>
            <td>
                <textarea id="<?php echo esc_attr( $name ); ?>" name="<?php echo esc_attr( $name ); ?>" rows="<?php echo intval( $rows ); ?>" class="large-text"><?php echo esc_textarea( $value ); ?></textarea>
                <?php if ( $desc ) : ?>
                    <p class="description"><?php echo esc_html( $desc ); ?></p>
                <?php endif; ?>
            </td>
        </tr>
        <?php
    }

    // ========== DASHBOARD ==========
    public function render_dashboard() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'ACE Theme Manager - Dashboard' ); ?>

            <div class="ace-stats-row">
                <?php
                global $wpdb;
                $table = $wpdb->prefix . 'ace_leads';
                $total_leads = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
                $unread_leads = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE is_read = %d", 0 ) );
                $services = get_option( 'ace_services', array() );
                ?>
                <div class="ace-stat-card">
                    <div class="number"><?php echo intval( $total_leads ); ?></div>
                    <div class="label"><?php esc_html_e( 'Total Leads', 'ace-theme-manager' ); ?></div>
                </div>
                <div class="ace-stat-card">
                    <div class="number"><?php echo intval( $unread_leads ); ?></div>
                    <div class="label"><?php esc_html_e( 'Unread Leads', 'ace-theme-manager' ); ?></div>
                </div>
                <div class="ace-stat-card">
                    <div class="number"><?php echo count( $services ); ?></div>
                    <div class="label"><?php esc_html_e( 'Active Services', 'ace-theme-manager' ); ?></div>
                </div>
                <div class="ace-stat-card">
                    <div class="number"><?php echo intval( wp_count_posts( 'ace_project' )->publish ); ?></div>
                    <div class="label"><?php esc_html_e( 'Portfolio Projects', 'ace-theme-manager' ); ?></div>
                </div>
            </div>

            <div class="ace-form-section">
                <h2><?php esc_html_e( 'Quick Links', 'ace-theme-manager' ); ?></h2>
                <div class="ace-services-list">
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-general' ) ); ?>" class="ace-service-item">
                        <span class="service-name"><?php esc_html_e( 'General Settings', 'ace-theme-manager' ); ?></span>
                        <span class="service-actions">&rarr;</span>
                    </a>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-homepage' ) ); ?>" class="ace-service-item">
                        <span class="service-name"><?php esc_html_e( 'Edit Homepage', 'ace-theme-manager' ); ?></span>
                        <span class="service-actions">&rarr;</span>
                    </a>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-services' ) ); ?>" class="ace-service-item">
                        <span class="service-name"><?php esc_html_e( 'Manage Services', 'ace-theme-manager' ); ?></span>
                        <span class="service-actions">&rarr;</span>
                    </a>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-leads' ) ); ?>" class="ace-service-item">
                        <span class="service-name"><?php esc_html_e( 'View Leads', 'ace-theme-manager' ); ?></span>
                        <span class="service-actions">&rarr;</span>
                    </a>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-testimonials' ) ); ?>" class="ace-service-item">
                        <span class="service-name"><?php esc_html_e( 'Testimonials', 'ace-theme-manager' ); ?></span>
                        <span class="service-actions">&rarr;</span>
                    </a>
                    <a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="ace-service-item" target="_blank">
                        <span class="service-name"><?php esc_html_e( 'View Website', 'ace-theme-manager' ); ?></span>
                        <span class="service-actions">&rarr;</span>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    // ========== GENERAL SETTINGS ==========
    public function render_general() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'General Settings' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="general">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Logo & Branding', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Primary Logo', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_logo', get_option( 'ace_logo', '' ) ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'White Logo (Footer)', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_logo_white', get_option( 'ace_logo_white', '' ) ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Favicon', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_favicon', get_option( 'ace_favicon', '' ) ); ?></td>
                        </tr>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Company Information', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_company_name', get_option( 'ace_company_name', 'ACE Design & Build' ), 'Company Name' );
                        $this->text_field( 'ace_tagline', get_option( 'ace_tagline', 'Create with Confidence' ), 'Tagline' );
                        $this->text_field( 'ace_phone', get_option( 'ace_phone', '(310) 438-6866' ), 'Phone Number' );
                        $this->text_field( 'ace_email', get_option( 'ace_email', 'info@acedesignbuild.com' ), 'Email', '', 'email' );
                        $this->textarea_field( 'ace_address', get_option( 'ace_address', '6100 Wilshire Blvd #1200, Los Angeles, CA 90048, USA' ), 'Address', '', 2 );
                        $this->text_field( 'ace_business_hours', get_option( 'ace_business_hours', 'Monday - Sunday, 10:00 AM - 10:00 PM' ), 'Business Hours' );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Social Media URLs', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $socials = array( 'facebook' => 'Facebook', 'instagram' => 'Instagram', 'houzz' => 'Houzz', 'google' => 'Google Business', 'yelp' => 'Yelp', 'youtube' => 'YouTube' );
                        foreach ( $socials as $key => $label ) :
                            $this->text_field( 'ace_social_' . $key, get_option( 'ace_social_' . $key, '' ), $label . ' URL', '', 'url' );
                        endforeach;
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Google Maps', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php $this->textarea_field( 'ace_google_maps_embed', get_option( 'ace_google_maps_embed', '' ), 'Google Maps Embed Code', 'Paste your Google Maps iframe embed code here.', 4 ); ?>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Settings', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== HOMEPAGE ==========
    public function render_homepage() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Homepage Settings' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="homepage">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Hero Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Background Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_hero_image', get_option( 'ace_hero_image', '' ) ); ?></td>
                        </tr>
                        <?php
                        $this->text_field( 'ace_hero_headline', get_option( 'ace_hero_headline', "Los Angeles' Premier Design & Build Firm" ), 'Headline' );
                        $this->textarea_field( 'ace_hero_subheadline', get_option( 'ace_hero_subheadline', 'From dream kitchens to complete home transformations - we design, build, and deliver with precision. 15 years. Thousands of happy homeowners. Zero compromises.' ), 'Subheadline', '', 3 );
                        $this->text_field( 'ace_hero_cta_primary_text', get_option( 'ace_hero_cta_primary_text', 'Schedule Your Free Consultation' ), 'Primary CTA Text' );
                        $this->text_field( 'ace_hero_cta_secondary_text', get_option( 'ace_hero_cta_secondary_text', 'View Our Work' ), 'Secondary CTA Text' );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Services Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_services_title', get_option( 'ace_services_title', 'What We Build' ), 'Title' );
                        $this->textarea_field( 'ace_services_subtitle', get_option( 'ace_services_subtitle', 'Comprehensive residential construction and remodeling services, from concept to completion.' ), 'Subtitle', '', 2 );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Why ACE Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php $this->text_field( 'ace_why_ace_title', get_option( 'ace_why_ace_title', 'Why Los Angeles Homeowners Choose ACE' ), 'Title' ); ?>
                        <tr>
                            <th><?php esc_html_e( 'Side Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_why_ace_image', get_option( 'ace_why_ace_image', '' ) ); ?></td>
                        </tr>
                    </table>
                    <p class="description"><?php esc_html_e( 'Feature blocks are managed in the default content. To customize, edit the ace_why_ace_features option.', 'ace-theme-manager' ); ?></p>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Stats Section', 'ace-theme-manager' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'Stats are pre-configured: 2,000+ Projects, 15+ Years, 4.9 Rating, 98% Satisfaction.', 'ace-theme-manager' ); ?></p>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Featured Projects', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_featured_title', get_option( 'ace_featured_title', 'Our Recent Work' ), 'Title' );
                        $this->textarea_field( 'ace_featured_subtitle', get_option( 'ace_featured_subtitle', 'See the craftsmanship and attention to detail that defines every ACE project.' ), 'Subtitle', '', 2 );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Process Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_process_title', get_option( 'ace_process_title', 'How It Works' ), 'Title' );
                        $this->textarea_field( 'ace_process_subtitle', get_option( 'ace_process_subtitle', "From your first call to your final walkthrough - here's what working with ACE looks like." ), 'Subtitle', '', 2 );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'CTA Banner', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_cta_headline', get_option( 'ace_cta_headline', 'Ready to Build Something Extraordinary?' ), 'Headline' );
                        $this->textarea_field( 'ace_cta_subheadline', get_option( 'ace_cta_subheadline', "Schedule your free in-home consultation today. Let's turn your vision into reality." ), 'Subheadline', '', 2 );
                        $this->text_field( 'ace_cta_button_text', get_option( 'ace_cta_button_text', 'Get Your Free Consultation' ), 'Button Text' );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Service Area', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php $this->text_field( 'ace_service_area_title', get_option( 'ace_service_area_title', 'Proudly Serving Los Angeles & Surrounding Areas' ), 'Title' ); ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Blog Preview', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_blog_title', get_option( 'ace_blog_title', 'Expert Insights & Remodeling Tips' ), 'Title' );
                        $this->text_field( 'ace_blog_count', get_option( 'ace_blog_count', '3' ), 'Number of Posts', '', 'number' );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Final CTA + Form', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_final_cta_headline', get_option( 'ace_final_cta_headline', "Let's Start Your Project" ), 'Headline' );
                        $this->textarea_field( 'ace_final_cta_description', get_option( 'ace_final_cta_description', 'Get a free, no-obligation estimate for your remodeling or construction project. Our team responds within 2 hours during business hours.' ), 'Description', '', 3 );
                        ?>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Homepage Settings', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== SERVICES ==========
    public function render_services() {
        $services = get_option( 'ace_services', array() );
        $edit_slug = isset( $_GET['edit'] ) ? sanitize_text_field( wp_unslash( $_GET['edit'] ) ) : '';

        if ( $edit_slug ) {
            $this->render_service_editor( $edit_slug, $services );
            return;
        }
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Manage Services' ); ?>
            <div class="ace-form-section">
                <h2><?php esc_html_e( 'All Services', 'ace-theme-manager' ); ?></h2>
                <div class="ace-services-list">
                    <?php foreach ( $services as $service ) : ?>
                        <div class="ace-service-item">
                            <span class="service-name"><?php echo esc_html( $service['title'] ); ?></span>
                            <span class="service-actions">
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-services&edit=' . $service['slug'] ) ); ?>"><?php esc_html_e( 'Edit', 'ace-theme-manager' ); ?></a>
                                |
                                <a href="<?php echo esc_url( home_url( '/services/' . $service['slug'] . '/' ) ); ?>" target="_blank"><?php esc_html_e( 'View', 'ace-theme-manager' ); ?></a>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_service_editor( $slug, $services ) {
        $service = null;
        $service_index = -1;
        foreach ( $services as $i => $s ) {
            if ( $s['slug'] === $slug ) {
                $service = $s;
                $service_index = $i;
                break;
            }
        }
        if ( ! $service ) {
            echo '<div class="wrap"><p>' . esc_html__( 'Service not found.', 'ace-theme-manager' ) . '</p></div>';
            return;
        }
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Edit Service: ' . $service['title'] ); ?>
            <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=ace-services' ) ); ?>">&larr; <?php esc_html_e( 'Back to Services', 'ace-theme-manager' ); ?></a></p>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="service">
                <input type="hidden" name="service_slug" value="<?php echo esc_attr( $slug ); ?>">
                <input type="hidden" name="service_index" value="<?php echo intval( $service_index ); ?>">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Hero Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Hero Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'service_hero_image', isset( $service['hero_image'] ) ? $service['hero_image'] : '' ); ?></td>
                        </tr>
                        <?php
                        $this->text_field( 'service_h1', isset( $service['h1'] ) ? $service['h1'] : '', 'H1 Title' );
                        $this->text_field( 'service_hero_subheadline', isset( $service['hero_subheadline'] ) ? $service['hero_subheadline'] : '', 'Hero Subheadline' );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Introduction', 'ace-theme-manager' ); ?></h2>
                    <?php
                    wp_editor(
                        isset( $service['introduction'] ) ? $service['introduction'] : '',
                        'service_introduction',
                        array( 'textarea_name' => 'service_introduction', 'media_buttons' => true, 'textarea_rows' => 12 )
                    );
                    ?>
                    <table class="form-table" style="margin-top: 16px;">
                        <tr>
                            <th><?php esc_html_e( 'Side Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'service_side_image', isset( $service['side_image'] ) ? $service['side_image'] : '' ); ?></td>
                        </tr>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( "What's Included", 'ace-theme-manager' ); ?></h2>
                    <div class="ace-repeater" id="included-repeater">
                        <?php
                        $included = isset( $service['whats_included'] ) ? $service['whats_included'] : array();
                        foreach ( $included as $idx => $item ) : ?>
                            <div class="ace-repeater-item">
                                <span class="handle">&#9776;</span>
                                <div class="item-fields single-col">
                                    <input type="text" name="service_whats_included[]" value="<?php echo esc_attr( $item ); ?>">
                                </div>
                                <button type="button" class="ace-repeater-remove">&times;</button>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <button type="button" class="ace-repeater-add" data-target="included-repeater" data-template="included"><?php esc_html_e( '+ Add Item', 'ace-theme-manager' ); ?></button>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'CTA Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'service_cta_headline', isset( $service['cta_headline'] ) ? $service['cta_headline'] : '', 'CTA Headline' );
                        $this->text_field( 'service_cta_subheadline', isset( $service['cta_subheadline'] ) ? $service['cta_subheadline'] : '', 'CTA Subheadline' );
                        ?>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Service', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== ABOUT ==========
    public function render_about() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'About Page Settings' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="about">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Hero Section', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Background Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_about_hero_image', get_option( 'ace_about_hero_image', '' ) ); ?></td>
                        </tr>
                        <?php
                        $this->text_field( 'ace_about_headline', get_option( 'ace_about_headline', "15 Years of Building Los Angeles' Finest Homes" ), 'Headline' );
                        $this->textarea_field( 'ace_about_subheadline', get_option( 'ace_about_subheadline', "We're not just contractors. We're designers, engineers, craftsmen, and partners in bringing your vision to life." ), 'Subheadline', '', 2 );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Our Story', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php $this->text_field( 'ace_about_story_title', get_option( 'ace_about_story_title', 'The ACE Story' ), 'Title' ); ?>
                    </table>
                    <?php
                    wp_editor(
                        get_option( 'ace_about_story', '' ),
                        'ace_about_story',
                        array( 'textarea_name' => 'ace_about_story', 'media_buttons' => true, 'textarea_rows' => 15 )
                    );
                    ?>
                    <table class="form-table" style="margin-top: 16px;">
                        <tr>
                            <th><?php esc_html_e( 'Story Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_about_story_image', get_option( 'ace_about_story_image', '' ) ); ?></td>
                        </tr>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save About Page', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== TESTIMONIALS ==========
    public function render_testimonials() {
        $testimonials = get_option( 'ace_testimonials', array() );
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Testimonials' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="testimonials">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Manage Testimonials', 'ace-theme-manager' ); ?></h2>
                    <div class="ace-repeater" id="testimonials-repeater">
                        <?php foreach ( $testimonials as $idx => $t ) : ?>
                            <div class="ace-repeater-item">
                                <span class="handle">&#9776;</span>
                                <div class="item-fields">
                                    <div>
                                        <label><?php esc_html_e( 'Client Name', 'ace-theme-manager' ); ?></label>
                                        <input type="text" name="testimonials[<?php echo intval( $idx ); ?>][name]" value="<?php echo esc_attr( $t['name'] ); ?>">
                                    </div>
                                    <div>
                                        <label><?php esc_html_e( 'Location', 'ace-theme-manager' ); ?></label>
                                        <input type="text" name="testimonials[<?php echo intval( $idx ); ?>][location]" value="<?php echo esc_attr( isset( $t['location'] ) ? $t['location'] : '' ); ?>">
                                    </div>
                                    <div>
                                        <label><?php esc_html_e( 'Project Type', 'ace-theme-manager' ); ?></label>
                                        <input type="text" name="testimonials[<?php echo intval( $idx ); ?>][project_type]" value="<?php echo esc_attr( isset( $t['project_type'] ) ? $t['project_type'] : '' ); ?>">
                                    </div>
                                    <div>
                                        <label><?php esc_html_e( 'Rating', 'ace-theme-manager' ); ?></label>
                                        <select name="testimonials[<?php echo intval( $idx ); ?>][rating]">
                                            <?php for ( $r = 5; $r >= 1; $r-- ) : ?>
                                                <option value="<?php echo intval( $r ); ?>"<?php selected( isset( $t['rating'] ) ? $t['rating'] : 5, $r ); ?>><?php echo intval( $r ); ?> Stars</option>
                                            <?php endfor; ?>
                                        </select>
                                    </div>
                                    <div style="grid-column: 1 / -1;">
                                        <label><?php esc_html_e( 'Testimonial Text', 'ace-theme-manager' ); ?></label>
                                        <textarea name="testimonials[<?php echo intval( $idx ); ?>][text]" rows="3"><?php echo esc_textarea( $t['text'] ); ?></textarea>
                                    </div>
                                </div>
                                <button type="button" class="ace-repeater-remove">&times;</button>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <button type="button" class="ace-repeater-add" data-target="testimonials-repeater" data-template="testimonial"><?php esc_html_e( '+ Add Testimonial', 'ace-theme-manager' ); ?></button>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Testimonials', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== LEADS ==========
    public function render_leads() {
        ?>
        <div class="wrap ace-leads-wrap">
            <?php $this->render_header( 'Lead Submissions' ); ?>
            <?php
            if ( ! class_exists( 'ACE_Lead_Table' ) ) {
                require_once ACE_PLUGIN_PATH . 'includes/class-lead-table.php';
            }
            $table = new ACE_Lead_Table();
            $table->prepare_items();
            ?>
            <form method="get">
                <input type="hidden" name="page" value="ace-leads">
                <?php
                $table->search_box( __( 'Search Leads', 'ace-theme-manager' ), 'lead-search' );
                $table->display();
                ?>
            </form>
        </div>
        <?php
    }

    // ========== FORM SETTINGS ==========
    public function render_form_settings() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Lead Form Settings' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="form">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Form Configuration', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->textarea_field( 'ace_form_recipients', get_option( 'ace_form_recipients', 'info@acedesignbuild.com' ), 'Email Recipients', 'One email per line. All recipients will receive lead notifications.', 3 );
                        $this->text_field( 'ace_form_thank_you_url', get_option( 'ace_form_thank_you_url', '/thank-you/' ), 'Thank You Page URL', 'Leave empty to show inline success message.' );
                        $this->textarea_field( 'ace_form_success_message', get_option( 'ace_form_success_message', "Thank you! Your request has been submitted. We'll be in touch within 2 hours during business hours." ), 'Success Message', '', 3 );
                        ?>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Form Settings', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== TRACKING ==========
    public function render_tracking() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Tracking & Scripts' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="tracking">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Analytics & Tracking', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->text_field( 'ace_ga4_id', get_option( 'ace_ga4_id', '' ), 'GA4 Measurement ID', 'e.g., G-XXXXXXXXXX' );
                        $this->text_field( 'ace_gtm_id', get_option( 'ace_gtm_id', '' ), 'GTM Container ID', 'e.g., GTM-XXXXXXX' );
                        $this->text_field( 'ace_fb_pixel_id', get_option( 'ace_fb_pixel_id', '' ), 'Facebook Pixel ID' );
                        $this->text_field( 'ace_gads_conversion_id', get_option( 'ace_gads_conversion_id', '' ), 'Google Ads Conversion ID', 'e.g., AW-XXXXXXXXX' );
                        $this->text_field( 'ace_gads_conversion_label', get_option( 'ace_gads_conversion_label', '' ), 'Google Ads Conversion Label' );
                        ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Custom Scripts', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php
                        $this->textarea_field( 'ace_custom_head_scripts', get_option( 'ace_custom_head_scripts', '' ), 'Head Scripts', 'Scripts added before &lt;/head&gt;', 6 );
                        $this->textarea_field( 'ace_custom_body_scripts', get_option( 'ace_custom_body_scripts', '' ), 'Body Open Scripts', 'Scripts added after &lt;body&gt;', 4 );
                        $this->textarea_field( 'ace_custom_footer_scripts', get_option( 'ace_custom_footer_scripts', '' ), 'Footer Scripts', 'Scripts added before &lt;/body&gt;', 4 );
                        ?>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Tracking Settings', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== VISIBILITY ==========
    public function render_visibility() {
        $sections = get_option( 'ace_sections', array() );
        $toggle_items = array(
            'home_hero'         => 'Homepage - Hero',
            'home_services'     => 'Homepage - Services',
            'home_why_ace'      => 'Homepage - Why ACE',
            'home_stats'        => 'Homepage - Stats Counter',
            'home_projects'     => 'Homepage - Featured Projects',
            'home_testimonials' => 'Homepage - Testimonials',
            'home_process'      => 'Homepage - Process',
            'home_cta'          => 'Homepage - CTA Banner',
            'home_service_area' => 'Homepage - Service Area',
            'home_blog'         => 'Homepage - Blog Preview',
            'home_lead_form'    => 'Homepage - Lead Form',
            'footer_cta'        => 'Footer CTA Section',
        );
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Section Visibility' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="visibility">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Toggle Sections On/Off', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <?php foreach ( $toggle_items as $key => $label ) :
                            $checked = ! isset( $sections[ $key ] ) || $sections[ $key ];
                            ?>
                            <tr>
                                <th><?php echo esc_html( $label ); ?></th>
                                <td>
                                    <label class="ace-toggle">
                                        <input type="checkbox" name="sections[<?php echo esc_attr( $key ); ?>]" value="1"<?php checked( $checked ); ?>>
                                        <span class="slider"></span>
                                    </label>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Global Elements', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Floating Mobile CTA Bar', 'ace-theme-manager' ); ?></th>
                            <td>
                                <label class="ace-toggle">
                                    <input type="checkbox" name="ace_mobile_cta_bar" value="1"<?php checked( get_option( 'ace_mobile_cta_bar', true ) ); ?>>
                                    <span class="slider"></span>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Announcement Bar', 'ace-theme-manager' ); ?></th>
                            <td>
                                <label class="ace-toggle">
                                    <input type="checkbox" name="ace_announcement_bar" value="1"<?php checked( get_option( 'ace_announcement_bar', false ) ); ?>>
                                    <span class="slider"></span>
                                </label>
                            </td>
                        </tr>
                        <?php
                        $this->text_field( 'ace_announcement_text', get_option( 'ace_announcement_text', '' ), 'Announcement Text' );
                        $this->text_field( 'ace_announcement_link', get_option( 'ace_announcement_link', '' ), 'Announcement Link', '', 'url' );
                        ?>
                    </table>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save Visibility', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== SEO ==========
    public function render_seo() {
        $seo = get_option( 'ace_seo', array() );
        ?>
        <div class="wrap">
            <?php $this->render_header( 'SEO Settings' ); ?>
            <form method="post" action="">
                <?php wp_nonce_field( 'ace_save_settings', 'ace_nonce' ); ?>
                <input type="hidden" name="ace_settings_section" value="seo">

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Default SEO', 'ace-theme-manager' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Default OG Image', 'ace-theme-manager' ); ?></th>
                            <td><?php $this->media_upload_field( 'ace_og_image', get_option( 'ace_og_image', '' ) ); ?></td>
                        </tr>
                    </table>
                </div>

                <div class="ace-form-section">
                    <h2><?php esc_html_e( 'Page-Specific Meta', 'ace-theme-manager' ); ?></h2>
                    <?php
                    $pages = array(
                        'home' => 'Homepage',
                        'about' => 'About Page',
                        'portfolio' => 'Portfolio',
                        'contact' => 'Contact',
                        'blog' => 'Blog',
                    );
                    foreach ( $pages as $key => $label ) :
                        ?>
                        <h3 style="margin-top: 20px;"><?php echo esc_html( $label ); ?></h3>
                        <table class="form-table">
                            <?php
                            $this->text_field( 'seo[' . $key . '_title]', isset( $seo[ $key . '_title' ] ) ? $seo[ $key . '_title' ] : '', 'Meta Title' );
                            $this->textarea_field( 'seo[' . $key . '_description]', isset( $seo[ $key . '_description' ] ) ? $seo[ $key . '_description' ] : '', 'Meta Description', '', 2 );
                            ?>
                        </table>
                    <?php endforeach; ?>
                </div>

                <p class="submit">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Save SEO Settings', 'ace-theme-manager' ); ?></button>
                </p>
            </form>
        </div>
        <?php
    }

    // ========== IMPORT / EXPORT ==========
    public function render_import_export() {
        ?>
        <div class="wrap">
            <?php $this->render_header( 'Import / Export' ); ?>

            <div class="ace-form-section">
                <h2><?php esc_html_e( 'Export Settings', 'ace-theme-manager' ); ?></h2>
                <p><?php esc_html_e( 'Download all ACE settings as a JSON file.', 'ace-theme-manager' ); ?></p>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <?php wp_nonce_field( 'ace_export_settings', 'ace_export_nonce' ); ?>
                    <input type="hidden" name="action" value="ace_export_settings">
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Export Settings', 'ace-theme-manager' ); ?></button>
                </form>
            </div>

            <div class="ace-form-section">
                <h2><?php esc_html_e( 'Import Settings', 'ace-theme-manager' ); ?></h2>
                <p class="description" style="color: #dc2626;"><?php esc_html_e( 'Warning: Importing will overwrite all existing settings.', 'ace-theme-manager' ); ?></p>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
                    <?php wp_nonce_field( 'ace_import_settings', 'ace_import_nonce' ); ?>
                    <input type="hidden" name="action" value="ace_import_settings">
                    <input type="file" name="ace_import_file" accept=".json" style="margin-bottom: 12px;">
                    <br>
                    <button type="submit" class="ace-submit-btn"><?php esc_html_e( 'Import Settings', 'ace-theme-manager' ); ?></button>
                </form>
            </div>
        </div>
        <?php
    }

    // ========== SAVE HANDLER ==========
    public function handle_save() {
        if ( ! isset( $_POST['ace_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ace_nonce'] ) ), 'ace_save_settings' ) ) {
            return;
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $section = isset( $_POST['ace_settings_section'] ) ? sanitize_text_field( wp_unslash( $_POST['ace_settings_section'] ) ) : '';

        switch ( $section ) {
            case 'general':
                $fields = array(
                    'ace_logo', 'ace_logo_white', 'ace_favicon',
                    'ace_company_name', 'ace_tagline', 'ace_phone', 'ace_email',
                    'ace_address', 'ace_business_hours', 'ace_google_maps_embed',
                    'ace_social_facebook', 'ace_social_instagram', 'ace_social_houzz',
                    'ace_social_google', 'ace_social_yelp', 'ace_social_youtube',
                );
                foreach ( $fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        if ( strpos( $field, 'logo' ) !== false || strpos( $field, 'favicon' ) !== false || strpos( $field, 'social' ) !== false ) {
                            update_option( $field, esc_url_raw( wp_unslash( $_POST[ $field ] ) ) );
                        } elseif ( $field === 'ace_google_maps_embed' ) {
                            update_option( $field, wp_unslash( $_POST[ $field ] ) ); // Allow HTML for embed
                        } elseif ( $field === 'ace_email' ) {
                            update_option( $field, sanitize_email( wp_unslash( $_POST[ $field ] ) ) );
                        } else {
                            update_option( $field, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
                        }
                    }
                }
                break;

            case 'homepage':
                $text_fields = array(
                    'ace_hero_headline', 'ace_hero_subheadline', 'ace_hero_cta_primary_text',
                    'ace_hero_cta_secondary_text', 'ace_services_title', 'ace_services_subtitle',
                    'ace_why_ace_title', 'ace_featured_title', 'ace_featured_subtitle',
                    'ace_process_title', 'ace_process_subtitle',
                    'ace_cta_headline', 'ace_cta_subheadline', 'ace_cta_button_text',
                    'ace_service_area_title', 'ace_blog_title', 'ace_blog_count',
                    'ace_final_cta_headline', 'ace_final_cta_description',
                );
                foreach ( $text_fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
                    }
                }
                $image_fields = array( 'ace_hero_image', 'ace_why_ace_image' );
                foreach ( $image_fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, esc_url_raw( wp_unslash( $_POST[ $field ] ) ) );
                    }
                }
                break;

            case 'service':
                $services = get_option( 'ace_services', array() );
                $index = isset( $_POST['service_index'] ) ? intval( $_POST['service_index'] ) : -1;
                if ( $index >= 0 && isset( $services[ $index ] ) ) {
                    if ( isset( $_POST['service_h1'] ) ) {
                        $services[ $index ]['h1'] = sanitize_text_field( wp_unslash( $_POST['service_h1'] ) );
                    }
                    if ( isset( $_POST['service_hero_subheadline'] ) ) {
                        $services[ $index ]['hero_subheadline'] = sanitize_text_field( wp_unslash( $_POST['service_hero_subheadline'] ) );
                    }
                    if ( isset( $_POST['service_hero_image'] ) ) {
                        $services[ $index ]['hero_image'] = esc_url_raw( wp_unslash( $_POST['service_hero_image'] ) );
                    }
                    if ( isset( $_POST['service_side_image'] ) ) {
                        $services[ $index ]['side_image'] = esc_url_raw( wp_unslash( $_POST['service_side_image'] ) );
                    }
                    if ( isset( $_POST['service_introduction'] ) ) {
                        $services[ $index ]['introduction'] = wp_kses_post( wp_unslash( $_POST['service_introduction'] ) );
                    }
                    if ( isset( $_POST['service_whats_included'] ) && is_array( $_POST['service_whats_included'] ) ) {
                        $services[ $index ]['whats_included'] = array_map( 'sanitize_text_field', wp_unslash( $_POST['service_whats_included'] ) );
                    }
                    if ( isset( $_POST['service_cta_headline'] ) ) {
                        $services[ $index ]['cta_headline'] = sanitize_text_field( wp_unslash( $_POST['service_cta_headline'] ) );
                    }
                    if ( isset( $_POST['service_cta_subheadline'] ) ) {
                        $services[ $index ]['cta_subheadline'] = sanitize_text_field( wp_unslash( $_POST['service_cta_subheadline'] ) );
                    }
                    update_option( 'ace_services', $services );
                }
                break;

            case 'about':
                $about_fields = array(
                    'ace_about_hero_image', 'ace_about_story_image',
                );
                foreach ( $about_fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, esc_url_raw( wp_unslash( $_POST[ $field ] ) ) );
                    }
                }
                $text_about = array( 'ace_about_headline', 'ace_about_subheadline', 'ace_about_story_title' );
                foreach ( $text_about as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
                    }
                }
                if ( isset( $_POST['ace_about_story'] ) ) {
                    update_option( 'ace_about_story', wp_kses_post( wp_unslash( $_POST['ace_about_story'] ) ) );
                }
                break;

            case 'testimonials':
                if ( isset( $_POST['testimonials'] ) && is_array( $_POST['testimonials'] ) ) {
                    $testimonials = array();
                    foreach ( wp_unslash( $_POST['testimonials'] ) as $t ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
                        $testimonials[] = array(
                            'name'         => sanitize_text_field( $t['name'] ),
                            'location'     => sanitize_text_field( isset( $t['location'] ) ? $t['location'] : '' ),
                            'project_type' => sanitize_text_field( isset( $t['project_type'] ) ? $t['project_type'] : '' ),
                            'rating'       => intval( isset( $t['rating'] ) ? $t['rating'] : 5 ),
                            'text'         => sanitize_textarea_field( $t['text'] ),
                            'avatar'       => isset( $t['avatar'] ) ? esc_url_raw( $t['avatar'] ) : '',
                        );
                    }
                    update_option( 'ace_testimonials', $testimonials );
                }
                break;

            case 'form':
                $form_fields = array( 'ace_form_recipients', 'ace_form_thank_you_url', 'ace_form_success_message' );
                foreach ( $form_fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, sanitize_textarea_field( wp_unslash( $_POST[ $field ] ) ) );
                    }
                }
                break;

            case 'tracking':
                $tracking_fields = array(
                    'ace_ga4_id', 'ace_gtm_id', 'ace_fb_pixel_id',
                    'ace_gads_conversion_id', 'ace_gads_conversion_label',
                );
                foreach ( $tracking_fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
                    }
                }
                $script_fields = array( 'ace_custom_head_scripts', 'ace_custom_body_scripts', 'ace_custom_footer_scripts' );
                foreach ( $script_fields as $field ) {
                    if ( isset( $_POST[ $field ] ) ) {
                        update_option( $field, wp_unslash( $_POST[ $field ] ) ); // Raw scripts
                    }
                }
                break;

            case 'visibility':
                $section_keys = array(
                    'home_hero', 'home_services', 'home_why_ace', 'home_stats',
                    'home_projects', 'home_testimonials', 'home_process', 'home_cta',
                    'home_service_area', 'home_blog', 'home_lead_form', 'footer_cta',
                );
                $sections_data = array();
                foreach ( $section_keys as $key ) {
                    $sections_data[ $key ] = isset( $_POST['sections'][ $key ] ) ? 1 : 0;
                }
                update_option( 'ace_sections', $sections_data );

                if ( isset( $_POST['ace_mobile_cta_bar'] ) ) {
                    update_option( 'ace_mobile_cta_bar', 1 );
                } else {
                    update_option( 'ace_mobile_cta_bar', 0 );
                }
                if ( isset( $_POST['ace_announcement_bar'] ) ) {
                    update_option( 'ace_announcement_bar', 1 );
                } else {
                    update_option( 'ace_announcement_bar', 0 );
                }
                if ( isset( $_POST['ace_announcement_text'] ) ) {
                    update_option( 'ace_announcement_text', sanitize_text_field( wp_unslash( $_POST['ace_announcement_text'] ) ) );
                }
                if ( isset( $_POST['ace_announcement_link'] ) ) {
                    update_option( 'ace_announcement_link', esc_url_raw( wp_unslash( $_POST['ace_announcement_link'] ) ) );
                }
                break;

            case 'seo':
                if ( isset( $_POST['ace_og_image'] ) ) {
                    update_option( 'ace_og_image', esc_url_raw( wp_unslash( $_POST['ace_og_image'] ) ) );
                }
                if ( isset( $_POST['seo'] ) && is_array( $_POST['seo'] ) ) {
                    $seo_data = array();
                    foreach ( wp_unslash( $_POST['seo'] ) as $key => $val ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
                        $seo_data[ sanitize_key( $key ) ] = sanitize_text_field( $val );
                    }
                    update_option( 'ace_seo', $seo_data );
                }
                break;
        }

        // Redirect back with saved notice
        $redirect_url = add_query_arg( 'saved', '1' );
        wp_safe_redirect( $redirect_url );
        exit;
    }
}
