<?php
/**
 * Lead List Table.
 *
 * Extends the core WP_List_Table class to display captured leads in a
 * standard WordPress admin table with sorting, searching, filtering,
 * bulk actions, and pagination.
 *
 * @package ACE_Theme_Manager
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP_List_Table may not be loaded yet in all contexts.
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class ACE_Lead_Table
 *
 * Renders the leads management table in the WordPress admin area using
 * the WP_List_Table pattern. Supports search, service filtering, status
 * views, sortable columns, bulk actions, and CSV export.
 *
 * @since 1.0.0
 */
class ACE_Lead_Table extends WP_List_Table {

	/**
	 * Number of items to display per page.
	 *
	 * @var int
	 */
	const PER_PAGE = 20;

	/**
	 * Constructor.
	 *
	 * Sets up the list table with singular/plural labels and disables
	 * AJAX pagination.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'lead',
				'plural'   => 'leads',
				'ajax'     => false,
			)
		);
	}

	/**
	 * Define the table columns.
	 *
	 * @since 1.0.0
	 *
	 * @return array Associative array of column slugs to display labels.
	 */
	public function get_columns() {
		return array(
			'cb'          => '<input type="checkbox" />',
			'full_name'   => __( 'Name', 'ace-theme-manager' ),
			'phone'       => __( 'Phone', 'ace-theme-manager' ),
			'email'       => __( 'Email', 'ace-theme-manager' ),
			'service'     => __( 'Service', 'ace-theme-manager' ),
			'source_page' => __( 'Source', 'ace-theme-manager' ),
			'created_at'  => __( 'Date', 'ace-theme-manager' ),
			'is_read'     => __( 'Status', 'ace-theme-manager' ),
		);
	}

	/**
	 * Define which columns are sortable.
	 *
	 * @since 1.0.0
	 *
	 * @return array Associative array of sortable column slugs.
	 */
	public function get_sortable_columns() {
		return array(
			'full_name'  => array( 'full_name', false ),
			'email'      => array( 'email', false ),
			'service'    => array( 'service', false ),
			'created_at' => array( 'created_at', true ), // Default descending.
			'is_read'    => array( 'is_read', false ),
		);
	}

	/**
	 * Render the default column output.
	 *
	 * Handles output for columns that do not have a dedicated
	 * `column_{name}` method.
	 *
	 * @since 1.0.0
	 *
	 * @param object $item        The current lead row object.
	 * @param string $column_name The column slug being rendered.
	 * @return string Formatted column content.
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'created_at':
				$timestamp = strtotime( $item->created_at );
				return esc_html(
					date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp )
				);

			case 'is_read':
				if ( (int) $item->is_read === 1 ) {
					return '<span class="ace-badge ace-badge--read" style="color:#46b450;">' . esc_html__( 'Read', 'ace-theme-manager' ) . '</span>';
				}
				return '<span class="ace-badge ace-badge--new" style="font-weight:700;color:#996800;background:#fcf0c3;padding:3px 8px;border-radius:3px;font-size:12px;">' . esc_html__( 'New', 'ace-theme-manager' ) . '</span>';

			case 'phone':
				$phone = esc_html( $item->phone );
				$tel   = preg_replace( '/[^+\d]/', '', $item->phone );
				return '<a href="tel:' . esc_attr( $tel ) . '">' . $phone . '</a>';

			case 'email':
				$email = esc_html( $item->email );
				return '<a href="mailto:' . esc_attr( $item->email ) . '">' . $email . '</a>';

			case 'source_page':
				return esc_html( $item->source_page );

			case 'service':
				return esc_html( $item->service );

			default:
				return isset( $item->$column_name ) ? esc_html( $item->$column_name ) : '';
		}
	}

	/**
	 * Render the Name column with row actions.
	 *
	 * Displays the lead name as the primary column with View, Mark
	 * Read/Unread, and Delete row actions.
	 *
	 * @since 1.0.0
	 *
	 * @param object $item The current lead row object.
	 * @return string Column HTML with row actions.
	 */
	public function column_full_name( $item ) {
		$lead_id = absint( $item->id );
		$name    = esc_html( $item->full_name );

		// Bold unread names for visual emphasis.
		if ( 0 === (int) $item->is_read ) {
			$name = '<strong>' . $name . '</strong>';
		}

		// Build the admin page URL base.
		$page_url = admin_url( 'admin.php?page=ace-leads' );

		// View action.
		$view_url = wp_nonce_url(
			add_query_arg(
				array(
					'action'  => 'view',
					'lead_id' => $lead_id,
				),
				$page_url
			),
			'ace_lead_action_' . $lead_id
		);

		// Toggle read/unread action.
		if ( (int) $item->is_read === 1 ) {
			$toggle_label = __( 'Mark Unread', 'ace-theme-manager' );
			$toggle_action = 'mark_unread';
		} else {
			$toggle_label = __( 'Mark Read', 'ace-theme-manager' );
			$toggle_action = 'mark_read';
		}

		$toggle_url = wp_nonce_url(
			add_query_arg(
				array(
					'action'  => $toggle_action,
					'lead_id' => $lead_id,
				),
				$page_url
			),
			'ace_lead_action_' . $lead_id
		);

		// Delete action.
		$delete_url = wp_nonce_url(
			add_query_arg(
				array(
					'action'  => 'delete',
					'lead_id' => $lead_id,
				),
				$page_url
			),
			'ace_lead_action_' . $lead_id
		);

		$actions = array(
			'view'          => sprintf( '<a href="%s">%s</a>', esc_url( $view_url ), esc_html__( 'View', 'ace-theme-manager' ) ),
			$toggle_action  => sprintf( '<a href="%s">%s</a>', esc_url( $toggle_url ), esc_html( $toggle_label ) ),
			'delete'        => sprintf(
				'<a href="%s" class="submitdelete" onclick="return confirm(\'%s\');">%s</a>',
				esc_url( $delete_url ),
				esc_js( __( 'Are you sure you want to delete this lead?', 'ace-theme-manager' ) ),
				esc_html__( 'Delete', 'ace-theme-manager' )
			),
		);

		return $name . $this->row_actions( $actions );
	}

	/**
	 * Render the checkbox column for bulk actions.
	 *
	 * @since 1.0.0
	 *
	 * @param object $item The current lead row object.
	 * @return string Checkbox HTML.
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="lead_ids[]" value="%d" />',
			absint( $item->id )
		);
	}

	/**
	 * Define available bulk actions.
	 *
	 * @since 1.0.0
	 *
	 * @return array Associative array of bulk action slugs to labels.
	 */
	public function get_bulk_actions() {
		return array(
			'mark_read'   => __( 'Mark as Read', 'ace-theme-manager' ),
			'mark_unread' => __( 'Mark as Unread', 'ace-theme-manager' ),
			'delete'      => __( 'Delete', 'ace-theme-manager' ),
		);
	}

	/**
	 * Process bulk actions.
	 *
	 * Handles mark_read, mark_unread, and delete operations for the
	 * selected lead IDs. Verifies the bulk action nonce before proceeding.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function process_bulk_action() {
		$action = $this->current_action();

		if ( ! $action ) {
			return;
		}

		// Verify the bulk action nonce.
		$nonce = isset( $_REQUEST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ) : '';

		if ( ! wp_verify_nonce( $nonce, 'bulk-leads' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'ace-theme-manager' ), 403 );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$lead_ids = isset( $_REQUEST['lead_ids'] ) ? array_map( 'absint', (array) $_REQUEST['lead_ids'] ) : array();

		if ( empty( $lead_ids ) ) {
			return;
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'ace_leads';

		switch ( $action ) {
			case 'mark_read':
				foreach ( $lead_ids as $id ) {
					$wpdb->update(
						$table_name,
						array( 'is_read' => 1 ),
						array( 'id' => $id ),
						array( '%d' ),
						array( '%d' )
					);
				}
				break;

			case 'mark_unread':
				foreach ( $lead_ids as $id ) {
					$wpdb->update(
						$table_name,
						array( 'is_read' => 0 ),
						array( 'id' => $id ),
						array( '%d' ),
						array( '%d' )
					);
				}
				break;

			case 'delete':
				if ( ! current_user_can( 'manage_options' ) ) {
					wp_die( esc_html__( 'You do not have permission to delete leads.', 'ace-theme-manager' ), 403 );
				}

				foreach ( $lead_ids as $id ) {
					$wpdb->delete(
						$table_name,
						array( 'id' => $id ),
						array( '%d' )
					);
				}
				break;
		}
	}

	/**
	 * Prepare table items for display.
	 *
	 * Queries the `ace_leads` database table with support for search,
	 * service filtering, status filtering, sorting, and pagination.
	 * Sets column headers, items, and pagination arguments.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function prepare_items() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';

		// Process any pending bulk actions first.
		$this->process_bulk_action();

		// Column configuration.
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );

		// Build the WHERE clause.
		$where  = array();
		$values = array();

		// Search filter (name, email, phone).
		$search = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
		if ( ! empty( $search ) ) {
			$like     = '%' . $wpdb->esc_like( $search ) . '%';
			$where[]  = '(full_name LIKE %s OR email LIKE %s OR phone LIKE %s)';
			$values[] = $like;
			$values[] = $like;
			$values[] = $like;
		}

		// Service filter.
		$service_filter = isset( $_REQUEST['service_filter'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['service_filter'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
		if ( ! empty( $service_filter ) ) {
			$where[]  = 'service = %s';
			$values[] = $service_filter;
		}

		// Status view filter (is_read).
		$status_filter = isset( $_REQUEST['lead_status'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['lead_status'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
		if ( 'unread' === $status_filter ) {
			$where[] = 'is_read = 0';
		} elseif ( 'read' === $status_filter ) {
			$where[] = 'is_read = 1';
		}

		$where_sql = '';
		if ( ! empty( $where ) ) {
			$where_sql = 'WHERE ' . implode( ' AND ', $where );
		}

		// Sorting.
		$allowed_orderby = array( 'full_name', 'email', 'service', 'created_at', 'is_read' );
		$orderby         = isset( $_REQUEST['orderby'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) ) : 'created_at'; // phpcs:ignore WordPress.Security.NonceVerification
		$orderby         = in_array( $orderby, $allowed_orderby, true ) ? $orderby : 'created_at';
		$order           = isset( $_REQUEST['order'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_REQUEST['order'] ) ) ) : 'DESC'; // phpcs:ignore WordPress.Security.NonceVerification
		$order           = in_array( $order, array( 'ASC', 'DESC' ), true ) ? $order : 'DESC';

		// Total items count for pagination.
		$count_sql = "SELECT COUNT(*) FROM {$table_name} {$where_sql}";

		if ( ! empty( $values ) ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			$total_items = (int) $wpdb->get_var( $wpdb->prepare( $count_sql, $values ) );
		} else {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			$total_items = (int) $wpdb->get_var( $count_sql );
		}

		// Pagination.
		$per_page     = self::PER_PAGE;
		$current_page = $this->get_pagenum();
		$offset       = ( $current_page - 1 ) * $per_page;

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $per_page,
				'total_pages' => ceil( $total_items / $per_page ),
			)
		);

		// Build the data query.
		$sql = "SELECT * FROM {$table_name} {$where_sql} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";

		$query_values   = $values;
		$query_values[] = $per_page;
		$query_values[] = $offset;

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$this->items = $wpdb->get_results( $wpdb->prepare( $sql, $query_values ) );
	}

	/**
	 * Render extra table navigation elements.
	 *
	 * Adds a service filter dropdown, filter button, and CSV export button
	 * above the table.
	 *
	 * @since 1.0.0
	 *
	 * @param string $which Position: 'top' or 'bottom'.
	 * @return void
	 */
	protected function extra_tablenav( $which ) {
		if ( 'top' !== $which ) {
			return;
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';

		// Fetch distinct service values for the filter dropdown.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$services = $wpdb->get_col( "SELECT DISTINCT service FROM {$table_name} WHERE service != '' ORDER BY service ASC" );

		$current_service = isset( $_REQUEST['service_filter'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['service_filter'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
		?>
		<div class="alignleft actions">
			<select name="service_filter" id="ace-service-filter">
				<option value=""><?php esc_html_e( 'All Services', 'ace-theme-manager' ); ?></option>
				<?php foreach ( $services as $service ) : ?>
					<option value="<?php echo esc_attr( $service ); ?>" <?php selected( $current_service, $service ); ?>>
						<?php echo esc_html( $service ); ?>
					</option>
				<?php endforeach; ?>
			</select>
			<?php
			submit_button( __( 'Filter', 'ace-theme-manager' ), '', 'filter_action', false, array( 'id' => 'ace-lead-filter-submit' ) );
			?>
		</div>

		<div class="alignleft actions">
			<?php
			$export_url = wp_nonce_url(
				admin_url( 'admin-post.php?action=ace_export_leads' ),
				'ace_export_leads',
				'_wpnonce'
			);
			?>
			<a href="<?php echo esc_url( $export_url ); ?>" class="button" id="ace-export-csv">
				<?php esc_html_e( 'Export CSV', 'ace-theme-manager' ); ?>
			</a>
		</div>
		<?php
	}

	/**
	 * Get the status filter views (All, Unread, Read).
	 *
	 * Returns an array of links that filter the table by read status,
	 * each displaying the count of matching leads.
	 *
	 * @since 1.0.0
	 *
	 * @return array Associative array of view slugs to HTML link strings.
	 */
	protected function get_views() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$total_count  = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$unread_count = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name} WHERE is_read = 0" );
		$read_count   = $total_count - $unread_count;

		$current_status = isset( $_REQUEST['lead_status'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['lead_status'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
		$base_url       = admin_url( 'admin.php?page=ace-leads' );

		$views = array();

		// All.
		$all_class = empty( $current_status ) ? ' class="current"' : '';
		$views['all'] = sprintf(
			'<a href="%s"%s>%s <span class="count">(%d)</span></a>',
			esc_url( $base_url ),
			$all_class,
			esc_html__( 'All', 'ace-theme-manager' ),
			$total_count
		);

		// Unread.
		$unread_class = ( 'unread' === $current_status ) ? ' class="current"' : '';
		$views['unread'] = sprintf(
			'<a href="%s"%s>%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'lead_status', 'unread', $base_url ) ),
			$unread_class,
			esc_html__( 'Unread', 'ace-theme-manager' ),
			$unread_count
		);

		// Read.
		$read_class = ( 'read' === $current_status ) ? ' class="current"' : '';
		$views['read'] = sprintf(
			'<a href="%s"%s>%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'lead_status', 'read', $base_url ) ),
			$read_class,
			esc_html__( 'Read', 'ace-theme-manager' ),
			$read_count
		);

		return $views;
	}

	/**
	 * Message displayed when there are no leads.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function no_items() {
		esc_html_e( 'No leads found.', 'ace-theme-manager' );
	}
}
