<?php
/**
 * Template helper functions for ACE Theme
 */

/**
 * Get theme option with fallback
 * Works with the ACE Theme Manager plugin options
 */
function ace_get_option( $key, $default = '' ) {
    $value = get_option( 'ace_' . $key, $default );
    return ( $value !== '' && $value !== false ) ? $value : $default;
}

/**
 * Get company phone number
 */
function ace_get_phone() {
    return ace_get_option( 'phone', '(310) 438-6866' );
}

/**
 * Get phone link (tel: format)
 */
function ace_get_phone_link() {
    $phone = ace_get_phone();
    return 'tel:' . preg_replace( '/[^0-9+]/', '', $phone );
}

/**
 * Get company email
 */
function ace_get_email() {
    return ace_get_option( 'email', 'info@acedesignbuild.com' );
}

/**
 * Get company address
 */
function ace_get_address() {
    return ace_get_option( 'address', '6100 Wilshire Blvd #1200, Los Angeles, CA 90048, USA' );
}

/**
 * Get business hours
 */
function ace_get_hours() {
    return ace_get_option( 'business_hours', 'Monday - Sunday, 10:00 AM - 10:00 PM' );
}

/**
 * Get company name
 */
function ace_get_company_name() {
    return ace_get_option( 'company_name', 'ACE Design & Build' );
}

/**
 * Get tagline
 */
function ace_get_tagline() {
    return ace_get_option( 'tagline', 'Create with Confidence' );
}

/**
 * Get logo URL from plugin option or custom logo
 */
function ace_get_logo_url() {
    $logo = ace_get_option( 'logo', '' );
    if ( $logo ) {
        return $logo;
    }
    $custom_logo_id = get_theme_mod( 'custom_logo' );
    if ( $custom_logo_id ) {
        return wp_get_attachment_image_url( $custom_logo_id, 'full' );
    }
    return ACE_THEME_URI . '/assets/images/logo-placeholder.svg';
}

/**
 * Get white logo URL
 */
function ace_get_logo_white_url() {
    return ace_get_option( 'logo_white', ace_get_logo_url() );
}

/**
 * Get social media URLs
 */
function ace_get_social_links() {
    return array(
        'facebook'  => ace_get_option( 'social_facebook', '' ),
        'instagram' => ace_get_option( 'social_instagram', '' ),
        'houzz'     => ace_get_option( 'social_houzz', '' ),
        'google'    => ace_get_option( 'social_google', '' ),
        'yelp'      => ace_get_option( 'social_yelp', '' ),
        'youtube'   => ace_get_option( 'social_youtube', '' ),
    );
}

/**
 * Render SVG icon by name
 * Icons: phone, email, clock, map-pin, check-circle, star, shield, award, home, hammer, ruler, users, arrow-right, quote, play-circle, calendar, dollar-sign, menu, x, chevron-down, facebook, instagram, youtube, yelp
 */
function ace_icon( $name, $size = 24, $class = '' ) {
    $icons = array(
        'phone' => '<path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"/>',
        'email' => '<path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"/><polyline points="22,6 12,13 2,6"/>',
        'clock' => '<circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/>',
        'map-pin' => '<path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/><circle cx="12" cy="10" r="3"/>',
        'check-circle' => '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/>',
        'star' => '<polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>',
        'shield' => '<path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>',
        'award' => '<circle cx="12" cy="8" r="7"/><polyline points="8.21 13.89 7 23 12 20 17 23 15.79 13.88"/>',
        'home' => '<path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/><polyline points="9 22 9 12 15 12 15 22"/>',
        'hammer' => '<path d="M15 12l-8.5 8.5c-.83.83-2.17.83-3 0 0 0 0 0 0 0a2.12 2.12 0 0 1 0-3L12 9"/><path d="M17.64 15L22 10.64"/><path d="M20.91 11.7l-1.25-1.25c-.6-.6-.93-1.4-.93-2.25V6.5l-3-3-2.3 2.3 1.6 1.6c.6.6.6 1.57 0 2.17L14 10.6"/>',
        'ruler' => '<path d="M21.21 15.89A1 1 0 0 0 22 15V9a1 1 0 0 0-.29-.71l-4-4a1 1 0 0 0-1.42 0l-8 8a1 1 0 0 0 0 1.42l4 4A1 1 0 0 0 13 18h6a1 1 0 0 0 .71-.29zM6.5 9L9 11.5"/><line x1="10" y1="5" x2="12" y2="7"/><line x1="3" y1="12" x2="5" y2="14"/>',
        'users' => '<path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M23 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/>',
        'arrow-right' => '<line x1="5" y1="12" x2="19" y2="12"/><polyline points="12 5 19 12 12 19"/>',
        'quote' => '<path d="M3 21c3 0 7-1 7-8V5c0-1.25-.756-2.017-2-2H4c-1.25 0-2 .75-2 1.972V11c0 1.25.75 2 2 2 1 0 1 0 1 1v1c0 1-1 2-2 2s-1 .008-1 1.031V21z"/><path d="M15 21c3 0 7-1 7-8V5c0-1.25-.757-2.017-2-2h-4c-1.25 0-2 .75-2 1.972V11c0 1.25.75 2 2 2h.75c0 2.25.25 4-2.75 4v3z"/>',
        'play-circle' => '<circle cx="12" cy="12" r="10"/><polygon points="10 8 16 12 10 16 10 8"/>',
        'calendar' => '<rect x="3" y="4" width="18" height="18" rx="2" ry="2"/><line x1="16" y1="2" x2="16" y2="6"/><line x1="8" y1="2" x2="8" y2="6"/><line x1="3" y1="10" x2="21" y2="10"/>',
        'dollar-sign' => '<line x1="12" y1="1" x2="12" y2="23"/><path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>',
        'menu' => '<line x1="3" y1="12" x2="21" y2="12"/><line x1="3" y1="6" x2="21" y2="6"/><line x1="3" y1="18" x2="21" y2="18"/>',
        'x' => '<line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/>',
        'chevron-down' => '<polyline points="6 9 12 15 18 9"/>',
        'facebook' => '<path d="M18 2h-3a5 5 0 0 0-5 5v3H7v4h3v8h4v-8h3l1-4h-4V7a1 1 0 0 1 1-1h3z"/>',
        'instagram' => '<rect x="2" y="2" width="20" height="20" rx="5" ry="5"/><path d="M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z"/><line x1="17.5" y1="6.5" x2="17.51" y2="6.5"/>',
        'youtube' => '<path d="M22.54 6.42a2.78 2.78 0 0 0-1.94-2C18.88 4 12 4 12 4s-6.88 0-8.6.46a2.78 2.78 0 0 0-1.94 2A29 29 0 0 0 1 11.75a29 29 0 0 0 .46 5.33A2.78 2.78 0 0 0 3.4 19.1c1.72.46 8.6.46 8.6.46s6.88 0 8.6-.46a2.78 2.78 0 0 0 1.94-2 29 29 0 0 0 .46-5.25 29 29 0 0 0-.46-5.33z"/><polygon points="9.75 15.02 15.5 11.75 9.75 8.48 9.75 15.02"/>',
        'yelp' => '<path d="M20.16 12.73c-.26-.33-.67-.48-1.05-.48-.19 0-.39.04-.56.13l-2.44 1.22a1 1 0 0 0-.54.89v.01a1 1 0 0 0 .54.89l2.44 1.22c.17.09.37.13.56.13.38 0 .79-.15 1.05-.48.36-.46.36-1.1 0-1.56l-.01-.01.01-.01c.36-.46.36-1.1 0-1.56zM12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2z"/>',
    );

    if ( ! isset( $icons[ $name ] ) ) {
        return '';
    }

    $class_attr = $class ? ' class="ace-icon ' . esc_attr( $class ) . '"' : ' class="ace-icon"';

    return '<svg' . $class_attr . ' width="' . intval( $size ) . '" height="' . intval( $size ) . '" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" xmlns="http://www.w3.org/2000/svg">' . $icons[ $name ] . '</svg>';
}

/**
 * Output SVG icon (echo version)
 */
function ace_the_icon( $name, $size = 24, $class = '' ) {
    echo ace_icon( $name, $size, $class ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup is hardcoded
}

/**
 * Get all services data
 */
function ace_get_services() {
    $services = get_option( 'ace_services', array() );
    if ( empty( $services ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $services = isset( $defaults['ace_services'] ) ? $defaults['ace_services'] : array();
    }
    return $services;
}

/**
 * Get single service by slug
 */
function ace_get_service( $slug ) {
    $services = ace_get_services();
    foreach ( $services as $service ) {
        if ( isset( $service['slug'] ) && $service['slug'] === $slug ) {
            return $service;
        }
    }
    return false;
}

/**
 * Get testimonials
 */
function ace_get_testimonials() {
    $testimonials = get_option( 'ace_testimonials', array() );
    if ( empty( $testimonials ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $testimonials = isset( $defaults['ace_testimonials'] ) ? $defaults['ace_testimonials'] : array();
    }
    return $testimonials;
}

/**
 * Get stats data
 */
function ace_get_stats() {
    $stats = get_option( 'ace_stats', array() );
    if ( empty( $stats ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $stats = isset( $defaults['ace_stats'] ) ? $defaults['ace_stats'] : array();
    }
    return $stats;
}

/**
 * Get process steps
 */
function ace_get_process_steps() {
    $steps = get_option( 'ace_process_steps', array() );
    if ( empty( $steps ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $steps = isset( $defaults['ace_process_steps'] ) ? $defaults['ace_process_steps'] : array();
    }
    return $steps;
}

/**
 * Get Why ACE features
 */
function ace_get_why_ace_features() {
    $features = get_option( 'ace_why_ace_features', array() );
    if ( empty( $features ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $features = isset( $defaults['ace_why_ace_features'] ) ? $defaults['ace_why_ace_features'] : array();
    }
    return $features;
}

/**
 * Check if a section is visible
 */
function ace_is_section_visible( $section_key ) {
    $sections = get_option( 'ace_sections', array() );
    return ! isset( $sections[ $section_key ] ) || $sections[ $section_key ];
}

/**
 * Get reading time estimate for a post
 */
function ace_reading_time( $post_id = null ) {
    $post = get_post( $post_id );
    if ( ! $post ) {
        return '1 min read';
    }
    $word_count = str_word_count( wp_strip_all_tags( $post->post_content ) );
    $minutes = max( 1, ceil( $word_count / 250 ) );
    return $minutes . ' min read';
}

/**
 * Truncate text to a certain number of words
 */
function ace_truncate( $text, $words = 20, $more = '...' ) {
    return wp_trim_words( $text, $words, $more );
}

/**
 * Generate breadcrumbs
 */
function ace_breadcrumbs() {
    if ( is_front_page() ) {
        return;
    }

    $items = array();
    $items[] = '<a href="' . esc_url( home_url( '/' ) ) . '">' . esc_html__( 'Home', 'ace-theme' ) . '</a>';

    if ( is_singular( 'ace_project' ) ) {
        $items[] = '<a href="' . esc_url( home_url( '/our-work/' ) ) . '">' . esc_html__( 'Our Work', 'ace-theme' ) . '</a>';
        $items[] = '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_singular( 'post' ) ) {
        $items[] = '<a href="' . esc_url( get_permalink( get_option( 'page_for_posts' ) ) ) . '">' . esc_html__( 'Blog', 'ace-theme' ) . '</a>';
        $items[] = '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_page() ) {
        $ancestors = get_post_ancestors( get_the_ID() );
        foreach ( array_reverse( $ancestors ) as $ancestor ) {
            $items[] = '<a href="' . esc_url( get_permalink( $ancestor ) ) . '">' . esc_html( get_the_title( $ancestor ) ) . '</a>';
        }
        $items[] = '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_archive() ) {
        $items[] = '<span>' . esc_html( get_the_archive_title() ) . '</span>';
    } elseif ( is_search() ) {
        $items[] = '<span>' . esc_html__( 'Search Results', 'ace-theme' ) . '</span>';
    } elseif ( is_404() ) {
        $items[] = '<span>' . esc_html__( 'Page Not Found', 'ace-theme' ) . '</span>';
    }

    echo '<nav class="breadcrumbs" aria-label="' . esc_attr__( 'Breadcrumb', 'ace-theme' ) . '">';
    echo '<div class="container">';
    echo wp_kses_post( implode( ' <span class="separator">/</span> ', $items ) );
    echo '</div>';
    echo '</nav>';
}

/**
 * Get service areas
 */
function ace_get_service_areas() {
    $areas = get_option( 'ace_service_areas', array() );
    if ( empty( $areas ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $areas = isset( $defaults['ace_service_areas'] ) ? $defaults['ace_service_areas'] : array();
    }
    return $areas;
}

/**
 * Get about values
 */
function ace_get_about_values() {
    $values = get_option( 'ace_about_values', array() );
    if ( empty( $values ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $values = isset( $defaults['ace_about_values'] ) ? $defaults['ace_about_values'] : array();
    }
    return $values;
}
