<?php
/**
 * Service Page Template
 *
 * Used for individual service pages. The service data is loaded
 * from the ace_services option based on the page slug.
 *
 * @package ACE_Theme
 */

get_header();

// Determine service slug from the URL or page slug
$service_slug = get_query_var( 'ace_service', '' );
if ( ! $service_slug ) {
    // Fallback: try to get from the page slug
    $service_slug = get_post_field( 'post_name', get_the_ID() );
}

$service = ace_get_service( $service_slug );

// If no service found, try matching by URL segment
if ( ! $service ) {
    $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
    $uri_parts = explode( '/', trim( $request_uri, '/' ) );
    if ( count( $uri_parts ) >= 2 && $uri_parts[0] === 'services' ) {
        $service_slug = $uri_parts[1];
        $service = ace_get_service( $service_slug );
    }
}

if ( ! $service ) {
    // Redirect to services page or show 404
    get_template_part( '404' );
    return;
}

// Extract service data with defaults
$h1 = isset( $service['h1'] ) ? $service['h1'] : $service['title'] . ' in Los Angeles';
$hero_sub = isset( $service['hero_subheadline'] ) ? $service['hero_subheadline'] : '';
$hero_image = isset( $service['hero_image'] ) ? $service['hero_image'] : '';
$introduction = isset( $service['introduction'] ) ? $service['introduction'] : '';
$side_image = isset( $service['side_image'] ) ? $service['side_image'] : '';
$whats_included = isset( $service['whats_included'] ) ? $service['whats_included'] : array();
$benefits = isset( $service['benefits'] ) ? $service['benefits'] : array();
$process_steps = isset( $service['process_steps'] ) ? $service['process_steps'] : array();
$gallery = isset( $service['gallery'] ) ? $service['gallery'] : array();
$faqs = isset( $service['faqs'] ) ? $service['faqs'] : array();
$testimonials_ids = isset( $service['testimonials'] ) ? $service['testimonials'] : array();
$related = isset( $service['related_services'] ) ? $service['related_services'] : array();
$cta_headline = isset( $service['cta_headline'] ) ? $service['cta_headline'] : 'Ready to Start Your ' . $service['title'] . ' Project?';
$cta_subheadline = isset( $service['cta_subheadline'] ) ? $service['cta_subheadline'] : 'Schedule your free in-home consultation today.';
?>

<!-- SECTION 1: SERVICE HERO -->
<section class="service-hero">
    <?php $bg_style = $hero_image ? 'background-image: url(' . esc_url( $hero_image ) . ');' : ''; ?>
    <div class="service-hero-overlay" style="<?php echo esc_attr( $bg_style ); ?>">
        <div class="container service-hero-grid">
            <div class="service-hero-content">
                <h1><?php echo esc_html( $h1 ); ?></h1>
                <?php if ( $hero_sub ) : ?>
                    <p class="service-hero-subtitle"><?php echo esc_html( $hero_sub ); ?></p>
                <?php endif; ?>
                <!-- Trust badges -->
                <div class="trust-bar trust-bar-sm">
                    <div class="trust-item">
                        <?php echo ace_icon( 'star', 16 ); ?>
                        <span><?php esc_html_e( '4.9 Google Rating', 'ace-theme' ); ?></span>
                    </div>
                    <div class="trust-item">
                        <?php echo ace_icon( 'check-circle', 16 ); ?>
                        <span><?php esc_html_e( 'Licensed & Insured', 'ace-theme' ); ?></span>
                    </div>
                    <div class="trust-item">
                        <?php echo ace_icon( 'award', 16 ); ?>
                        <span><?php esc_html_e( '15+ Years Experience', 'ace-theme' ); ?></span>
                    </div>
                </div>
            </div>
            <div class="service-hero-form">
                <?php get_template_part( 'template-parts/lead-form', null, array( 'service' => $service_slug, 'form_id' => 'service-lead-form' ) ); ?>
            </div>
        </div>
    </div>
</section>

<!-- SECTION 2: SERVICE INTRODUCTION -->
<?php if ( $introduction ) : ?>
<section class="section bg-white">
    <div class="container">
        <div class="service-intro-grid">
            <div class="service-intro-content">
                <?php echo wp_kses_post( wpautop( $introduction ) ); ?>

                <?php if ( ! empty( $whats_included ) ) : ?>
                    <h3><?php esc_html_e( "What's Included", 'ace-theme' ); ?></h3>
                    <ul class="included-list">
                        <?php foreach ( $whats_included as $item ) : ?>
                            <li>
                                <?php echo ace_icon( 'check-circle', 18 ); ?>
                                <span><?php echo esc_html( $item ); ?></span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <div class="service-intro-image">
                <?php if ( $side_image ) : ?>
                    <img src="<?php echo esc_url( $side_image ); ?>" alt="<?php echo esc_attr( $service['title'] ); ?>" loading="lazy">
                <?php else : ?>
                    <div class="image-placeholder large">
                        <?php echo ace_icon( 'home', 64 ); ?>
                        <span><?php echo esc_html( $service['title'] ); ?></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 3: BENEFITS -->
<?php if ( ! empty( $benefits ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Why Choose ACE for %s', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <div class="benefits-grid">
            <?php foreach ( $benefits as $benefit ) : ?>
                <div class="benefit-card">
                    <div class="benefit-icon">
                        <?php echo ace_icon( isset( $benefit['icon'] ) ? $benefit['icon'] : 'check-circle', 40 ); ?>
                    </div>
                    <h3><?php echo esc_html( $benefit['title'] ); ?></h3>
                    <p><?php echo esc_html( $benefit['description'] ); ?></p>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 4: PROCESS -->
<?php if ( ! empty( $process_steps ) ) : ?>
<section class="section bg-white">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Our %s Process', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <div class="process-timeline">
            <?php foreach ( $process_steps as $index => $step ) : ?>
                <div class="process-step">
                    <div class="step-number"><?php echo intval( $index + 1 ); ?></div>
                    <div class="step-content">
                        <?php if ( is_array( $step ) && isset( $step['title'] ) ) : ?>
                            <h3><?php echo esc_html( $step['title'] ); ?></h3>
                            <p><?php echo esc_html( $step['description'] ); ?></p>
                        <?php else : ?>
                            <p><?php echo esc_html( $step ); ?></p>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 5: GALLERY -->
<?php if ( ! empty( $gallery ) ) : ?>
<section class="section bg-navy service-gallery-section">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Our %s Projects', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <div class="gallery-grid">
            <?php foreach ( $gallery as $image_url ) : ?>
                <div class="gallery-item">
                    <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $service['title'] ); ?> project" loading="lazy">
                </div>
            <?php endforeach; ?>
        </div>
        <div class="text-center" style="margin-top: 32px;">
            <a href="<?php echo esc_url( home_url( '/our-work/' ) ); ?>" class="link-arrow text-white">
                <?php esc_html_e( 'View More in Our Portfolio', 'ace-theme' ); ?> <?php echo ace_icon( 'arrow-right', 16 ); ?>
            </a>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 6: FAQ -->
<?php if ( ! empty( $faqs ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Frequently Asked Questions About %s', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <?php get_template_part( 'template-parts/faq-accordion', null, array( 'faqs' => $faqs ) ); ?>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 7: TESTIMONIALS -->
<section class="section bg-white">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php esc_html_e( 'What Our Clients Say', 'ace-theme' ); ?></h2>
        </div>
        <div class="testimonials-slider">
            <?php
            $all_testimonials = ace_get_testimonials();
            if ( ! empty( $all_testimonials ) ) :
                foreach ( $all_testimonials as $index => $testimonial ) : ?>
                    <div class="testimonial-card<?php echo 0 === $index ? ' active' : ''; ?>" data-index="<?php echo intval( $index ); ?>">
                        <div class="testimonial-quote-icon">
                            <?php echo ace_icon( 'quote', 32 ); ?>
                        </div>
                        <blockquote class="testimonial-text">
                            <?php echo esc_html( $testimonial['text'] ); ?>
                        </blockquote>
                        <div class="testimonial-rating">
                            <?php
                            $rating = isset( $testimonial['rating'] ) ? intval( $testimonial['rating'] ) : 5;
                            for ( $i = 0; $i < $rating; $i++ ) {
                                echo ace_icon( 'star', 16, 'star-filled' );
                            }
                            ?>
                        </div>
                        <div class="testimonial-author">
                            <?php if ( ! empty( $testimonial['avatar'] ) ) : ?>
                                <img src="<?php echo esc_url( $testimonial['avatar'] ); ?>" alt="<?php echo esc_attr( $testimonial['name'] ); ?>" class="testimonial-avatar" loading="lazy">
                            <?php endif; ?>
                            <div class="testimonial-info">
                                <strong><?php echo esc_html( $testimonial['name'] ); ?></strong>
                                <?php if ( ! empty( $testimonial['location'] ) ) : ?>
                                    <span><?php echo esc_html( $testimonial['location'] ); ?></span>
                                <?php endif; ?>
                                <?php if ( ! empty( $testimonial['project_type'] ) ) : ?>
                                    <span class="testimonial-project"><?php echo esc_html( $testimonial['project_type'] ); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach;
            endif; ?>
        </div>
        <!-- Slider Navigation -->
        <div class="testimonials-nav">
            <button class="testimonial-prev" aria-label="<?php esc_attr_e( 'Previous testimonial', 'ace-theme' ); ?>">
                <?php echo ace_icon( 'arrow-right', 20 ); ?>
            </button>
            <div class="testimonials-dots">
                <?php
                if ( ! empty( $all_testimonials ) ) :
                    $total = count( $all_testimonials );
                    for ( $i = 0; $i < $total; $i++ ) : ?>
                        <button class="dot<?php echo 0 === $i ? ' active' : ''; ?>" data-slide="<?php echo intval( $i ); ?>" aria-label="<?php echo esc_attr( sprintf( __( 'Go to testimonial %d', 'ace-theme' ), $i + 1 ) ); ?>"></button>
                    <?php endfor;
                endif; ?>
            </div>
            <button class="testimonial-next" aria-label="<?php esc_attr_e( 'Next testimonial', 'ace-theme' ); ?>">
                <?php echo ace_icon( 'arrow-right', 20 ); ?>
            </button>
        </div>
    </div>
</section>

<!-- SECTION 8: RELATED SERVICES -->
<?php if ( ! empty( $related ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php esc_html_e( 'You Might Also Be Interested In', 'ace-theme' ); ?></h2>
        </div>
        <div class="related-services-grid">
            <?php
            foreach ( $related as $related_slug ) :
                $related_service = ace_get_service( $related_slug );
                if ( $related_service ) :
                    get_template_part( 'template-parts/service-card', null, array( 'service' => $related_service ) );
                endif;
            endforeach;
            ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 9: FINAL CTA -->
<section class="cta-banner bg-navy">
    <div class="container">
        <div class="cta-banner-inner">
            <div class="cta-banner-content">
                <h2><?php echo esc_html( $cta_headline ); ?></h2>
                <p><?php echo esc_html( $cta_subheadline ); ?></p>
            </div>
            <div class="cta-banner-actions">
                <a href="<?php echo esc_url( home_url( '/contact/' ) ); ?>" class="btn btn-primary btn-lg">
                    <?php esc_html_e( 'Get Your Free Consultation', 'ace-theme' ); ?>
                </a>
                <a href="<?php echo esc_url( ace_get_phone_link() ); ?>" class="btn btn-ghost btn-lg">
                    <?php echo ace_icon( 'phone', 18 ); ?>
                    <?php echo esc_html( ace_get_phone() ); ?>
                </a>
            </div>
        </div>
    </div>
</section>

<?php get_footer(); ?>
