<?php
if (!defined('ABSPATH')) exit;

class ACE_BLP_Leads {

    public function __construct() {
        add_action('wp_ajax_ace_backyard_submit_lead', array($this, 'handle_submission'));
        add_action('wp_ajax_nopriv_ace_backyard_submit_lead', array($this, 'handle_submission'));
        add_action('wp_ajax_ace_blp_export_csv', array($this, 'export_csv'));
        add_action('wp_ajax_ace_blp_delete_lead', array($this, 'delete_lead'));
        add_action('wp_ajax_ace_blp_toggle_contacted', array($this, 'toggle_contacted'));
        add_action('wp_ajax_ace_blp_get_leads', array($this, 'get_leads_ajax'));
    }

    public static function create_table() {
        global $wpdb;
        $table = $wpdb->prefix . 'ace_backyard_leads';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL,
            phone VARCHAR(30) NOT NULL,
            email VARCHAR(100) DEFAULT '',
            address VARCHAR(255) DEFAULT '',
            message TEXT DEFAULT '',
            zip_code VARCHAR(20) DEFAULT '',
            source VARCHAR(30) NOT NULL,
            heard_about VARCHAR(100) DEFAULT '',
            calculator_data TEXT DEFAULT '',
            contacted TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_created (created_at),
            INDEX idx_source (source)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    public function handle_submission() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['_nonce'] ?? '', 'ace_blp_form_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed.'));
        }

        // WordPress adds magic quotes to $_POST — strip them
        $_POST = wp_unslash($_POST);

        $name = sanitize_text_field($_POST['name'] ?? '');
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        $email = sanitize_email($_POST['email'] ?? '');
        $address = sanitize_text_field($_POST['address'] ?? '');
        $zip_code = sanitize_text_field($_POST['zip_code'] ?? '');
        $message = sanitize_textarea_field($_POST['message'] ?? '');
        $source = sanitize_text_field($_POST['source'] ?? 'unknown');
        $heard_about = sanitize_text_field($_POST['heard_about'] ?? '');
        $calculator_data = sanitize_text_field($_POST['calculator_data'] ?? '');

        if (empty($name) || empty($phone)) {
            wp_send_json_error(array('message' => 'Name and phone are required.'));
        }

        global $wpdb;
        $table = $wpdb->prefix . 'ace_backyard_leads';

        $result = $wpdb->insert($table, array(
            'name' => $name,
            'phone' => $phone,
            'email' => $email,
            'address' => $address,
            'zip_code' => $zip_code,
            'message' => $message,
            'source' => $source,
            'heard_about' => $heard_about,
            'calculator_data' => $calculator_data,
            'contacted' => 0,
            'created_at' => current_time('mysql'),
        ), array('%s','%s','%s','%s','%s','%s','%s','%s','%s','%d','%s'));

        if ($result === false) {
            wp_send_json_error(array('message' => 'Failed to save. Please try again.'));
        }

        // Send email notification
        $this->send_notification($name, $phone, $email, $address, $zip_code, $message, $source, $heard_about, $calculator_data);

        wp_send_json_success(array('message' => 'Thank you! We will be in touch within 2 hours.'));
    }

    private function send_notification($name, $phone, $email, $address, $zip_code, $message, $source, $heard_about, $calculator_data) {
        $settings = get_option('ace_backyard_lp_settings', array());
        $to = !empty($settings['email']) ? $settings['email'] : get_option('admin_email');
        $subject = 'New Backyard Lead: ' . $name . ' - ' . ucfirst($source);

        $body = '<html><body>';
        $body .= '<h2 style="color:#C6A355;">New Backyard Remodeling Lead</h2>';
        $body .= '<table style="border-collapse:collapse;width:100%;max-width:600px;">';
        $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Name</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($name) . '</td></tr>';
        $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Phone</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($phone) . '</td></tr>';
        if ($email) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Email</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($email) . '</td></tr>';
        if ($address) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Address</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($address) . '</td></tr>';
        if ($zip_code) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Zip Code</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($zip_code) . '</td></tr>';
        if ($message) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Message</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($message) . '</td></tr>';
        $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Source</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html(ucfirst($source)) . '</td></tr>';
        if ($heard_about) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;">Heard About Us</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($heard_about) . '</td></tr>';

        if ($calculator_data) {
            $calc = json_decode($calculator_data, true);
            if ($calc) {
                $body .= '<tr><td style="padding:8px;border:1px solid #ddd;font-weight:bold;" colspan="2"><strong>Calculator Selections:</strong></td></tr>';
                if (!empty($calc['size'])) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Yard Size</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($calc['size']) . '</td></tr>';
                if (!empty($calc['features'])) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Features</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html(implode(', ', $calc['features'])) . '</td></tr>';
                if (!empty($calc['finish'])) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Finish Level</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($calc['finish']) . '</td></tr>';
                if (!empty($calc['price'])) $body .= '<tr><td style="padding:8px;border:1px solid #ddd;">Estimated Price</td><td style="padding:8px;border:1px solid #ddd;">' . esc_html($calc['price']) . '</td></tr>';
            }
        }

        $body .= '</table></body></html>';

        $headers = array('Content-Type: text/html; charset=UTF-8');
        if ($email) {
            $headers[] = 'Reply-To: ' . $email;
        }

        wp_mail($to, $subject, $body, $headers);
    }

    public function get_leads_ajax() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        global $wpdb;
        $table = $wpdb->prefix . 'ace_backyard_leads';

        $where = '1=1';
        $params = array();

        if (!empty($_GET['source'])) {
            $where .= ' AND source = %s';
            $params[] = sanitize_text_field($_GET['source']);
        }
        if (!empty($_GET['date_from'])) {
            $where .= ' AND created_at >= %s';
            $params[] = sanitize_text_field($_GET['date_from']) . ' 00:00:00';
        }
        if (!empty($_GET['date_to'])) {
            $where .= ' AND created_at <= %s';
            $params[] = sanitize_text_field($_GET['date_to']) . ' 23:59:59';
        }

        if (!empty($params)) {
            $leads = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table WHERE $where ORDER BY created_at DESC", ...$params));
        } else {
            $leads = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC");
        }

        wp_send_json_success($leads);
    }

    public function export_csv() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        global $wpdb;
        $table = $wpdb->prefix . 'ace_backyard_leads';
        $leads = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC", ARRAY_A);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=backyard-leads-' . date('Y-m-d') . '.csv');

        $output = fopen('php://output', 'w');
        fputcsv($output, array('ID', 'Date', 'Name', 'Phone', 'Email', 'Address', 'Zip Code', 'Message', 'Source', 'Heard About', 'Calculator Data', 'Contacted'));

        foreach ($leads as $lead) {
            fputcsv($output, array(
                $lead['id'],
                $lead['created_at'],
                $lead['name'],
                $lead['phone'],
                $lead['email'],
                $lead['address'],
                isset($lead['zip_code']) ? $lead['zip_code'] : '',
                $lead['message'],
                $lead['source'],
                $lead['heard_about'],
                $lead['calculator_data'],
                $lead['contacted'] ? 'Yes' : 'No',
            ));
        }

        fclose($output);
        exit;
    }

    public function delete_lead() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        if (!wp_verify_nonce($_POST['_nonce'] ?? '', 'ace_blp_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        $id = intval($_POST['lead_id'] ?? 0);
        if (!$id) wp_send_json_error('Invalid ID');

        global $wpdb;
        $wpdb->delete($wpdb->prefix . 'ace_backyard_leads', array('id' => $id), array('%d'));

        wp_send_json_success();
    }

    public function toggle_contacted() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        if (!wp_verify_nonce($_POST['_nonce'] ?? '', 'ace_blp_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        $id = intval($_POST['lead_id'] ?? 0);
        $contacted = intval($_POST['contacted'] ?? 0);
        if (!$id) wp_send_json_error('Invalid ID');

        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'ace_backyard_leads',
            array('contacted' => $contacted),
            array('id' => $id),
            array('%d'),
            array('%d')
        );

        wp_send_json_success();
    }
}
