<?php
/**
 * Import / Export handler.
 *
 * Provides admin-post actions for exporting plugin settings as JSON,
 * importing settings from a JSON file, and exporting leads as CSV.
 *
 * @package ACE_Theme_Manager
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class ACE_Import_Export
 *
 * Registers admin-post handlers for settings import/export and leads CSV export.
 *
 * @since 1.0.0
 */
class ACE_Import_Export {

	/**
	 * Constructor.
	 *
	 * Hooks the export and import handlers to their respective admin_post actions.
	 */
	public function __construct() {
		add_action( 'admin_post_ace_export_settings', array( $this, 'export_settings' ) );
		add_action( 'admin_post_ace_import_settings', array( $this, 'import_settings' ) );
		add_action( 'admin_post_ace_export_leads',    array( $this, 'export_leads_csv' ) );
	}

	/**
	 * Export all ACE plugin settings as a JSON file download.
	 *
	 * Queries wp_options for every row whose option_name begins with 'ace_',
	 * builds an associative array, and streams the result as a downloadable
	 * JSON file.
	 *
	 * @return void Terminates with exit.
	 */
	public function export_settings() {
		// Capability check.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die(
				esc_html__( 'You do not have permission to export settings.', 'ace-theme-manager' ),
				esc_html__( 'Forbidden', 'ace-theme-manager' ),
				array( 'response' => 403 )
			);
		}

		// Nonce verification.
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'ace_export_settings' ) ) {
			wp_die(
				esc_html__( 'Security check failed.', 'ace-theme-manager' ),
				esc_html__( 'Forbidden', 'ace-theme-manager' ),
				array( 'response' => 403 )
			);
		}

		global $wpdb;

		// Retrieve all options with the ace_ prefix.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$rows = $wpdb->get_results(
			"SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE 'ace\_%'",
			ARRAY_A
		);

		$settings = array();

		if ( $rows ) {
			foreach ( $rows as $row ) {
				$value = maybe_unserialize( $row['option_value'] );
				$settings[ $row['option_name'] ] = $value;
			}
		}

		$filename = 'ace-settings-' . gmdate( 'Y-m-d' ) . '.json';

		// Prevent caching.
		nocache_headers();

		header( 'Content-Type: application/json; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=' . $filename );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		echo wp_json_encode( $settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE );
		exit;
	}

	/**
	 * Import ACE plugin settings from an uploaded JSON file.
	 *
	 * Validates the upload, decodes the JSON, and updates each option
	 * in the database. Redirects back to the referring admin page with
	 * a success or error notice.
	 *
	 * @return void Redirects and terminates.
	 */
	public function import_settings() {
		// Capability check.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die(
				esc_html__( 'You do not have permission to import settings.', 'ace-theme-manager' ),
				esc_html__( 'Forbidden', 'ace-theme-manager' ),
				array( 'response' => 403 )
			);
		}

		// Nonce verification.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'ace_import_settings' ) ) {
			wp_die(
				esc_html__( 'Security check failed.', 'ace-theme-manager' ),
				esc_html__( 'Forbidden', 'ace-theme-manager' ),
				array( 'response' => 403 )
			);
		}

		// Determine the redirect URL (back to the referring page).
		$redirect_url = isset( $_POST['_wp_http_referer'] )
			? esc_url_raw( wp_unslash( $_POST['_wp_http_referer'] ) )
			: admin_url( 'admin.php?page=ace-settings' );

		// Validate uploaded file exists.
		if ( empty( $_FILES['ace_import_file'] ) || empty( $_FILES['ace_import_file']['tmp_name'] ) ) {
			wp_safe_redirect( add_query_arg( 'ace_notice', 'import_no_file', $redirect_url ) );
			exit;
		}

		$file = $_FILES['ace_import_file']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput

		// Check for upload errors.
		if ( $file['error'] !== UPLOAD_ERR_OK ) {
			wp_safe_redirect( add_query_arg( 'ace_notice', 'import_upload_error', $redirect_url ) );
			exit;
		}

		// Validate file extension.
		$file_info = wp_check_filetype( $file['name'], array( 'json' => 'application/json' ) );

		if ( 'json' !== $file_info['ext'] ) {
			wp_safe_redirect( add_query_arg( 'ace_notice', 'import_invalid_type', $redirect_url ) );
			exit;
		}

		// Read and decode JSON.
		$json_content = file_get_contents( $file['tmp_name'] ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents

		if ( false === $json_content || empty( $json_content ) ) {
			wp_safe_redirect( add_query_arg( 'ace_notice', 'import_read_error', $redirect_url ) );
			exit;
		}

		$settings = json_decode( $json_content, true );

		if ( null === $settings || ! is_array( $settings ) ) {
			wp_safe_redirect( add_query_arg( 'ace_notice', 'import_invalid_json', $redirect_url ) );
			exit;
		}

		// Import each setting.
		$imported_count = 0;

		foreach ( $settings as $option_name => $option_value ) {
			// Sanitize the key - only allow ace_ prefixed options.
			$option_name = sanitize_key( $option_name );

			if ( 0 !== strpos( $option_name, 'ace_' ) ) {
				continue;
			}

			update_option( $option_name, $option_value );
			$imported_count++;
		}

		// Redirect with success.
		wp_safe_redirect(
			add_query_arg(
				array(
					'ace_notice'   => 'import_success',
					'ace_imported' => $imported_count,
				),
				$redirect_url
			)
		);
		exit;
	}

	/**
	 * Export all leads from the ace_leads table as a CSV download.
	 *
	 * Queries every row from the leads table and streams the result as a
	 * downloadable CSV file with appropriate headers.
	 *
	 * @return void Terminates with exit.
	 */
	public function export_leads_csv() {
		// Capability check.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die(
				esc_html__( 'You do not have permission to export leads.', 'ace-theme-manager' ),
				esc_html__( 'Forbidden', 'ace-theme-manager' ),
				array( 'response' => 403 )
			);
		}

		// Nonce verification.
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'ace_export_leads' ) ) {
			wp_die(
				esc_html__( 'Security check failed.', 'ace-theme-manager' ),
				esc_html__( 'Forbidden', 'ace-theme-manager' ),
				array( 'response' => 403 )
			);
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$leads = $wpdb->get_results( "SELECT * FROM {$table_name} ORDER BY created_at DESC", ARRAY_A );

		$filename = 'ace-leads-' . gmdate( 'Y-m-d' ) . '.csv';

		// Prevent caching.
		nocache_headers();

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=' . $filename );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		$output = fopen( 'php://output', 'w' );

		if ( false === $output ) {
			wp_die(
				esc_html__( 'Unable to open output stream for CSV export.', 'ace-theme-manager' ),
				esc_html__( 'Server Error', 'ace-theme-manager' ),
				array( 'response' => 500 )
			);
		}

		// CSV column headers.
		fputcsv( $output, array(
			'ID',
			'Name',
			'Phone',
			'Email',
			'Service',
			'Message',
			'Contact Method',
			'Source Page',
			'Source URL',
			'IP',
			'Date',
			'Read',
			'Notes',
		) );

		// Data rows.
		if ( $leads ) {
			foreach ( $leads as $lead ) {
				fputcsv( $output, array(
					$lead['id'],
					$lead['full_name'],
					$lead['phone'],
					$lead['email'],
					$lead['service'],
					$lead['message'],
					$lead['preferred_contact'],
					$lead['source_page'],
					$lead['source_url'],
					$lead['ip_address'],
					$lead['created_at'],
					$lead['is_read'] ? 'Yes' : 'No',
					$lead['notes'],
				) );
			}
		}

		fclose( $output );
		exit;
	}
}
