<?php
if (!defined('ABSPATH')) exit;

class ACE_WebP_Admin {

    private $converter;

    public function __construct($converter) {
        $this->converter = $converter;
        add_action('admin_menu', [$this, 'add_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('wp_ajax_ace_webp_bulk', [$this, 'ajax_bulk_convert']);
        add_action('wp_ajax_ace_webp_save', [$this, 'ajax_save_settings']);
        add_action('wp_ajax_ace_webp_stats', [$this, 'ajax_get_stats']);
    }

    public function add_menu() {
        add_options_page(
            'WebP Optimizer',
            'WebP Optimizer',
            'manage_options',
            'ace-webp',
            [$this, 'render_page']
        );
    }

    public function enqueue_assets($hook) {
        if ($hook !== 'settings_page_ace-webp') return;
        wp_enqueue_style('ace-webp-admin', ACE_WEBP_URL . 'assets/admin.css', [], ACE_WEBP_VERSION);
        wp_enqueue_script('ace-webp-admin', ACE_WEBP_URL . 'assets/admin.js', ['jquery'], ACE_WEBP_VERSION, true);
        wp_localize_script('ace-webp-admin', 'aceWebP', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('ace_webp_nonce'),
        ]);
    }

    public function render_page() {
        $quality = (int) get_option('ace_webp_quality', 82);
        $auto = get_option('ace_webp_auto', 'yes');
        $method = $this->converter->get_method();
        $stats = $this->converter->get_stats();
        ?>
        <div class="wrap ace-webp-wrap">
            <h1>ACE WebP Optimizer</h1>
            <div class="ace-webp-grid">
                <div class="ace-webp-card">
                    <h2>Settings</h2>
                    <div class="ace-webp-field">
                        <label>Quality: <strong id="qualityVal"><?php echo $quality; ?>%</strong></label>
                        <input type="range" id="aceQuality" min="50" max="100" value="<?php echo $quality; ?>">
                        <p class="description">82% recommended. Lower = smaller file, slightly less quality.</p>
                    </div>
                    <div class="ace-webp-field">
                        <label>
                            <input type="checkbox" id="aceAuto" <?php checked($auto, 'yes'); ?>>
                            Auto-convert new uploads
                        </label>
                    </div>
                    <div class="ace-webp-field">
                        <p>Conversion method: <code><?php echo strtoupper($method); ?></code></p>
                    </div>
                    <button id="aceSaveSettings" class="button button-primary">Save Settings</button>
                    <span id="aceSaveMsg" class="ace-webp-msg"></span>
                </div>
                <div class="ace-webp-card">
                    <h2>Bulk Convert</h2>
                    <p>Convert all existing JPG/PNG images to WebP.</p>
                    <button id="aceBulkStart" class="button button-primary">Start Bulk Conversion</button>
                    <button id="aceBulkStop" class="button" style="display:none;">Stop</button>
                    <div id="aceBulkProgress" style="display:none;">
                        <div class="ace-webp-progress-bar">
                            <div class="ace-webp-progress-fill" id="aceProgressFill"></div>
                        </div>
                        <p id="aceBulkStatus">Processing...</p>
                    </div>
                </div>
                <div class="ace-webp-card">
                    <h2>Statistics</h2>
                    <div class="ace-webp-stats">
                        <div class="ace-webp-stat">
                            <span class="ace-webp-stat-num" id="statTotal"><?php echo $stats['total']; ?></span>
                            <span class="ace-webp-stat-label">Total Images</span>
                        </div>
                        <div class="ace-webp-stat">
                            <span class="ace-webp-stat-num" id="statConverted"><?php echo $stats['converted']; ?></span>
                            <span class="ace-webp-stat-label">Converted</span>
                        </div>
                        <div class="ace-webp-stat">
                            <span class="ace-webp-stat-num" id="statSaved"><?php echo $stats['saved_mb']; ?> MB</span>
                            <span class="ace-webp-stat-label">Space Saved</span>
                        </div>
                    </div>
                    <button id="aceRefreshStats" class="button">Refresh Stats</button>
                </div>
            </div>
        </div>
        <?php
    }

    public function ajax_save_settings() {
        check_ajax_referer('ace_webp_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error();

        $quality = max(50, min(100, (int) ($_POST['quality'] ?? 82)));
        $auto = isset($_POST['auto']) && $_POST['auto'] === 'yes' ? 'yes' : 'no';

        update_option('ace_webp_quality', $quality);
        update_option('ace_webp_auto', $auto);

        wp_send_json_success('Saved');
    }

    public function ajax_bulk_convert() {
        check_ajax_referer('ace_webp_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error();

        $files = $this->converter->get_unconverted(20);
        $converted = 0;

        foreach ($files as $file) {
            $result = $this->converter->convert_file($file);
            if ($result !== false) $converted++;
        }

        $remaining = count($this->converter->get_unconverted(1));

        wp_send_json_success([
            'converted' => $converted,
            'processed' => count($files),
            'remaining' => $remaining > 0,
        ]);
    }

    public function ajax_get_stats() {
        check_ajax_referer('ace_webp_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error();
        wp_send_json_success($this->converter->get_stats());
    }
}
