<?php
if (!defined('ABSPATH')) exit;

class ACE_WebP_Converter {

    private $method = 'none';

    public function __construct() {
        if (function_exists('imagewebp') && function_exists('imagecreatefromjpeg')) {
            $this->method = 'gd';
        } elseif (class_exists('Imagick') && !empty(Imagick::queryFormats('WEBP'))) {
            $this->method = 'imagick';
        }
    }

    public function get_method() {
        return $this->method;
    }

    public function convert_on_upload($metadata, $attachment_id) {
        $upload_dir = wp_upload_dir();
        $base_dir = trailingslashit($upload_dir['basedir']);

        if (!empty($metadata['file'])) {
            $this->convert_file($base_dir . $metadata['file']);
        }

        if (!empty($metadata['sizes'])) {
            $sub_dir = trailingslashit(dirname($metadata['file']));
            foreach ($metadata['sizes'] as $size) {
                if (!empty($size['file'])) {
                    $this->convert_file($base_dir . $sub_dir . $size['file']);
                }
            }
        }

        return $metadata;
    }

    public function convert_file($file_path) {
        if (!file_exists($file_path)) return false;

        $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        if (!in_array($ext, ['jpg', 'jpeg', 'png'], true)) return false;
        if (filesize($file_path) < 10240) return false;

        $webp_path = $file_path . '.webp';
        if (file_exists($webp_path)) return $webp_path;

        $quality = (int) get_option('ace_webp_quality', 82);
        $result = false;

        if ($this->method === 'gd') {
            $result = $this->convert_gd($file_path, $webp_path, $ext, $quality);
        } elseif ($this->method === 'imagick') {
            $result = $this->convert_imagick($file_path, $webp_path, $quality);
        }

        if (!$result) return false;

        if (filesize($webp_path) >= filesize($file_path)) {
            @unlink($webp_path);
            return false;
        }

        return $webp_path;
    }

    private function convert_gd($src, $dest, $ext, $quality) {
        $image = false;
        switch ($ext) {
            case 'jpg':
            case 'jpeg':
                $image = @imagecreatefromjpeg($src);
                break;
            case 'png':
                $image = @imagecreatefrompng($src);
                break;
        }
        if (!$image) return false;

        if ($ext === 'png') {
            imagepalettetotruecolor($image);
            imagealphablending($image, true);
            imagesavealpha($image, true);
        }

        $result = @imagewebp($image, $dest, $quality);
        imagedestroy($image);
        return $result;
    }

    private function convert_imagick($src, $dest, $quality) {
        try {
            $img = new Imagick($src);
            $img->setImageFormat('webp');
            $img->setImageCompressionQuality($quality);
            $img->setOption('webp:lossless', 'false');
            $img->setOption('webp:method', '4');
            $result = $img->writeImage($dest);
            $img->clear();
            $img->destroy();
            return $result;
        } catch (Exception $e) {
            return false;
        }
    }

    public function get_stats() {
        $upload_dir = wp_upload_dir();
        $base = $upload_dir['basedir'];
        $total_images = 0;
        $converted = 0;
        $saved_bytes = 0;

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($base, FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            $ext = strtolower($file->getExtension());
            if (in_array($ext, ['jpg', 'jpeg', 'png'], true)) {
                $total_images++;
                $webp = $file->getPathname() . '.webp';
                if (file_exists($webp)) {
                    $converted++;
                    $saved_bytes += $file->getSize() - filesize($webp);
                }
            }
        }

        return [
            'total'     => $total_images,
            'converted' => $converted,
            'saved_mb'  => round($saved_bytes / 1048576, 1),
        ];
    }

    public function get_unconverted($limit = 20) {
        $upload_dir = wp_upload_dir();
        $base = $upload_dir['basedir'];
        $files = [];

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($base, FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            $ext = strtolower($file->getExtension());
            if (!in_array($ext, ['jpg', 'jpeg', 'png'], true)) continue;
            if ($file->getSize() < 10240) continue;
            if (file_exists($file->getPathname() . '.webp')) continue;

            $files[] = $file->getPathname();
            if (count($files) >= $limit) break;
        }

        return $files;
    }
}
