<?php
/**
 * Service Page Template
 *
 * Used for individual service pages. The service data is loaded
 * from the ace_services option based on the page slug.
 *
 * @package ACE_Theme
 */

get_header();

// Determine service slug from the URL or page slug
$service_slug = get_query_var( 'ace_service', '' );
if ( ! $service_slug ) {
    // Fallback: try to get from the page slug
    $service_slug = get_post_field( 'post_name', get_the_ID() );
}

$service = ace_get_service( $service_slug );

// If no service found, try matching by URL segment
if ( ! $service ) {
    $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
    $uri_parts = explode( '/', trim( $request_uri, '/' ) );
    if ( count( $uri_parts ) >= 2 && $uri_parts[0] === 'services' ) {
        $service_slug = $uri_parts[1];
        $service = ace_get_service( $service_slug );
    }
}

if ( ! $service ) {
    // Redirect to services page or show 404
    get_template_part( '404' );
    return;
}

// Extract service data with defaults
$h1 = isset( $service['h1'] ) ? $service['h1'] : $service['title'] . ' in Los Angeles';
$hero_sub = isset( $service['hero_subheadline'] ) ? $service['hero_subheadline'] : '';
$hero_image = isset( $service['hero_image'] ) ? $service['hero_image'] : '';
$introduction = isset( $service['introduction'] ) ? $service['introduction'] : '';
$side_image = isset( $service['side_image'] ) ? $service['side_image'] : '';
$whats_included = isset( $service['whats_included'] ) ? $service['whats_included'] : array();
$benefits = isset( $service['benefits'] ) ? $service['benefits'] : array();
$process_steps = isset( $service['process_steps'] ) ? $service['process_steps'] : array();
$gallery = isset( $service['gallery'] ) ? $service['gallery'] : array();
$faqs = isset( $service['faqs'] ) ? $service['faqs'] : array();
$testimonials_ids = isset( $service['testimonials'] ) ? $service['testimonials'] : array();
$related = isset( $service['related_services'] ) ? $service['related_services'] : array();
$cta_headline = isset( $service['cta_headline'] ) ? $service['cta_headline'] : 'Ready to Start Your ' . $service['title'] . ' Project?';
$cta_subheadline = isset( $service['cta_subheadline'] ) ? $service['cta_subheadline'] : 'Schedule your free in-home consultation today.';
?>

<!-- SECTION 1: SERVICE HERO -->
<section class="service-hero">
    <?php $bg_style = $hero_image ? 'background-image: url(' . esc_url( $hero_image ) . ');' : ''; ?>
    <div class="service-hero-overlay" style="<?php echo esc_attr( $bg_style ); ?>">
        <div class="container service-hero-grid">
            <div class="service-hero-content">
                <h1><?php echo esc_html( $h1 ); ?></h1>
                <?php if ( $hero_sub ) : ?>
                    <p class="service-hero-subtitle"><?php echo esc_html( $hero_sub ); ?></p>
                <?php endif; ?>
                <!-- Trust badges -->
                <div class="trust-bar trust-bar-sm">
                    <div class="trust-item">
                        <?php echo ace_get_icon( 'trust', 'rating', 'inline' ); ?>
                        <span><?php esc_html_e( '4.9 Google Rating', 'ace-theme' ); ?></span>
                    </div>
                    <div class="trust-item">
                        <?php echo ace_get_icon( 'trust', 'licensed', 'inline' ); ?>
                        <span><?php esc_html_e( 'Licensed & Insured', 'ace-theme' ); ?></span>
                    </div>
                    <div class="trust-item">
                        <?php echo ace_get_icon( 'trust', 'experience', 'inline' ); ?>
                        <span><?php esc_html_e( '15+ Years Experience', 'ace-theme' ); ?></span>
                    </div>
                </div>

                <?php if ( ! empty( $whats_included ) ) : ?>
                <!-- Key Benefits - Glass Cards -->
                <?php $benefit_icons = ace_get_service_benefit_icons( $service_slug ); ?>
                <div class="hero-benefits">
                    <?php foreach ( array_slice( $whats_included, 0, 4 ) as $i => $item ) : ?>
                        <div class="hero-benefit-item">
                            <div class="hero-benefit-icon">
                                <i class="fas <?php echo esc_attr( isset( $benefit_icons[ $i ] ) ? $benefit_icons[ $i ] : 'fa-check-circle' ); ?>"></i>
                            </div>
                            <span class="hero-benefit-text"><?php echo esc_html( $item ); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>

                <!-- Quick Stats - Mini Cards -->
                <?php $hero_stats = ace_get_service_hero_stats( $service_slug ); ?>
                <div class="hero-mini-stats">
                    <?php foreach ( $hero_stats as $stat ) : ?>
                        <div class="hero-mini-stat">
                            <strong><?php echo esc_html( $stat['number'] ); ?></strong>
                            <span><?php echo esc_html( $stat['label'] ); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Direct Call CTA - Button Card -->
                <div class="hero-call-cta">
                    <p><?php esc_html_e( 'Prefer to talk? Call us directly:', 'ace-theme' ); ?></p>
                    <a href="<?php echo esc_url( ace_get_phone_link() ); ?>" class="hero-phone-link">
                        <i class="fas fa-phone-alt"></i>
                        <?php echo esc_html( ace_get_phone() ); ?>
                    </a>
                </div>
            </div>
            <div class="service-hero-form">
                <?php get_template_part( 'template-parts/lead-form', null, array( 'service' => $service_slug, 'form_id' => 'service-lead-form' ) ); ?>
            </div>
        </div>
    </div>
</section>

<!-- SECTION 2: SERVICE INTRODUCTION -->
<?php if ( $introduction ) : ?>
<section class="section bg-white">
    <div class="container">
        <div class="service-intro-grid">
            <div class="service-intro-content">
                <?php echo wp_kses_post( wpautop( $introduction ) ); ?>

            </div>
            <div class="service-intro-image">
                <?php if ( $side_image ) : ?>
                    <img src="<?php echo esc_url( $side_image ); ?>" alt="<?php echo esc_attr( $service['title'] ); ?>" loading="lazy">
                <?php else : ?>
                    <div class="image-placeholder large">
                        <i class="fas <?php echo esc_attr( ace_get_service_icon_class( $service_slug ) ); ?>" style="font-size: 3rem; color: #C6A355;"></i>
                        <span><?php echo esc_html( $service['title'] ); ?></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 2B: WHAT'S INCLUDED (Feature Cards) -->
<?php if ( ! empty( $whats_included ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php esc_html_e( "What's Included", 'ace-theme' ); ?></h2>
        </div>
        <div class="included-features-grid">
            <?php
            $fc_icons = ace_get_feature_card_icons( $service_slug );
            foreach ( $whats_included as $wi_idx => $item ) :
            ?>
                <div class="included-feature-card">
                    <div class="included-feature-icon">
                        <div class="ace-icon-container">
                            <i class="fas <?php echo esc_attr( isset( $fc_icons[ $wi_idx ] ) ? $fc_icons[ $wi_idx ] : 'fa-check' ); ?>"></i>
                        </div>
                    </div>
                    <p><?php echo esc_html( $item ); ?></p>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 3: BENEFITS -->
<?php if ( ! empty( $benefits ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Why Choose ACE for %s', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <div class="benefits-grid">
            <?php
            $benefit_fa_map = array(
                'check-circle' => 'fa-circle-check',
                'star'         => 'fa-star',
                'shield'       => 'fa-shield-halved',
                'award'        => 'fa-award',
                'home'         => 'fa-house',
                'hammer'       => 'fa-hammer',
                'ruler'        => 'fa-ruler',
                'users'        => 'fa-users',
                'dollar-sign'  => 'fa-dollar-sign',
                'calendar'     => 'fa-calendar',
            );
            foreach ( $benefits as $benefit ) :
                $b_icon = isset( $benefit['icon'] ) ? $benefit['icon'] : 'check-circle';
                $b_fa   = isset( $benefit_fa_map[ $b_icon ] ) ? $benefit_fa_map[ $b_icon ] : 'fa-' . $b_icon;
            ?>
                <div class="benefit-card">
                    <div class="benefit-icon">
                        <div class="ace-icon-container-lg">
                            <i class="fas <?php echo esc_attr( $b_fa ); ?>"></i>
                        </div>
                    </div>
                    <h3><?php echo esc_html( $benefit['title'] ); ?></h3>
                    <p><?php echo esc_html( $benefit['description'] ); ?></p>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 4: PROCESS -->
<?php if ( ! empty( $process_steps ) ) : ?>
<section class="section bg-white">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Our %s Process', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <div class="process-timeline">
            <?php foreach ( $process_steps as $index => $step ) : ?>
                <div class="process-step">
                    <div class="step-number"><?php echo intval( $index + 1 ); ?></div>
                    <div class="step-content">
                        <?php if ( is_array( $step ) && isset( $step['title'] ) ) : ?>
                            <h3><?php echo esc_html( $step['title'] ); ?></h3>
                            <p><?php echo esc_html( $step['description'] ); ?></p>
                        <?php else : ?>
                            <p><?php echo esc_html( $step ); ?></p>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 5: GALLERY -->
<?php if ( ! empty( $gallery ) ) : ?>
<section class="section bg-navy service-gallery-section">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Our %s Projects', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <div class="gallery-grid">
            <?php foreach ( $gallery as $image_url ) : ?>
                <div class="gallery-item">
                    <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $service['title'] ); ?> project" loading="lazy">
                </div>
            <?php endforeach; ?>
        </div>
        <div class="text-center" style="margin-top: 32px;">
            <a href="<?php echo esc_url( home_url( '/our-work/' ) ); ?>" class="link-arrow text-white">
                <?php esc_html_e( 'View More in Our Portfolio', 'ace-theme' ); ?> <?php echo ace_icon( 'arrow-right', 16 ); ?>
            </a>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 6: FAQ -->
<?php if ( ! empty( $faqs ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php printf( esc_html__( 'Frequently Asked Questions About %s', 'ace-theme' ), esc_html( $service['title'] ) ); ?></h2>
        </div>
        <?php get_template_part( 'template-parts/faq-accordion', null, array( 'faqs' => $faqs ) ); ?>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 7: TESTIMONIALS -->
<?php
$all_testimonials = ace_get_testimonials();
$service_testimonials = array_filter( $all_testimonials, function( $t ) use ( $service_slug ) {
    return isset( $t['service'] ) && $t['service'] === $service_slug;
} );
if ( ! empty( $service_testimonials ) ) :
?>
<section class="section bg-white">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php esc_html_e( 'What Our Clients Say', 'ace-theme' ); ?></h2>
        </div>
        <div class="testimonial-carousel">
            <button class="carousel-arrow carousel-prev" aria-label="<?php esc_attr_e( 'Previous testimonial', 'ace-theme' ); ?>"><i class="fas fa-chevron-left"></i></button>

            <div class="carousel-track">
                <?php $card_index = 0; foreach ( $service_testimonials as $t ) :
                    $rating = isset( $t['rating'] ) ? intval( $t['rating'] ) : 5;
                    $initial = strtoupper( substr( $t['name'], 0, 1 ) );
                ?>
                <div class="service-testimonial-card<?php echo 0 === $card_index ? ' active' : ''; ?>">
                    <div class="service-testimonial-stars">
                        <?php for ( $i = 0; $i < $rating; $i++ ) : ?>
                            <i class="fas fa-star"></i>
                        <?php endfor; ?>
                    </div>
                    <blockquote class="service-testimonial-text">
                        &ldquo;<?php echo esc_html( $t['text'] ); ?>&rdquo;
                    </blockquote>
                    <div class="service-testimonial-author">
                        <div class="service-testimonial-avatar"><?php echo esc_html( $initial ); ?></div>
                        <div class="service-testimonial-info">
                            <strong><?php echo esc_html( $t['name'] ); ?></strong>
                            <?php if ( ! empty( $t['location'] ) ) : ?>
                                <span><?php echo esc_html( $t['location'] ); ?></span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php $card_index++; endforeach; ?>
            </div>

            <button class="carousel-arrow carousel-next" aria-label="<?php esc_attr_e( 'Next testimonial', 'ace-theme' ); ?>"><i class="fas fa-chevron-right"></i></button>

            <div class="carousel-dots">
                <?php for ( $d = 0; $d < count( $service_testimonials ); $d++ ) : ?>
                    <button class="carousel-dot<?php echo 0 === $d ? ' active' : ''; ?>" data-index="<?php echo intval( $d ); ?>" aria-label="<?php printf( esc_attr__( 'Testimonial %d', 'ace-theme' ), $d + 1 ); ?>"></button>
                <?php endfor; ?>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 8: RELATED SERVICES -->
<?php if ( ! empty( $related ) ) : ?>
<section class="section bg-light">
    <div class="container">
        <div class="section-header text-center">
            <h2><?php esc_html_e( 'You Might Also Be Interested In', 'ace-theme' ); ?></h2>
        </div>
        <div class="related-services-grid">
            <?php
            foreach ( $related as $related_slug ) :
                $related_service = ace_get_service( $related_slug );
                if ( $related_service ) :
                    get_template_part( 'template-parts/service-card', null, array( 'service' => $related_service ) );
                endif;
            endforeach;
            ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- SECTION 9: FINAL CTA -->
<section class="cta-banner bg-navy">
    <div class="container">
        <div class="cta-banner-inner">
            <div class="cta-banner-content">
                <h2><?php echo esc_html( $cta_headline ); ?></h2>
                <p><?php echo esc_html( $cta_subheadline ); ?></p>
            </div>
            <div class="cta-banner-actions">
                <a href="<?php echo esc_url( home_url( '/contact/' ) ); ?>" class="btn btn-primary btn-lg" data-popup="consultation">
                    <?php esc_html_e( 'Get Your Free Consultation', 'ace-theme' ); ?>
                </a>
                <a href="<?php echo esc_url( ace_get_phone_link() ); ?>" class="btn btn-ghost btn-lg">
                    <?php echo ace_icon( 'phone', 18 ); ?>
                    <?php echo esc_html( ace_get_phone() ); ?>
                </a>
            </div>
        </div>
    </div>
</section>

<?php get_footer(); ?>
