<?php
/**
 * FRESH Manager — Team Members Admin Page
 */
defined( 'ABSPATH' ) || exit;

/* ---------- HELPERS ---------- */

/**
 * Get team members from plugin options.
 */
function freshm_get_team_members() {
    $members = get_option( 'freshm_team_members', array() );
    if ( ! is_array( $members ) ) return array();
    return array_values( array_filter( $members, function( $m ) {
        return ! empty( $m['name'] );
    } ) );
}

/**
 * Get default team members data.
 */
function freshm_get_default_team() {
    return array(
        array(
            'name'  => 'בר גרשונוביץ',
            'role'  => 'מנכ"ל, אסטרטגיה וקמפיינים',
            'photo' => '',
            'bio'   => "בר הקים את FRESH מתוך אובססיה אחת: להוכיח שמסעדות יכולות לצמוח דיגיטלית בלי לוותר על האותנטיות שלהן. עם ניסיון של למעלה מעשור בשיווק דיגיטלי, הוא מוביל את האסטרטגיה של כל לקוח ועדיין לא מוכן לשחרר את הקמפיינים - כי שם נמצאים הפרטים הקטנים שעושים את ההבדל בין קמפיין בסדר לקמפיין שממלא מסעדה. בר מאמין שעסק טוב נבנה על מספרים ברורים, יחסי אמון ועוד כוס קפה אחת עם הלקוח. מגיע ראשון למשרד, יוצא אחרון, ותמיד עם רשימת רעיונות חדשים.",
            'favorite_dish' => 'אנטריקוט על הגריל עם חמאת עשבי תיבול ופירה חלק',
            'status' => 'publish',
        ),
        array(
            'name'  => 'אמיר זרצקי',
            'role'  => 'סמנכ"ל ומנהל מכירות',
            'photo' => '',
            'bio'   => "אמיר הוא האיש שמסעדנים מתקשרים אליו כשהם רוצים להבין מה באמת אפשר לעשות עם הדיגיטל. עם רקע עשיר במכירות ובפיתוח עסקי, אמיר מוביל את תהליך הליווי של לקוחות חדשים - מהשיחה הראשונה ועד שהתוצאות מתחילות לרוץ. הוא יודע להקשיב, לשאול את השאלות הנכונות, ולבנות תוכנית שמתאימה בדיוק לגודל ולשלב של המסעדה. אמיר הוא מהאנשים שאחרי שיחה של 15 דקות אתם מרגישים שהוא מכיר את המסעדה שלכם טוב כמעט כמוכם. זה לא קסם - זה ניסיון.",
            'favorite_dish' => 'שוק טלה צלוי לאט עם ירקות שורש ורוטב דמי-גלאס',
            'status' => 'publish',
        ),
        array(
            'name'  => 'אמילי אוטקו',
            'role'  => 'סמנכ"לית תוכן',
            'photo' => '',
            'bio'   => "אמילי אחראית על כל מילה, כל משפט וכל סיפור שיוצא מ-FRESH. היא מובילה את אסטרטגיית התוכן עבור כל הלקוחות - מזווית הצילום ועד הכיתוב של הפוסט, מהסטורי הבוקר ועד הקמפיין העונתי. אמילי מאמינה שתוכן טוב למסעדה לא מתחיל במילים - הוא מתחיל בהבנה של מה גורם לאנשים לבחור מסעדה. היא יודעת לקחת מנה פשוטה ולהפוך אותה לסיפור שאנשים רוצים לשתף. וזה בדיוק מה שמבדיל פוסט שעובר ברקע לפוסט שגורם למישהו להזמין שולחן.",
            'favorite_dish' => 'פסטה קליו א פפה עם פרמזן מגורר ופלפל שחור טרי',
            'status' => 'publish',
        ),
        array(
            'name'  => 'טליה אלבז',
            'role'  => 'מנהלת סושיאל ושיתופי פעולה עם יוצרי תוכן',
            'photo' => '',
            'bio'   => "טליה היא מי שגורמת למסעדות להופיע בפיד של האנשים הנכונים - ולא רק כמודעה, אלא כתוכן שאנשים באמת רוצים לראות. היא מנהלת את שיתופי הפעולה עם בלוגרים, יוצרי תוכן וכוכבי רשתות חברתיות, ומובילה את הנוכחות הסושיאלית של לקוחות FRESH. טליה יודעת לבחור את היוצר המתאים, לבנות בריף שמייצר תוכן אותנטי, ולוודא שהתוצאה מביאה תוצאות - לא רק לייקים.",
            'favorite_dish' => 'סלמון טאטאקי עם אבוקדו, אדממה וטריאקי',
            'status' => 'publish',
        ),
        array(
            'name'  => 'עדן',
            'role'  => 'מנהלת סושיאל',
            'photo' => '',
            'bio'   => "עדן חיה ונושמת רשתות חברתיות. היא זו שמנהלת את הפידים, הסטוריז והקהילות של מסעדות FRESH ביום-יום. מתזמון פוסטים ועד ניהול תגובות, מסטורי שאלות ועד רילס שמזמינים אינטראקציה - עדן מוודאת שהנוכחות הדיגיטלית של כל מסעדה חיה, פעילה, ועקבית. כי בסושיאל, עקביות היא הסוד.",
            'favorite_dish' => 'שקשוקה עם פלפלים קלויים, פטה מפוררת ולחם טאבון חם',
            'status' => 'publish',
        ),
        array(
            'name'  => 'שניר הכוכב',
            'role'  => 'צלם וידאו וסטילס',
            'photo' => '',
            'bio'   => "שניר הוא מי שהופך צלחת אוכל לתמונה שגורמת לאנשים לבלוע רוק. ממש. עם עין חדה לפרטים ויכולת להוציא את הצד הכי מפתה של כל מנה, שניר מוביל את מחלקת הצילום של FRESH - סטילס, וידאו ורילסים שנראים כאילו יצאו מפיד של מסעדת מישלן. הוא מבין שצילום אוכל זה לא רק תאורה וזווית - זה להעביר ריח וטעם דרך מסך.",
            'favorite_dish' => "המבורגר כבד אווז עם ג'אם בצל, צ'דר מעושן וצ'יפס בטטה",
            'status' => 'publish',
        ),
        array(
            'name'  => 'סתיו מלמד',
            'role'  => 'מנהלת מערכות משלוחים והזמנות שולחן',
            'photo' => '',
            'bio'   => "סתיו שולטת בכל מה שקורה מאחורי הקלעים של הפלטפורמות - Wolt, 10bis, Cibus, Tabit, Eatbit ועוד. היא יודעת איך לשפר דירוג, להוריד עמלות, לעדכן תפריטים ולמקסם הזמנות - בלי שהמסעדן צריך לפתוח את האפליקציה. סתיו מנהלת את הפלטפורמות בצורה שכל הזמנה שנכנסת שווה יותר ושכל ביטול נמנע מראש.",
            'favorite_dish' => 'פיצה מרגריטה עם בצק דק, מוצרלה דה בופלה ובזיליקום טרי',
            'status' => 'publish',
        ),
        array(
            'name'  => 'נועם כרמי',
            'role'  => 'שיפור מוניטין וביקורות',
            'photo' => '',
            'bio'   => "נועם יודע שביקורת אחת שלילית ב-Google יכולה לעלות למסעדה עשרות לקוחות. ולכן הוא לא משאיר את זה ליד המקרה. הוא בונה אסטרטגיות לאיסוף ביקורות חיוביות, מגיב בצורה מקצועית לכל ביקורת - חיובית ושלילית - ומוודא שהתמונה הכללית של המסעדה ב-Google, ב-TripAdvisor וברשתות החברתיות משקפת את מה שהמסעדה באמת מציעה.",
            'favorite_dish' => "ראמן טונקוטסו עם צ'אשו, ביצה רכה ונורי",
            'status' => 'publish',
        ),
        array(
            'name'  => 'בר איגל',
            'role'  => 'עורכת וידאו ראשית',
            'photo' => '',
            'bio'   => "בר לוקחת גלם ממסעדה והופכת אותו לתוכן שגורם לאנשים לעצור לגלול. כעורכת הוידאו הראשית של FRESH, היא אחראית על כל רילס, סרטון וקמפיין ויזואלי שיוצא מהמערכת. היא יודעת בדיוק מתי לחתוך, מתי להאט, ואיזה שניה של סירטוט גבינה או שבירת קרום היא זו שתעשה את ההבדל.",
            'favorite_dish' => 'פונדו שוקולד עם תותים, מרשמלו ובראוניז חמים',
            'status' => 'publish',
        ),
        array(
            'name'  => 'הילה ארנון',
            'role'  => 'מנהלת מחלקת מיתוג ועיצוב',
            'photo' => '',
            'bio'   => "הילה מובילה את כל מה שקשור לזהות החזותית של מסעדות FRESH - מלוגו ושפה עיצובית ועד תפריט מודפס ואריזות משלוחים. היא מאמינה שמיתוג מסעדה טוב מתחיל בהבנה של מה המסעדה באמת רוצה להגיד - ורק אז עוברים לצבעים, פונטים ועיצוב. הילה יודעת שתפריט מעוצב נכון לא רק נראה טוב - הוא מוכר יותר. היא גם ממש אובססיבית לפרטים. הפיקסל הזה זז שמאלה? היא כבר ראתה.",
            'favorite_dish' => 'סושי אומקסה עם סלמון להבה, אבוקדו ומיונז כמהין',
            'status' => 'publish',
        ),
        array(
            'name'  => 'עמית חן',
            'role'  => 'ניתוח, דוחות ופיתוח AI',
            'photo' => '',
            'bio'   => "עמית הוא מי שהופך מספרים לתובנות שמזיזות את העסק קדימה. הוא בונה דשבורדים, מנתח ביצועים ומפתח כלי AI פנימיים שעוזרים ל-FRESH לעבוד חכם יותר. כשקמפיין רץ - עמית יודע להגיד תוך דקות אם הוא עובד, מה צריך לשנות, ואיפה כל שקל הלך. עמית מאמין שאינטואיציה זה נחמד, אבל דאטה זה מה שמנצח. ולכן כל החלטה ב-FRESH נתמכת במספרים אמיתיים - לא בתחושות בטן.",
            'favorite_dish' => 'חומוס חם עם פול, טחינה גולמית וביצה עלומה',
            'status' => 'publish',
        ),
    );
}

/* ---------- SAVE HANDLER ---------- */
function freshm_save_team_handler() {
    if ( ! isset( $_POST['freshm_save_team'] ) || ! current_user_can( 'manage_options' ) ) return;
    check_admin_referer( 'freshm_team_action', 'freshm_team_nonce' );

    // Seed defaults if requested
    if ( ! empty( $_POST['freshm_seed_team'] ) ) {
        update_option( 'freshm_team_members', freshm_get_default_team() );
        freshm_sync_team_to_cpt( freshm_get_default_team() );
        wp_safe_redirect( add_query_arg( 'freshm_saved', '1' ) );
        exit;
    }

    $raw = isset( $_POST['freshm_team'] ) ? $_POST['freshm_team'] : array();
    $members = array();

    foreach ( $raw as $item ) {
        if ( empty( $item['name'] ) ) continue;
        $members[] = array(
            'name'          => sanitize_text_field( $item['name'] ),
            'role'          => sanitize_text_field( $item['role'] ?? '' ),
            'photo'         => esc_url_raw( $item['photo'] ?? '' ),
            'bio'           => sanitize_textarea_field( $item['bio'] ?? '' ),
            'favorite_dish' => sanitize_text_field( $item['favorite_dish'] ?? '' ),
            'status'        => ( $item['status'] ?? 'publish' ) === 'draft' ? 'draft' : 'publish',
        );
    }

    update_option( 'freshm_team_members', $members );
    freshm_sync_team_to_cpt( $members );

    wp_safe_redirect( add_query_arg( 'freshm_saved', '1' ) );
    exit;
}
add_action( 'admin_init', 'freshm_save_team_handler' );

/* ---------- AUTO-SYNC TO CPT ---------- */
function freshm_sync_team_to_cpt( $members ) {
    $existing = get_posts( array(
        'post_type'      => 'fresh_team',
        'posts_per_page' => -1,
        'post_status'    => array( 'publish', 'draft', 'pending' ),
    ) );
    $by_title = array();
    foreach ( $existing as $p ) $by_title[ $p->post_title ] = $p;

    $synced = array();
    $order = 0;
    foreach ( $members as $m ) {
        if ( empty( $m['name'] ) ) continue;
        $synced[] = $m['name'];

        $post_data = array(
            'post_type'    => 'fresh_team',
            'post_title'   => $m['name'],
            'post_name'    => sanitize_title( $m['name'] ),
            'post_content' => '',
            'post_status'  => $m['status'] ?? 'publish',
            'menu_order'   => $order++,
        );

        if ( isset( $by_title[ $m['name'] ] ) ) {
            $post_data['ID'] = $by_title[ $m['name'] ]->ID;
            wp_update_post( $post_data );
            $pid = $by_title[ $m['name'] ]->ID;
        } else {
            $pid = wp_insert_post( $post_data );
        }

        if ( ! is_wp_error( $pid ) && $pid ) {
            update_post_meta( $pid, 'team_role', $m['role'] );
            update_post_meta( $pid, 'team_bio', $m['bio'] );
            update_post_meta( $pid, 'team_favorite_dish', $m['favorite_dish'] );

            // Set featured image from URL
            if ( $m['photo'] && function_exists( 'freshm_set_featured_image_from_url' ) ) {
                freshm_set_featured_image_from_url( $pid, $m['photo'] );
            }
        }
    }

    // Trash CPT posts no longer in plugin data
    foreach ( $by_title as $title => $post ) {
        if ( ! in_array( $title, $synced, true ) ) {
            wp_trash_post( $post->ID );
        }
    }
}

/* ---------- ADMIN PAGE RENDERER ---------- */
function freshm_page_team() {
    $saved = isset( $_GET['freshm_saved'] );
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'fresh-team' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>צוות</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">&#10004; השינויים נשמרו והנתונים סונכרנו לאתר!</div>
        <?php endif; ?>
        <?php freshm_team_inner(); ?>
    </div>
        </div>
    </div>
    <?php
}

/**
 * Inner content for team — used standalone and inside unified tabs.
 */
function freshm_team_inner() {
    $members = get_option( 'freshm_team_members', array() );
    if ( ! is_array( $members ) ) $members = array();
    ?>
    <form method="post" enctype="multipart/form-data">
        <?php wp_nonce_field( 'freshm_team_action', 'freshm_team_nonce' ); ?>
        <input type="hidden" name="freshm_save_team" value="1">

        <div class="freshm-cards-toolbar">
            <button type="button" class="freshm-btn freshm-add-card" data-target="team">+ הוסף חבר צוות</button>
            <button type="submit" name="freshm_seed_team" value="1" class="freshm-btn" style="margin-right:1rem;" onclick="return confirm('טען מחדש את כל 11 חברי הצוות? הנתונים הקיימים יוחלפו.');">&#x1f504; טען ברירת מחדל</button>
            <span class="freshm-cards-count"><?php echo count( $members ); ?> חברי צוות</span>
        </div>

        <div class="freshm-cards-list" id="freshm-team-list">
            <?php
            if ( ! empty( $members ) ) :
                foreach ( $members as $idx => $m ) :
                    freshm_render_team_card( $idx, $m );
                endforeach;
            endif;
            ?>
        </div>

        <!-- Card Template (hidden) -->
        <script type="text/html" id="freshm-team-template">
            <?php freshm_render_team_card( '__IDX__', array() ); ?>
        </script>

        <div class="freshm-save-bar">
            <button type="submit" class="freshm-btn-save">&#x1f4be; שמור הכל וסנכרן</button>
        </div>
    </form>

    <script>
    jQuery(function($){
        // Add new card
        $(document).on('click', '.freshm-add-card[data-target="team"]', function(){
            var $tpl = $('#freshm-team-template');
            if (!$tpl.length) return;
            var html = $tpl.html().replace(/__IDX__/g, Date.now());
            var $card = $(html);
            $card.addClass('open');
            $('#freshm-team-list').append($card);
            $('html, body').animate({ scrollTop: $card.offset().top - 100 }, 300);
            var count = $('#freshm-team-list').find('.freshm-card').length;
            $(this).closest('form').find('.freshm-cards-count').text(count + ' חברי צוות');
        });

        // Delete card
        $(document).on('click', '.freshm-delete-card', function(e){
            e.stopPropagation();
            if (!confirm('למחוק חבר צוות זה?')) return;
            var $card = $(this).closest('.freshm-card');
            var $list = $card.closest('.freshm-cards-list');
            $card.slideUp(300, function(){
                $(this).remove();
                var count = $list.find('.freshm-card').length;
                $list.closest('form').find('.freshm-cards-count').text(count + ' חברי צוות');
            });
        });

        // Update card title in real-time
        $(document).on('input', '#freshm-team-list .freshm-card-name-input', function(){
            var val = $(this).val() || 'חבר צוות חדש';
            var role = $(this).closest('.freshm-card-body').find('.freshm-card-role-input').val();
            $(this).closest('.freshm-card').find('.freshm-card-title').text(val + (role ? ' - ' + role : ''));
        });
        $(document).on('input', '#freshm-team-list .freshm-card-role-input', function(){
            var name = $(this).closest('.freshm-card-body').find('.freshm-card-name-input').val() || 'חבר צוות חדש';
            var role = $(this).val();
            $(this).closest('.freshm-card').find('.freshm-card-title').text(name + (role ? ' - ' + role : ''));
        });

        // Sortable
        if ($.fn.sortable) {
            $('#freshm-team-list').sortable({
                handle: '.freshm-card-handle',
                placeholder: 'ui-sortable-placeholder',
                opacity: 0.8,
                tolerance: 'pointer'
            });
        }
    });
    </script>
    <?php
}

/* ---------- SINGLE TEAM CARD RENDERER ---------- */
function freshm_render_team_card( $idx, $m ) {
    $m = wp_parse_args( $m, array(
        'name' => '', 'role' => '', 'photo' => '', 'bio' => '', 'favorite_dish' => '', 'status' => 'publish',
    ) );
    $n = "freshm_team[{$idx}]";
    $status_label = $m['status'] === 'draft' ? 'טיוטה' : 'פעיל';
    $status_class = $m['status'] === 'draft' ? 'draft' : 'published';
    ?>
    <div class="freshm-card" data-type="team">
        <div class="freshm-card-header" onclick="this.parentElement.classList.toggle('open')">
            <span class="freshm-card-handle">&#9776;</span>
            <span class="freshm-card-title"><?php echo $m['name'] ? esc_html( $m['name'] . ' - ' . $m['role'] ) : 'חבר צוות חדש'; ?></span>
            <span class="freshm-card-badge <?php echo $status_class; ?>"><?php echo $status_label; ?></span>
            <span class="freshm-card-toggle">&#9660;</span>
        </div>
        <div class="freshm-card-body">
            <div class="freshm-card-section">
                <div class="freshm-grid">
                    <div class="freshm-field">
                        <label>שם *</label>
                        <input type="text" name="<?php echo $n; ?>[name]" value="<?php echo esc_attr( $m['name'] ); ?>" class="freshm-card-name-input" required>
                    </div>
                    <div class="freshm-field">
                        <label>תפקיד</label>
                        <input type="text" name="<?php echo $n; ?>[role]" value="<?php echo esc_attr( $m['role'] ); ?>" class="freshm-card-role-input">
                    </div>
                    <div class="freshm-field">
                        <label>סטטוס</label>
                        <select name="<?php echo $n; ?>[status]">
                            <option value="publish" <?php selected( $m['status'], 'publish' ); ?>>פעיל</option>
                            <option value="draft" <?php selected( $m['status'], 'draft' ); ?>>טיוטה</option>
                        </select>
                    </div>
                </div>

                <div class="freshm-field">
                    <label>תמונה</label>
                    <div class="freshm-image-field">
                        <input type="hidden" name="<?php echo $n; ?>[photo]" value="<?php echo esc_attr( $m['photo'] ); ?>" class="freshm-image-input">
                        <div class="freshm-image-preview"><?php if ( $m['photo'] ) echo '<img src="' . esc_url( $m['photo'] ) . '">'; ?></div>
                        <button type="button" class="freshm-btn freshm-upload-btn">&#x1f4f7;</button>
                        <button type="button" class="freshm-btn freshm-remove-btn" <?php echo $m['photo'] ? '' : 'style="display:none"'; ?>>&#x2715;</button>
                    </div>
                </div>

                <div class="freshm-field">
                    <label>ביוגרפיה</label>
                    <textarea name="<?php echo $n; ?>[bio]" rows="5" style="width:100%;min-height:120px;"><?php echo esc_textarea( $m['bio'] ); ?></textarea>
                </div>

                <div class="freshm-field">
                    <label>המנה האהובה</label>
                    <input type="text" name="<?php echo $n; ?>[favorite_dish]" value="<?php echo esc_attr( $m['favorite_dish'] ); ?>" placeholder="למשל: המבורגר עם צ'יפס">
                </div>
            </div>

            <div class="freshm-card-actions">
                <button type="button" class="freshm-delete-card">&#x1f5d1; מחק חבר צוות</button>
            </div>
        </div>
    </div>
    <?php
}
