<?php
/**
 * Plugin Name: FRESH Manager
 * Plugin URI: https://fresh.agency
 * Description: ניהול מלא של תמת FRESH — תוכן, תמונות, וידאו, צבעים, מגה-מנו ועוד. ללא צורך בקוד.
 * Version: 1.0.0
 * Author: FRESH Agency
 * Author URI: https://fresh.agency
 * Text Domain: fresh-manager
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

defined( 'ABSPATH' ) || exit;

define( 'FRESHM_VERSION', '1.1.0' );
define( 'FRESHM_PATH', plugin_dir_path( __FILE__ ) );
define( 'FRESHM_URL', plugin_dir_url( __FILE__ ) );

// Include modules
require_once FRESHM_PATH . 'includes/service-icons.php';
require_once FRESHM_PATH . 'includes/service-data.php';
require_once FRESHM_PATH . 'includes/category-data.php';
require_once FRESHM_PATH . 'admin/views/page-stories.php';
require_once FRESHM_PATH . 'admin/views/page-services-mgr.php';
require_once FRESHM_PATH . 'admin/views/page-categories.php';
require_once FRESHM_PATH . 'admin/views/page-team.php';

/* ==========================================================================
   ADMIN MENU
   ========================================================================== */
add_action( 'admin_menu', 'freshm_admin_menu' );
function freshm_admin_menu() {
    add_menu_page(
        'FRESH Manager',
        'FRESH Manager',
        'manage_options',
        'fresh-manager',
        'freshm_page_global',
        'dashicons-restaurant',
        3
    );
    add_submenu_page( 'fresh-manager', 'הגדרות כלליות', 'הגדרות כלליות', 'manage_options', 'fresh-manager', 'freshm_page_global' );
    add_submenu_page( 'fresh-manager', 'דף הבית', 'דף הבית', 'manage_options', 'fresh-homepage', 'freshm_page_homepage' );
    add_submenu_page( 'fresh-manager', 'מי אנחנו', 'מי אנחנו', 'manage_options', 'fresh-about', 'freshm_page_about' );
    add_submenu_page( 'fresh-manager', 'שירותים', 'שירותים', 'manage_options', 'freshm-services', 'freshm_page_services_unified' );
    add_submenu_page( 'fresh-manager', 'הצלחת לקוחות', 'הצלחת לקוחות', 'manage_options', 'freshm-stories', 'freshm_page_successes_unified' );
    add_submenu_page( 'fresh-manager', 'צור קשר', 'צור קשר', 'manage_options', 'fresh-contact', 'freshm_page_contact' );
    add_submenu_page( 'fresh-manager', 'גלריית וידאו', 'גלריית וידאו', 'manage_options', 'fresh-videos', 'freshm_page_videos' );
    add_submenu_page( 'fresh-manager', 'בלוג', 'בלוג', 'manage_options', 'fresh-blog', 'freshm_page_blog' );
    add_submenu_page( 'fresh-manager', 'שאלות ותשובות', 'שאלות ותשובות', 'manage_options', 'fresh-faq', 'freshm_page_faq' );
    add_submenu_page( 'fresh-manager', 'דף 404', 'דף 404', 'manage_options', 'fresh-404', 'freshm_page_404' );
    add_submenu_page( 'fresh-manager', 'קודי מעקב', 'קודי מעקב', 'manage_options', 'fresh-tracking', 'freshm_page_tracking' );
    // Hidden pages (merged into parent pages, kept for backward compat URLs)
    add_submenu_page( null, 'צוות', '', 'manage_options', 'fresh-team', 'freshm_page_team' );
    add_submenu_page( null, 'פוטר', '', 'manage_options', 'fresh-footer', 'freshm_page_footer' );
}

/* ==========================================================================
   OLD SLUG REDIRECTS — redirect removed menu pages to new unified pages
   ========================================================================== */
add_action( 'admin_init', 'freshm_redirect_old_pages', 1 );
function freshm_redirect_old_pages() {
    if ( ! isset( $_GET['page'] ) ) return;
    $page = sanitize_text_field( $_GET['page'] );
    $redirects = array(
        'fresh-services'   => array( 'page' => 'freshm-services', 'tab' => 'settings' ),
        'fresh-portfolio'  => array( 'page' => 'freshm-stories',  'tab' => 'settings' ),
        'fresh-categories' => array( 'page' => 'freshm-services', 'tab' => 'pages' ),
    );
    if ( isset( $redirects[ $page ] ) ) {
        $r = $redirects[ $page ];
        wp_safe_redirect( admin_url( 'admin.php?page=' . $r['page'] . '&tab=' . $r['tab'] ) );
        exit;
    }
}

/* ==========================================================================
   ADMIN ASSETS
   ========================================================================== */
add_action( 'admin_enqueue_scripts', 'freshm_admin_assets' );
function freshm_admin_assets( $hook ) {
    if ( strpos( $hook, 'fresh' ) === false ) return;

    wp_enqueue_media();
    wp_enqueue_style( 'wp-color-picker' );
    wp_enqueue_script( 'wp-color-picker' );
    wp_enqueue_script( 'jquery-ui-sortable' );
    wp_enqueue_style( 'freshm-admin', FRESHM_URL . 'admin/css/admin.css', array(), FRESHM_VERSION );
    wp_enqueue_script( 'freshm-admin', FRESHM_URL . 'admin/js/admin.js', array( 'jquery', 'wp-color-picker' ), FRESHM_VERSION, true );
    wp_localize_script( 'freshm-admin', 'freshmData', array(
        'uploadTitle'  => 'בחרו תמונה',
        'uploadButton' => 'השתמש בתמונה זו',
    ) );

    // Hide WP submenu — we use our own sidebar
    echo '<style>#toplevel_page_fresh-manager .wp-submenu { display: none !important; }</style>';
}

/* ==========================================================================
   SIDEBAR NAVIGATION
   ========================================================================== */
function freshm_sidebar_icon( $name ) {
    $icons = array(
        'home'     => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/><polyline points="9 22 9 12 15 12 15 22"/></svg>',
        'users'    => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M23 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/></svg>',
        'star'     => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/></svg>',
        'trophy'   => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M6 9H4.5a2.5 2.5 0 0 1 0-5H6"/><path d="M18 9h1.5a2.5 2.5 0 0 0 0-5H18"/><path d="M4 22h16"/><path d="M10 14.66V17c0 .55-.47.98-.97 1.21C7.85 18.75 7 20 7 22"/><path d="M14 14.66V17c0 .55.47.98.97 1.21C16.15 18.75 17 20 17 22"/><path d="M18 2H6v7a6 6 0 0 0 12 0V2Z"/></svg>',
        'video'    => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="23 7 16 12 23 17 23 7"/><rect x="1" y="5" width="15" height="14" rx="2" ry="2"/></svg>',
        'edit'     => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/></svg>',
        'help'     => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>',
        'phone'    => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"/></svg>',
        'settings' => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06A1.65 1.65 0 0 0 4.68 15a1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06A1.65 1.65 0 0 0 9 4.68a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06A1.65 1.65 0 0 0 19.4 9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/></svg>',
        'alert'    => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="12"/><line x1="12" y1="16" x2="12.01" y2="16"/></svg>',
        'code'     => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="16 18 22 12 16 6"/><polyline points="8 6 2 12 8 18"/></svg>',
    );
    return isset( $icons[ $name ] ) ? $icons[ $name ] : '';
}

function freshm_render_sidebar( $current_slug = '' ) {
    if ( ! $current_slug && isset( $_GET['page'] ) ) {
        $current_slug = sanitize_key( $_GET['page'] );
    }

    $nav_items = array(
        array( 'slug' => 'fresh-homepage',  'label' => 'דף הבית',         'icon' => 'home' ),
        array( 'slug' => 'fresh-about',     'label' => 'מי אנחנו',        'icon' => 'users' ),
        array( 'slug' => 'freshm-services', 'label' => 'שירותים',          'icon' => 'star' ),
        array( 'slug' => 'freshm-stories',  'label' => 'הצלחת לקוחות',     'icon' => 'trophy' ),
        array( 'slug' => 'fresh-videos',    'label' => 'גלריית וידאו',     'icon' => 'video' ),
        array( 'slug' => 'fresh-blog',      'label' => 'בלוג',             'icon' => 'edit' ),
        array( 'slug' => 'fresh-faq',       'label' => 'שאלות ותשובות',    'icon' => 'help' ),
        array( 'slug' => 'fresh-contact',   'label' => 'צור קשר',          'icon' => 'phone' ),
        'divider',
        array( 'slug' => 'fresh-manager',   'label' => 'הגדרות כלליות',    'icon' => 'settings' ),
        array( 'slug' => 'fresh-404',       'label' => 'דף 404',           'icon' => 'alert' ),
        array( 'slug' => 'fresh-tracking',  'label' => 'קודי מעקב',        'icon' => 'code' ),
    );
    ?>
    <div class="freshm-sidebar">
        <div class="freshm-sidebar-logo">
            <span class="fm-logo-badge">F</span>
            <span class="fm-logo-text">FRESH <span>Manager</span></span>
        </div>
        <ul class="freshm-sidebar-nav">
            <?php foreach ( $nav_items as $item ) :
                if ( $item === 'divider' ) : ?>
                    <li><div class="freshm-sidebar-divider"></div></li>
                <?php else :
                    $active = ( $current_slug === $item['slug'] ) ? ' active' : '';
                    $url = admin_url( 'admin.php?page=' . $item['slug'] );
                ?>
                    <li><a href="<?php echo esc_url( $url ); ?>" class="<?php echo $active; ?>">
                        <span class="fm-nav-icon"><?php echo freshm_sidebar_icon( $item['icon'] ); ?></span>
                        <?php echo esc_html( $item['label'] ); ?>
                    </a></li>
                <?php endif;
            endforeach; ?>
        </ul>
    </div>
    <?php
}

/* ==========================================================================
   SAVE HANDLER
   ========================================================================== */
add_action( 'admin_init', 'freshm_handle_save' );
function freshm_handle_save() {
    if ( ! isset( $_POST['freshm_save'] ) || ! current_user_can( 'manage_options' ) ) return;
    check_admin_referer( 'freshm_save_action', 'freshm_nonce' );

    $group = sanitize_text_field( wp_unslash( $_POST['freshm_save'] ) );

    // Strip WP magic quotes from POST data before processing
    $raw = wp_unslash( $_POST );

    // Collect all fields with freshm_ prefix
    $data = array();
    foreach ( $raw as $key => $value ) {
        if ( strpos( $key, 'freshm_' ) === 0 && $key !== 'freshm_save' && $key !== 'freshm_nonce' ) {
            $clean_key = sanitize_key( $key );
            if ( is_array( $value ) ) {
                $data[ $clean_key ] = freshm_sanitize_deep( $value );
            } elseif ( strpos( $clean_key, 'map_embed' ) !== false || strpos( $clean_key, 'embed' ) !== false ) {
                // Allow iframes for map/embed fields
                $data[ $clean_key ] = wp_kses( $value, array(
                    'iframe' => array( 'src' => true, 'width' => true, 'height' => true, 'style' => true, 'frameborder' => true, 'allowfullscreen' => true, 'loading' => true, 'referrerpolicy' => true, 'allow' => true ),
                ) );
            } elseif ( strpos( $clean_key, 'code_' ) !== false ) {
                // Allow raw code for tracking code injection fields
                $data[ $clean_key ] = $value;
            } else {
                $data[ $clean_key ] = wp_kses_post( $value );
            }
        }
    }

    // Merge with existing data so partial-page saves don't wipe other fields
    $existing = get_option( 'freshm_' . $group, array() );
    if ( is_array( $existing ) ) {
        $data = array_merge( $existing, $data );
    }
    update_option( 'freshm_' . $group, $data );

    // Redirect back with success message, preserving tab
    $args = array( 'freshm_saved' => '1' );
    if ( ! empty( $_POST['freshm_tab'] ) ) {
        $args['tab'] = sanitize_key( $_POST['freshm_tab'] );
    }
    wp_safe_redirect( add_query_arg( $args ) );
    exit;
}

function freshm_sanitize_deep( $arr ) {
    $clean = array();
    foreach ( $arr as $key => $value ) {
        if ( is_array( $value ) ) {
            $clean[ $key ] = freshm_sanitize_deep( $value );
        } else {
            $clean[ $key ] = wp_kses_post( $value );
        }
    }
    return $clean;
}

/* ==========================================================================
   HELPER: GET OPTION
   ========================================================================== */
function freshm_get( $group, $key, $default = '' ) {
    $data = get_option( 'freshm_' . $group, array() );
    if ( isset( $data[ 'freshm_' . $key ] ) ) {
        return wp_unslash( $data[ 'freshm_' . $key ] );
    }
    return $default;
}

function freshm_get_all( $group ) {
    $data = get_option( 'freshm_' . $group, array() );
    return wp_unslash( $data );
}

// Shorthand for getting a field value with fallback
function fm( $group, $key, $default = '' ) {
    return freshm_get( $group, $key, $default );
}

/* ==========================================================================
   DATA HELPERS — Services & Client Stories
   ========================================================================== */
/** Get all services from plugin option (published only by default) */
function freshm_get_services( $status = 'publish' ) {
    $all = get_option( 'freshm_services_data', array() );
    if ( $status === 'all' ) return $all;
    return array_filter( $all, function( $s ) use ( $status ) {
        return ( $s['status'] ?? 'publish' ) === $status;
    } );
}

/** Get a single service by slug */
function freshm_get_service_by_slug( $slug ) {
    $all = get_option( 'freshm_services_data', array() );
    foreach ( $all as $s ) {
        if ( ( $s['slug'] ?? '' ) === $slug ) return $s;
    }
    return null;
}

/** Get service categories */
function freshm_get_service_cats() {
    return get_option( 'freshm_service_categories', array() );
}

/** Get all client stories from plugin option */
function freshm_get_stories( $status = 'publish' ) {
    $all = get_option( 'freshm_client_stories', array() );
    if ( $status === 'all' ) return $all;
    return array_filter( $all, function( $s ) use ( $status ) {
        return ( $s['status'] ?? 'publish' ) === $status;
    } );
}

/** Get a single story by slug */
function freshm_get_story_by_slug( $slug ) {
    $all = get_option( 'freshm_client_stories', array() );
    foreach ( $all as $s ) {
        if ( ( $s['slug'] ?? '' ) === $slug ) return $s;
    }
    return null;
}

/* ==========================================================================
   ADMIN PAGE RENDERER
   ========================================================================== */
function freshm_page_wrap_start( $title, $group, $slug = '' ) {
    $slug  = $slug ?: ( isset( $_GET['page'] ) ? sanitize_key( $_GET['page'] ) : '' );
    $saved = isset( $_GET['freshm_saved'] );

    echo '<div class="freshm-admin-layout">';
    freshm_render_sidebar( $slug );
    echo '<div class="freshm-content">';

    echo '<div class="freshm-wrap">';
    echo '<div class="freshm-header">';
    echo '<h1>' . esc_html( $title ) . '</h1>';
    echo '</div>';
    if ( $saved ) echo '<div class="freshm-notice freshm-notice-success">✅ השינויים נשמרו בהצלחה!</div>';
    echo '<form method="post" enctype="multipart/form-data">';
    wp_nonce_field( 'freshm_save_action', 'freshm_nonce' );
    echo '<input type="hidden" name="freshm_save" value="' . esc_attr( $group ) . '">';
}

function freshm_page_wrap_end() {
    echo '<div class="freshm-save-bar">';
    echo '<button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>';
    echo '</div>';
    echo '</form></div>'; // close form + .freshm-wrap
    echo '</div></div>';  // close .freshm-content + .freshm-admin-layout
}

/* ==========================================================================
   FIELD RENDERERS
   ========================================================================== */
function freshm_field_text( $group, $key, $label, $default = '', $placeholder = '' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<input type="text" name="freshm_' . esc_attr( $key ) . '" value="' . esc_attr( $val ) . '" placeholder="' . esc_attr( $placeholder ) . '">';
    echo '</div>';
}

function freshm_field_textarea( $group, $key, $label, $default = '', $rows = 3 ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<textarea name="freshm_' . esc_attr( $key ) . '" rows="' . intval( $rows ) . '">' . esc_textarea( $val ) . '</textarea>';
    echo '</div>';
}

function freshm_field_wysiwyg( $group, $key, $label, $default = '' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field freshm-field-wide">';
    echo '<label>' . esc_html( $label ) . '</label>';
    wp_editor( $val, 'freshm_' . $key, array(
        'textarea_name' => 'freshm_' . $key,
        'textarea_rows' => 8,
        'media_buttons' => true,
        'teeny'         => false,
    ) );
    echo '</div>';
}

function freshm_field_image( $group, $key, $label, $default = '' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<div class="freshm-image-field">';
    echo '<input type="hidden" name="freshm_' . esc_attr( $key ) . '" value="' . esc_attr( $val ) . '" class="freshm-image-input">';
    echo '<div class="freshm-image-preview">';
    if ( $val ) echo '<img src="' . esc_url( $val ) . '">';
    echo '</div>';
    echo '<button type="button" class="freshm-btn freshm-upload-btn">📷 בחר תמונה</button>';
    echo '<button type="button" class="freshm-btn freshm-remove-btn" ' . ( $val ? '' : 'style="display:none"' ) . '>✕ הסר</button>';
    echo '</div></div>';
}

function freshm_field_color( $group, $key, $label, $default = '#c9a84c' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<input type="text" name="freshm_' . esc_attr( $key ) . '" value="' . esc_attr( $val ) . '" class="freshm-color-picker">';
    echo '</div>';
}

function freshm_field_url( $group, $key, $label, $default = '', $placeholder = 'https://' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<input type="url" name="freshm_' . esc_attr( $key ) . '" value="' . esc_url( $val ) . '" placeholder="' . esc_attr( $placeholder ) . '" dir="ltr">';
    echo '</div>';
}

function freshm_field_checkbox( $group, $key, $label, $default = '' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field freshm-field-checkbox">';
    echo '<input type="hidden" name="freshm_' . esc_attr( $key ) . '" value="">';
    echo '<label><input type="checkbox" name="freshm_' . esc_attr( $key ) . '" value="1"' . checked( $val, '1', false ) . '> ' . esc_html( $label ) . '</label>';
    echo '</div>';
}

function freshm_field_select( $group, $key, $label, $options, $default = '' ) {
    $val = fm( $group, $key, $default );
    echo '<div class="freshm-field">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<select name="freshm_' . esc_attr( $key ) . '">';
    foreach ( $options as $opt_val => $opt_label ) {
        echo '<option value="' . esc_attr( $opt_val ) . '"' . selected( $val, $opt_val, false ) . '>' . esc_html( $opt_label ) . '</option>';
    }
    echo '</select></div>';
}

function freshm_section_title( $title, $desc = '' ) {
    echo '<div class="freshm-section-title">';
    echo '<h2>' . esc_html( $title ) . '</h2>';
    if ( $desc ) echo '<p>' . esc_html( $desc ) . '</p>';
    echo '</div>';
}

/* ==========================================================================
   REPEATER FIELD
   ========================================================================== */
function freshm_field_repeater( $group, $key, $label, $fields, $data = null ) {
    if ( $data === null ) {
        $all_data = freshm_get_all( $group );
        $data = isset( $all_data[ 'freshm_' . $key ] ) ? $all_data[ 'freshm_' . $key ] : array();
    }
    if ( ! is_array( $data ) ) $data = array();

    echo '<div class="freshm-field freshm-field-wide">';
    echo '<label>' . esc_html( $label ) . '</label>';
    echo '<div class="freshm-repeater" data-key="freshm_' . esc_attr( $key ) . '">';

    // Template (hidden)
    echo '<script type="text/html" class="freshm-repeater-template">';
    echo '<div class="freshm-repeater-item">';
    echo '<div class="freshm-repeater-handle">☰</div>';
    echo '<div class="freshm-repeater-fields">';
    foreach ( $fields as $fkey => $fconfig ) {
        $fname = 'freshm_' . $key . '[__INDEX__][' . $fkey . ']';
        echo '<div class="freshm-rf">';
        echo '<label>' . esc_html( $fconfig['label'] ) . '</label>';
        $type = $fconfig['type'] ?? 'text';
        if ( $type === 'textarea' ) {
            echo '<textarea name="' . $fname . '" rows="2"></textarea>';
        } elseif ( $type === 'image' ) {
            echo '<div class="freshm-image-field"><input type="hidden" name="' . $fname . '" class="freshm-image-input"><div class="freshm-image-preview"></div><button type="button" class="freshm-btn freshm-upload-btn">📷</button><button type="button" class="freshm-btn freshm-remove-btn" style="display:none">✕</button></div>';
        } else {
            echo '<input type="text" name="' . $fname . '" value="" placeholder="' . esc_attr( $fconfig['placeholder'] ?? '' ) . '">';
        }
        echo '</div>';
    }
    echo '</div>';
    echo '<button type="button" class="freshm-btn freshm-repeater-remove">🗑️</button>';
    echo '</div>';
    echo '</script>';

    // Existing items
    echo '<div class="freshm-repeater-items">';
    if ( ! empty( $data ) ) {
        foreach ( $data as $idx => $item ) {
            echo '<div class="freshm-repeater-item">';
            echo '<div class="freshm-repeater-handle">☰</div>';
            echo '<div class="freshm-repeater-fields">';
            foreach ( $fields as $fkey => $fconfig ) {
                $fname = 'freshm_' . $key . '[' . $idx . '][' . $fkey . ']';
                $fval = $item[ $fkey ] ?? '';
                echo '<div class="freshm-rf">';
                echo '<label>' . esc_html( $fconfig['label'] ) . '</label>';
                $type = $fconfig['type'] ?? 'text';
                if ( $type === 'textarea' ) {
                    echo '<textarea name="' . $fname . '" rows="2">' . esc_textarea( $fval ) . '</textarea>';
                } elseif ( $type === 'image' ) {
                    echo '<div class="freshm-image-field"><input type="hidden" name="' . $fname . '" value="' . esc_attr( $fval ) . '" class="freshm-image-input"><div class="freshm-image-preview">';
                    if ( $fval ) echo '<img src="' . esc_url( $fval ) . '">';
                    echo '</div><button type="button" class="freshm-btn freshm-upload-btn">📷</button><button type="button" class="freshm-btn freshm-remove-btn" ' . ( $fval ? '' : 'style="display:none"' ) . '>✕</button></div>';
                } else {
                    echo '<input type="text" name="' . $fname . '" value="' . esc_attr( $fval ) . '" placeholder="' . esc_attr( $fconfig['placeholder'] ?? '' ) . '">';
                }
                echo '</div>';
            }
            echo '</div>';
            echo '<button type="button" class="freshm-btn freshm-repeater-remove">🗑️</button>';
            echo '</div>';
        }
    }
    echo '</div>';
    echo '<button type="button" class="freshm-btn freshm-repeater-add">+ הוסף פריט</button>';
    echo '</div></div>';
}

/* ==========================================================================
   ADMIN PAGES
   ========================================================================== */

// ---- GLOBAL SETTINGS (with sub-tabs) ----
function freshm_page_global() {
    $tab   = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'header';
    $saved = isset( $_GET['freshm_saved'] );
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'fresh-manager' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>הגדרות כלליות</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">✅ השינויים נשמרו בהצלחה!</div>
        <?php endif; ?>

        <div class="freshm-tabs">
            <button type="button" class="freshm-tab<?php echo $tab === 'header' ? ' active' : ''; ?>" data-tab="header">Header + תפריט</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'footer' ? ' active' : ''; ?>" data-tab="footer">Footer</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'cookies' ? ' active' : ''; ?>" data-tab="cookies">Cookies</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'seo' ? ' active' : ''; ?>" data-tab="seo">SEO + צבעים</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'social' ? ' active' : ''; ?>" data-tab="social">סושיאל</button>
        </div>

        <!-- Tab: Header + תפריט -->
        <div class="freshm-tab-content<?php echo $tab === 'header' ? ' active' : ''; ?>" data-tab="header">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="global">
                <?php
                $g = 'global';
                freshm_section_title( 'מידע עסקי' );
                echo '<div class="freshm-grid">';
                freshm_field_image( $g, 'logo', 'לוגו' );
                freshm_field_text( $g, 'business_name', 'שם העסק', 'FRESH' );
                freshm_field_text( $g, 'tagline', 'תיאור קצר', 'הסוכנות הדיגיטלית למסעדות' );
                freshm_field_text( $g, 'phone', 'טלפון', '050-123-4567' );
                freshm_field_text( $g, 'whatsapp', 'WhatsApp (מספר בלבד)', '972501234567' );
                freshm_field_text( $g, 'email', 'אימייל', 'hello@fresh.agency' );
                freshm_field_textarea( $g, 'address', 'כתובת', 'רוטשילד 45, תל אביב' );
                freshm_field_text( $g, 'hours', 'שעות פעילות', 'א׳-ה׳ 09:00-18:00' );
                freshm_field_text( $g, 'nav_cta_text', 'טקסט כפתור CTA בניווט', 'דברו איתנו' );
                echo '</div>';
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: Footer -->
        <div class="freshm-tab-content<?php echo $tab === 'footer' ? ' active' : ''; ?>" data-tab="footer">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="footer">
                <?php
                $fg = 'footer';
                freshm_section_title( 'מותג' );
                freshm_field_textarea( $fg, 'brand_text', 'תיאור המותג', 'הסוכנות הדיגיטלית המתמחה בשיווק ופרסום מסעדות. מברנדינג ותוכן, דרך קמפיינים, ועד CRM — הכל תחת קורת גג אחת.' );

                freshm_section_title( 'עמודת שירותים' );
                freshm_field_repeater( $fg, 'services_links', 'קישורי שירותים', array(
                    'text' => array( 'label' => 'טקסט', 'placeholder' => 'שיווק דיגיטלי' ),
                    'url'  => array( 'label' => 'קישור', 'placeholder' => '/services/' ),
                ) );

                freshm_section_title( 'עמודת חברה' );
                freshm_field_repeater( $fg, 'company_links', 'קישורי חברה', array(
                    'text' => array( 'label' => 'טקסט', 'placeholder' => 'מי אנחנו' ),
                    'url'  => array( 'label' => 'קישור', 'placeholder' => '/about/' ),
                ) );

                freshm_section_title( 'מפה בפוטר' );
                freshm_field_select( $fg, 'map_enabled', 'הצג מפה בפוטר', array( '0' => 'כבוי', '1' => 'מופעל' ), '0' );
                freshm_field_textarea( $fg, 'map_embed', 'קוד הטמעת Google Maps', '', 3 );
                echo '<p class="freshm-info">📌 העתיקו את קוד ה-iframe מ-Google Maps &rarr; Share &rarr; Embed a map.</p>';

                freshm_section_title( 'קרדיט' );
                freshm_field_text( $fg, 'copyright', 'טקסט זכויות יוצרים', '© {year} FRESH. כל הזכויות שמורות.' );
                echo '<p class="freshm-info">💡 השתמשו ב-{year} כדי להציג את השנה הנוכחית אוטומטית.</p>';
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: Cookies -->
        <div class="freshm-tab-content<?php echo $tab === 'cookies' ? ' active' : ''; ?>" data-tab="cookies">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="global">
                <?php
                $g = 'global';
                freshm_section_title( 'באנר עוגיות (Cookie Consent)' );
                freshm_field_select( $g, 'cookie_enabled', 'הפעל באנר עוגיות', array( '0' => 'כבוי', '1' => 'מופעל' ), '0' );
                freshm_field_textarea( $g, 'cookie_text', 'טקסט הבאנר', 'אתר זה משתמש בעוגיות כדי לשפר את חוויית הגלישה שלכם.' );
                freshm_field_text( $g, 'cookie_btn_text', 'טקסט כפתור אישור', 'הבנתי, אני מסכים/ה' );
                freshm_field_url( $g, 'cookie_learn_url', 'קישור למדיניות פרטיות' );
                freshm_field_text( $g, 'cookie_learn_text', 'טקסט קישור', 'מדיניות פרטיות' );
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: SEO + צבעים -->
        <div class="freshm-tab-content<?php echo $tab === 'seo' ? ' active' : ''; ?>" data-tab="seo">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="global">
                <?php
                $g = 'global';
                freshm_section_title( 'צבעים', 'צבעי המותג - משפיעים על כל האתר' );
                echo '<div class="freshm-grid">';
                freshm_field_color( $g, 'color_primary', 'צבע ראשי (זהב)', '#c9a84c' );
                freshm_field_color( $g, 'color_bg', 'רקע ראשי', '#0A0A0C' );
                freshm_field_color( $g, 'color_bg2', 'רקע משני', '#111116' );
                freshm_field_color( $g, 'color_text', 'צבע טקסט', '#E8E6E1' );
                echo '</div>';
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: סושיאל -->
        <div class="freshm-tab-content<?php echo $tab === 'social' ? ' active' : ''; ?>" data-tab="social">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="global">
                <?php
                $g = 'global';
                freshm_section_title( 'רשתות חברתיות' );
                echo '<div class="freshm-grid">';
                freshm_field_url( $g, 'social_facebook', 'Facebook' );
                freshm_field_checkbox( $g, 'social_facebook_in_header', 'הצג ב-Header' );
                freshm_field_url( $g, 'social_instagram', 'Instagram' );
                freshm_field_checkbox( $g, 'social_instagram_in_header', 'הצג ב-Header' );
                freshm_field_url( $g, 'social_tiktok', 'TikTok' );
                freshm_field_checkbox( $g, 'social_tiktok_in_header', 'הצג ב-Header' );
                freshm_field_url( $g, 'social_linkedin', 'LinkedIn' );
                freshm_field_checkbox( $g, 'social_linkedin_in_header', 'הצג ב-Header' );
                freshm_field_url( $g, 'social_youtube', 'YouTube' );
                freshm_field_checkbox( $g, 'social_youtube_in_header', 'הצג ב-Header' );
                echo '</div>';
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

    </div>
        </div>
    </div>
    <?php
}

// ---- HOMEPAGE ----
function freshm_page_homepage() {
    freshm_page_wrap_start( 'דף הבית', 'homepage' );
    $g = 'homepage';

    // Tabs
    echo '<div class="freshm-tabs">';
    $tabs = array( 'hero' => 'Hero', 'logos' => 'לוגו בר', 'about' => 'אודות', 'stats' => 'סטטיסטיקות', 'services' => 'שירותים', 'why' => 'למה אנחנו', 'showreel' => 'שואוריל', 'portfolio' => 'עבודות', 'gallery' => 'גלריה', 'testimonials' => 'המלצות', 'process' => 'תהליך', 'cta' => 'CTA' );
    foreach ( $tabs as $tid => $tlabel ) {
        echo '<button type="button" class="freshm-tab' . ( $tid === 'hero' ? ' active' : '' ) . '" data-tab="' . $tid . '">' . $tlabel . '</button>';
    }
    echo '</div>';

    // Hero
    echo '<div class="freshm-tab-content active" data-tab="hero">';
    freshm_section_title( 'Hero Section' );
    freshm_field_text( $g, 'hero_badge', 'תווית Badge', 'הסוכנות הדיגיטלית #1 למסעדנות' );
    freshm_field_text( $g, 'hero_title_1', 'שורה ראשונה', 'הסוכנות הדיגיטלית' );
    freshm_field_text( $g, 'hero_title_2', 'שורה שנייה (זהב)', 'שמבינה מסעדות' );
    freshm_field_textarea( $g, 'hero_subtitle', 'תת-כותרת', 'מברנדינג ותוכן, דרך קמפיינים ממומנים, ועד CRM ואסטרטגיה — אנחנו מביאים לקוחות למסעדה שלכם.' );
    freshm_field_text( $g, 'hero_cta_text', 'טקסט כפתור CTA', 'קבעו שיחה ←' );
    freshm_field_url( $g, 'hero_cta_url', 'קישור CTA', '/contact/' );
    freshm_field_text( $g, 'hero_cta2_text', 'טקסט כפתור משני', 'הכירו אותנו ←' );
    freshm_field_url( $g, 'hero_cta2_url', 'קישור משני', '/#about' );
    freshm_field_image( $g, 'hero_bg', 'תמונת רקע (אופציונלי)' );
    freshm_field_repeater( $g, 'hero_stats', 'סטטיסטיקות Hero', array(
        'number' => array( 'label' => 'מספר', 'placeholder' => '150' ),
        'suffix' => array( 'label' => 'סיומת', 'placeholder' => '+' ),
        'label'  => array( 'label' => 'תיאור', 'placeholder' => 'מסעדות מרוצות' ),
    ) );
    echo '</div>';

    // Logo Bar
    echo '<div class="freshm-tab-content" data-tab="logos">';
    freshm_section_title( 'לוגו בר - לקוחות' );
    freshm_field_repeater( $g, 'logos', 'לוגואים', array(
        'image' => array( 'label' => 'לוגו', 'type' => 'image' ),
        'name'  => array( 'label' => 'שם (alt)', 'placeholder' => 'שם הלקוח' ),
    ) );
    echo '</div>';

    // About — "What We Do" cards
    echo '<div class="freshm-tab-content" data-tab="about">';
    freshm_section_title( 'סקשן "מה אנחנו עושים"' );
    freshm_field_text( $g, 'about_label', 'תווית', 'מה אנחנו עושים' );
    freshm_field_text( $g, 'about_title', 'כותרת', 'לא רק משווקים.' );
    freshm_field_text( $g, 'about_title_gold', 'כותרת (חלק זהב)', 'בונים מערכת שלמה שמגדילה את המסעדה.' );
    freshm_field_repeater( $g, 'about_features', 'כרטיסים (4)', array(
        'icon'  => array( 'label' => 'אייקון (אימוג\'י)', 'placeholder' => '📢' ),
        'title' => array( 'label' => 'כותרת', 'placeholder' => 'קמפיינים שמביאים סועדים' ),
        'desc'  => array( 'label' => 'תיאור', 'type' => 'textarea' ),
    ) );
    echo '</div>';

    // Stats
    echo '<div class="freshm-tab-content" data-tab="stats">';
    freshm_section_title( 'סקשן סטטיסטיקות' );
    freshm_field_repeater( $g, 'stats', 'סטטיסטיקות (4)', array(
        'number' => array( 'label' => 'מספר', 'placeholder' => '150' ),
        'suffix' => array( 'label' => 'סיומת', 'placeholder' => '+' ),
        'label'  => array( 'label' => 'תיאור', 'placeholder' => 'מסעדות מרוצות' ),
    ) );
    echo '</div>';

    // Services label
    echo '<div class="freshm-tab-content" data-tab="services">';
    freshm_section_title( 'סקשן שירותים', 'השירותים עצמם מנוהלים דרך Custom Post Type "שירותים"' );
    freshm_field_text( $g, 'services_label', 'תווית', 'מה אנחנו עושים' );
    freshm_field_text( $g, 'services_title', 'כותרת', '12 פתרונות' );
    freshm_field_text( $g, 'services_title_gold', 'כותרת (חלק זהב)', 'לצמיחת המסעדה שלך' );
    freshm_field_textarea( $g, 'services_subtitle', 'תת-כותרת' );
    echo '<p class="freshm-info">💡 לניהול השירותים עצמם - גשו ל-<a href="' . admin_url( 'edit.php?post_type=fresh_service' ) . '">שירותים</a> בתפריט.</p>';
    echo '</div>';

    // Why FRESH
    echo '<div class="freshm-tab-content" data-tab="why">';
    freshm_section_title( 'למה אנחנו' );
    freshm_field_text( $g, 'why_label', 'תווית', 'למה אנחנו' );
    freshm_field_text( $g, 'why_title', 'כותרת', 'למה מסעדות בוחרות' );
    freshm_field_text( $g, 'why_title_gold', 'כותרת (זהב)', 'לעבוד איתנו' );
    freshm_field_textarea( $g, 'why_subtitle', 'תת-כותרת', 'כי אנחנו לא סוכנות דיגיטל שפשוט גם עובדת עם מסעדות. אנחנו סוכנות שזה כל מה שהיא עושה.' );
    freshm_field_repeater( $g, 'why_values', 'ערכים (6)', array(
        'icon'  => array( 'label' => 'אייקון', 'placeholder' => '🎯' ),
        'title' => array( 'label' => 'כותרת' ),
        'desc'  => array( 'label' => 'תיאור', 'type' => 'textarea' ),
    ) );
    echo '</div>';

    // Showreel
    echo '<div class="freshm-tab-content" data-tab="showreel">';
    freshm_section_title( 'שואוריל' );
    freshm_field_text( $g, 'videos_label', 'תווית', 'ראו אותנו בפעולה' );
    freshm_field_text( $g, 'videos_title', 'כותרת', 'הסרטונים' );
    freshm_field_text( $g, 'videos_title_gold', 'כותרת (זהב)', 'שלנו' );
    freshm_field_url( $g, 'showreel_video', 'קישור וידאו (YouTube/Vimeo)', '', 'https://www.youtube.com/watch?v=...' );
    freshm_field_image( $g, 'showreel_poster', 'תמונת כיסוי (Poster)' );
    echo '</div>';

    // Portfolio
    echo '<div class="freshm-tab-content" data-tab="portfolio">';
    freshm_section_title( 'סקשן הצלחת לקוחות', 'הצלחות הלקוחות מנוהלות דרך "הצלחת לקוחות"' );
    freshm_field_text( $g, 'portfolio_label', 'תווית', 'עבודות נבחרות' );
    freshm_field_text( $g, 'portfolio_title', 'כותרת', 'הנה מה שעשינו' );
    freshm_field_text( $g, 'portfolio_title_gold', 'כותרת (זהב)', 'ללקוחות שלנו' );
    freshm_field_text( $g, 'portfolio_count', 'כמות עבודות להציג', '5' );
    echo '</div>';

    // Gallery
    echo '<div class="freshm-tab-content" data-tab="gallery">';
    freshm_section_title( 'גלריית תוכן' );
    freshm_field_text( $g, 'gallery_label', 'תווית', 'דוגמאות תוכן' );
    freshm_field_text( $g, 'gallery_title', 'כותרת', 'תוכן שאנחנו יוצרים' );
    freshm_field_repeater( $g, 'gallery_items', 'פריטי גלריה', array(
        'image' => array( 'label' => 'תמונה', 'type' => 'image' ),
        'title' => array( 'label' => 'כותרת', 'placeholder' => '' ),
        'type'  => array( 'label' => 'סוג (Instagram, TikTok...)', 'placeholder' => 'Instagram' ),
    ) );
    echo '</div>';

    // Testimonials
    echo '<div class="freshm-tab-content" data-tab="testimonials">';
    freshm_section_title( 'המלצות' );
    freshm_field_text( $g, 'testimonials_label', 'תווית', 'מה הלקוחות אומרים' );
    freshm_field_text( $g, 'testimonials_title', 'כותרת', 'ביקורות' );
    freshm_field_text( $g, 'testimonials_title_gold', 'כותרת (זהב)', 'מלקוחות אמיתיים' );
    freshm_field_repeater( $g, 'testimonials', 'המלצות', array(
        'text'    => array( 'label' => 'טקסט', 'type' => 'textarea' ),
        'name'    => array( 'label' => 'שם', 'placeholder' => '' ),
        'role'    => array( 'label' => 'תפקיד / עסק', 'placeholder' => '' ),
        'initial' => array( 'label' => 'אות ראשונה (לאווטר)', 'placeholder' => 'א' ),
        'image'   => array( 'label' => 'תמונה (אופציונלי)', 'type' => 'image' ),
    ) );
    echo '</div>';

    // Process
    echo '<div class="freshm-tab-content" data-tab="process">';
    freshm_section_title( 'תהליך העבודה' );
    freshm_field_text( $g, 'process_label', 'תווית', 'איך זה עובד' );
    freshm_field_text( $g, 'process_title', 'כותרת', '4 צעדים' );
    freshm_field_text( $g, 'process_title_gold', 'כותרת (זהב)', 'לצמיחה' );
    freshm_field_repeater( $g, 'process_steps', 'שלבים', array(
        'number' => array( 'label' => 'מספר', 'placeholder' => '01' ),
        'icon'   => array( 'label' => 'אייקון (אימוג\'י)', 'placeholder' => '📞' ),
        'title'  => array( 'label' => 'כותרת', 'placeholder' => '' ),
        'desc'   => array( 'label' => 'תיאור', 'type' => 'textarea' ),
    ) );
    echo '</div>';

    // CTA
    echo '<div class="freshm-tab-content" data-tab="cta">';
    freshm_section_title( 'CTA - קריאה לפעולה' );
    freshm_field_text( $g, 'cta_label', 'תווית', 'מוכנים?' );
    freshm_field_text( $g, 'cta_title', 'כותרת', 'בואו נהפוך את' );
    freshm_field_text( $g, 'cta_title_gold', 'כותרת (זהב)', 'המסעדה שלכם למותג' );
    freshm_field_textarea( $g, 'cta_subtitle', 'תת-כותרת' );
    freshm_field_text( $g, 'cta_btn_text', 'טקסט כפתור', 'קבעו שיחת ייעוץ ←' );
    freshm_field_url( $g, 'cta_btn_url', 'קישור כפתור', 'tel:+972501234567' );
    freshm_field_text( $g, 'cta_btn2_text', 'טקסט כפתור 2', 'WhatsApp' );
    freshm_field_url( $g, 'cta_btn2_url', 'קישור כפתור 2', 'https://wa.me/972501234567' );
    freshm_field_text( $g, 'cta_email_placeholder', 'Placeholder לשדה מייל', 'השאירו מייל ונחזור אליכם' );
    echo '</div>';

    freshm_page_wrap_end();
}

// ---- ABOUT PAGE (with sub-tabs: כללי, ערכים, צוות) ----
function freshm_page_about() {
    $tab   = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'general';
    $saved = isset( $_GET['freshm_saved'] );
    $g = 'about';
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'fresh-about' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>מי אנחנו</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">✅ השינויים נשמרו בהצלחה!</div>
        <?php endif; ?>

        <div class="freshm-tabs">
            <button type="button" class="freshm-tab<?php echo $tab === 'general' ? ' active' : ''; ?>" data-tab="general">כללי</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'values' ? ' active' : ''; ?>" data-tab="values">ערכים</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'team' ? ' active' : ''; ?>" data-tab="team">צוות</button>
        </div>

        <!-- Tab: כללי -->
        <div class="freshm-tab-content<?php echo $tab === 'general' ? ' active' : ''; ?>" data-tab="general">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="about">

                <?php
                freshm_section_title( 'Hero' );
                freshm_field_text( $g, 'hero_title', 'כותרת', 'הסיפור של' );
                freshm_field_text( $g, 'hero_title_gold', 'כותרת (זהב)', 'FRESH' );
                freshm_field_textarea( $g, 'hero_subtitle', 'תת-כותרת' );

                freshm_section_title( 'הסיפור שלנו' );
                freshm_field_text( $g, 'story_label', 'תווית', 'מי אנחנו' );
                freshm_field_text( $g, 'story_title', 'כותרת', 'הצוות שיודע' );
                freshm_field_text( $g, 'story_title_gold', 'כותרת (זהב)', 'לדבר אוכל' );
                freshm_field_textarea( $g, 'story_text', 'טקסט', '', 5 );
                freshm_field_image( $g, 'story_image', 'תמונה' );
                freshm_field_repeater( $g, 'story_features', 'תכונות', array(
                    'icon'  => array( 'label' => 'אייקון', 'placeholder' => '◆' ),
                    'title' => array( 'label' => 'כותרת (בולד)' ),
                    'desc'  => array( 'label' => 'תיאור', 'type' => 'textarea' ),
                ) );
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: ערכים -->
        <div class="freshm-tab-content<?php echo $tab === 'values' ? ' active' : ''; ?>" data-tab="values">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="about">

                <?php
                freshm_section_title( 'ערכים' );
                freshm_field_text( $g, 'values_label', 'תווית', 'הערכים שלנו' );
                freshm_field_text( $g, 'values_title', 'כותרת', 'למה' );
                freshm_field_text( $g, 'values_title_gold', 'כותרת (זהב)', 'דווקא FRESH' );
                freshm_field_repeater( $g, 'values', 'ערכים (6)', array(
                    'icon'  => array( 'label' => 'אייקון/אימוג\'י', 'placeholder' => '🎯' ),
                    'title' => array( 'label' => 'כותרת' ),
                    'desc'  => array( 'label' => 'תיאור', 'type' => 'textarea' ),
                ) );

                freshm_section_title( 'צוות — כותרות' );
                freshm_field_text( $g, 'team_label', 'תווית', 'הצוות שלנו' );
                freshm_field_text( $g, 'team_title', 'כותרת', 'האנשים' );
                freshm_field_text( $g, 'team_title_gold', 'כותרת (זהב)', 'מאחורי הקסם' );

                freshm_section_title( 'CTA תחתון' );
                freshm_field_text( $g, 'cta_label', 'תווית', 'מתעניינים?' );
                freshm_field_text( $g, 'cta_title', 'כותרת', 'בואו נכיר' );
                freshm_field_text( $g, 'cta_title_gold', 'כותרת (זהב)', 'פנים אל פנים' );
                freshm_field_textarea( $g, 'cta_subtitle', 'תת-כותרת', 'כל מסעדה שונה. ספרו לנו על המקום שלכם ונחזור אליכם עם רעיונות - בלי התחייבות.' );
                freshm_field_text( $g, 'cta_btn_text', 'טקסט כפתור', 'קבעו שיחה ←' );
                freshm_field_text( $g, 'cta_btn2_text', 'טקסט כפתור WhatsApp', 'WhatsApp' );
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: צוות -->
        <div class="freshm-tab-content<?php echo $tab === 'team' ? ' active' : ''; ?>" data-tab="team">
            <?php freshm_team_inner(); ?>
        </div>

    </div>
        </div>
    </div>
    <?php
}

// ---- UNIFIED SERVICES PAGE (3 tabs) ----
function freshm_page_services_unified() {
    $tab   = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'list';
    $saved = isset( $_GET['freshm_saved'] );
    $reset = isset( $_GET['freshm_reset'] );
    $migrated = isset( $_GET['freshm_migrated'] ) ? intval( $_GET['freshm_migrated'] ) : false;
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'freshm-services' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>שירותים</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">השינויים נשמרו בהצלחה!</div>
        <?php endif; ?>
        <?php if ( $reset ) : ?>
            <div class="freshm-notice freshm-notice-success">השירותים אופסו ל-12 ברירות המחדל בהצלחה!</div>
        <?php endif; ?>
        <?php if ( $migrated !== false ) :
            if ( $migrated > 0 ) : ?>
                <div class="freshm-notice freshm-notice-success">הועברו <?php echo $migrated; ?> שירותים בהצלחה מה-CPT לפלאגין!</div>
            <?php else : ?>
                <div class="freshm-notice" style="background:rgba(201,168,76,0.1);border-color:var(--gold);color:#C9A84C;padding:1rem 1.5rem;border-radius:8px;margin-bottom:1.5rem;">לא נמצאו שירותים חדשים להעברה (כולם כבר קיימים בפלאגין).</div>
            <?php endif;
        endif; ?>

        <div class="freshm-tabs">
            <button type="button" class="freshm-tab<?php echo $tab === 'list' ? ' active' : ''; ?>" data-tab="list">רשימת שירותים</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'settings' ? ' active' : ''; ?>" data-tab="settings">הגדרות + מגה-מנו</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'pages' ? ' active' : ''; ?>" data-tab="pages">עמודי שירותים (14)</button>
        </div>

        <div class="freshm-tab-content<?php echo $tab === 'list' ? ' active' : ''; ?>" data-tab="list">
            <?php freshm_services_mgr_inner(); ?>
        </div>

        <div class="freshm-tab-content<?php echo $tab === 'settings' ? ' active' : ''; ?>" data-tab="settings">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="services_settings">
                <input type="hidden" name="freshm_tab" value="settings">
                <?php freshm_services_settings_inner(); ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <div class="freshm-tab-content<?php echo $tab === 'pages' ? ' active' : ''; ?>" data-tab="pages">
            <?php freshm_categories_inner(); ?>
        </div>
    </div>
        </div>
    </div>
    <?php
}

/**
 * Inner settings fields for services + mega menu (no wrap/form).
 */
function freshm_services_settings_inner() {
    $g = 'services_settings';

    freshm_section_title( 'עמוד שירותים - Hero' );
    freshm_field_text( $g, 'hero_title', 'כותרת', '12 פתרונות' );
    freshm_field_text( $g, 'hero_title_gold', 'כותרת (זהב)', 'לצמיחת המסעדה שלך' );
    freshm_field_textarea( $g, 'hero_subtitle', 'תת-כותרת' );

    freshm_section_title( 'מגה-מנו', 'ניהול התצוגה של מגה-מנו כשעומדים על "שירותים" בניווט' );
    freshm_field_select( $g, 'megamenu_enabled', 'הפעל מגה-מנו', array( '1' => 'מופעל', '0' => 'כבוי' ), '1' );
    freshm_field_text( $g, 'megamenu_cta_text', 'טקסט CTA במגה-מנו', 'צפו בכל השירותים ←' );
    freshm_field_url( $g, 'megamenu_cta_url', 'קישור CTA', '/services/' );
    freshm_field_image( $g, 'megamenu_featured_image', 'תמונה מובלטת (אופציונלי)' );
    freshm_field_text( $g, 'megamenu_featured_title', 'כותרת תמונה מובלטת', 'חבילה מותאמת אישית' );
    freshm_field_textarea( $g, 'megamenu_featured_desc', 'תיאור קצר' );

    freshm_section_title( 'CTA תחתון בעמוד שירותים' );
    freshm_field_text( $g, 'cta_label', 'תווית', 'לא בטוחים מה מתאים לכם?' );
    freshm_field_text( $g, 'cta_title', 'כותרת', 'בואו נבנה' );
    freshm_field_text( $g, 'cta_title_gold', 'כותרת (זהב)', 'חבילה מותאמת אישית' );
    freshm_field_textarea( $g, 'cta_subtitle', 'תת-כותרת', 'כל מסעדה שונה. ספרו לנו על העסק שלכם ונתאים חבילה שמתאימה בדיוק.' );
    freshm_field_text( $g, 'cta_btn_text', 'טקסט כפתור', 'קבעו שיחה ←' );
    freshm_field_text( $g, 'cta_btn2_text', 'טקסט כפתור WhatsApp', 'WhatsApp' );
}

// ---- UNIFIED SUCCESSES PAGE (2 tabs) ----
function freshm_page_successes_unified() {
    $tab   = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'stories';
    $saved = isset( $_GET['freshm_saved'] );
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'freshm-stories' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>הצלחת לקוחות</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">השינויים נשמרו בהצלחה!</div>
        <?php endif; ?>

        <div class="freshm-tabs">
            <button type="button" class="freshm-tab<?php echo $tab === 'stories' ? ' active' : ''; ?>" data-tab="stories">הצלחת לקוחות</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'settings' ? ' active' : ''; ?>" data-tab="settings">הגדרות הצלחות</button>
        </div>

        <div class="freshm-tab-content<?php echo $tab === 'stories' ? ' active' : ''; ?>" data-tab="stories">
            <?php freshm_stories_inner(); ?>
        </div>

        <div class="freshm-tab-content<?php echo $tab === 'settings' ? ' active' : ''; ?>" data-tab="settings">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="portfolio_settings">
                <input type="hidden" name="freshm_tab" value="settings">
                <?php freshm_portfolio_settings_inner(); ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>
    </div>
        </div>
    </div>
    <?php
}

/**
 * Inner settings fields for portfolio/successes (no wrap/form).
 */
function freshm_portfolio_settings_inner() {
    $g = 'portfolio_settings';

    freshm_section_title( 'עמוד הצלחת לקוחות - Hero' );
    freshm_field_text( $g, 'hero_title', 'כותרת', 'תוצאות' );
    freshm_field_text( $g, 'hero_title_gold', 'כותרת (זהב)', 'שמדברות בעד עצמן' );
    freshm_field_textarea( $g, 'hero_subtitle', 'תת-כותרת' );
    freshm_field_text( $g, 'per_page', 'כמות פריטים להציג', '12' );

    freshm_section_title( 'CTA בתחתית' );
    freshm_field_text( $g, 'cta_label', 'תווית', 'רוצים להיות הבאים?' );
    freshm_field_text( $g, 'cta_title', 'כותרת', 'המסעדה שלכם' );
    freshm_field_text( $g, 'cta_title_gold', 'כותרת (זהב)', 'יכולה להיות כאן' );
    freshm_field_textarea( $g, 'cta_subtitle', 'תת-כותרת' );

    freshm_section_title( 'סיידבר הצלחת לקוח' );
    freshm_field_text( $g, 'sidebar_cta_title', 'כותרת CTA סיידבר', 'רוצים תוצאות כאלה?' );
    freshm_field_textarea( $g, 'sidebar_cta_text', 'טקסט CTA סיידבר', 'ספרו לנו על המסעדה שלכם ונתאים חבילה.' );
    freshm_field_text( $g, 'sidebar_cta_btn', 'טקסט כפתור סיידבר', 'דברו איתנו ←' );
}

// ---- SERVICES + MEGA MENU (kept for backward compat, now redirected) ----
function freshm_page_services() {
    freshm_page_wrap_start( 'שירותים + מגה-מנו', 'services_settings' );
    freshm_services_settings_inner();
    freshm_page_wrap_end();
}

// ---- PORTFOLIO (kept for backward compat, now redirected) ----
function freshm_page_portfolio() {
    freshm_page_wrap_start( 'הצלחת לקוחות', 'portfolio_settings' );
    freshm_portfolio_settings_inner();
    freshm_page_wrap_end();
}

// ---- CONTACT (with sub-tabs: טופס, פרטי קשר, מפה) ----
function freshm_page_contact() {
    $tab   = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'form';
    $saved = isset( $_GET['freshm_saved'] );
    $g = 'contact';
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'fresh-contact' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>צור קשר</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">✅ השינויים נשמרו בהצלחה!</div>
        <?php endif; ?>

        <div class="freshm-tabs">
            <button type="button" class="freshm-tab<?php echo $tab === 'form' ? ' active' : ''; ?>" data-tab="form">טופס</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'info' ? ' active' : ''; ?>" data-tab="info">פרטי קשר</button>
            <button type="button" class="freshm-tab<?php echo $tab === 'map' ? ' active' : ''; ?>" data-tab="map">מפה</button>
        </div>

        <!-- Tab: טופס -->
        <div class="freshm-tab-content<?php echo $tab === 'form' ? ' active' : ''; ?>" data-tab="form">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="contact">
                <?php
                freshm_section_title( 'עמוד צור קשר - Hero' );
                freshm_field_text( $g, 'hero_title', 'כותרת', 'בואו' );
                freshm_field_text( $g, 'hero_title_gold', 'כותרת (זהב)', 'נדבר' );
                freshm_field_textarea( $g, 'hero_subtitle', 'תת-כותרת' );

                freshm_section_title( 'טופס' );
                freshm_field_text( $g, 'form_title', 'כותרת טופס', 'שלחו לנו הודעה' );
                freshm_field_textarea( $g, 'form_intro', 'טקסט מבוא לטופס', 'כל מסעדה מצליחה מתחילה משיחה אחת. ספרו לנו קצת על המקום שלכם ומה הייתם רוצים לשפר - ואנחנו נחזור אליכם עם רעיונות ותוכנית ראשונית, בלי התחייבות ובלי שטויות. פשוט שיחה טובה בין אנשים שמבינים מסעדות.' );
                freshm_field_text( $g, 'form_btn_text', 'טקסט כפתור שליחה', 'רוצה לשמוע עוד ←' );
                freshm_field_text( $g, 'form_success', 'הודעת הצלחה', 'ההודעה נשלחה בהצלחה! נחזור אליכם בהקדם.' );
                freshm_field_text( $g, 'form_email_to', 'אימייל לקבלת הודעות', '' , 'ברירת מחדל: אימייל מנהל האתר' );
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: פרטי קשר -->
        <div class="freshm-tab-content<?php echo $tab === 'info' ? ' active' : ''; ?>" data-tab="info">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="contact">
                <?php
                freshm_section_title( 'כרטיסי מידע' );
                freshm_field_repeater( $g, 'info_cards', 'כרטיסי מידע', array(
                    'icon'  => array( 'label' => 'אייקון', 'placeholder' => '📞' ),
                    'title' => array( 'label' => 'כותרת', 'placeholder' => 'טלפון' ),
                    'value' => array( 'label' => 'ערך', 'placeholder' => '050-123-4567' ),
                    'link'  => array( 'label' => 'קישור (אופציונלי)', 'placeholder' => 'tel:+972501234567' ),
                ) );
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

        <!-- Tab: מפה -->
        <div class="freshm-tab-content<?php echo $tab === 'map' ? ' active' : ''; ?>" data-tab="map">
            <form method="post" enctype="multipart/form-data">
                <?php wp_nonce_field( 'freshm_save_action', 'freshm_nonce' ); ?>
                <input type="hidden" name="freshm_save" value="contact">
                <?php
                freshm_section_title( 'מפה' );
                freshm_field_textarea( $g, 'map_embed', 'קוד הטמעת Google Maps', '', 3 );
                echo '<p class="freshm-info">📌 הדביקו את ה-iframe URL מ-Google Maps Embed</p>';
                ?>
                <div class="freshm-save-bar">
                    <button type="submit" class="freshm-btn-save">💾 שמור שינויים</button>
                </div>
            </form>
        </div>

    </div>
        </div>
    </div>
    <?php
}

// ---- FOOTER ----
function freshm_page_footer() {
    freshm_page_wrap_start( 'פוטר', 'footer' );
    $g = 'footer';

    freshm_section_title( 'מותג' );
    freshm_field_textarea( $g, 'brand_text', 'תיאור המותג', 'הסוכנות הדיגיטלית המתמחה בשיווק ופרסום מסעדות. מברנדינג ותוכן, דרך קמפיינים, ועד CRM — הכל תחת קורת גג אחת.' );

    freshm_section_title( 'עמודת שירותים' );
    freshm_field_repeater( $g, 'services_links', 'קישורי שירותים', array(
        'text' => array( 'label' => 'טקסט', 'placeholder' => 'שיווק דיגיטלי' ),
        'url'  => array( 'label' => 'קישור', 'placeholder' => '/services/' ),
    ) );

    freshm_section_title( 'עמודת חברה' );
    freshm_field_repeater( $g, 'company_links', 'קישורי חברה', array(
        'text' => array( 'label' => 'טקסט', 'placeholder' => 'מי אנחנו' ),
        'url'  => array( 'label' => 'קישור', 'placeholder' => '/about/' ),
    ) );

    freshm_section_title( 'מפה בפוטר' );
    freshm_field_select( $g, 'map_enabled', 'הצג מפה בפוטר', array( '0' => 'כבוי', '1' => 'מופעל' ), '0' );
    freshm_field_textarea( $g, 'map_embed', 'קוד הטמעת Google Maps', '', 3 );
    echo '<p class="freshm-info">📌 העתיקו את קוד ה-iframe מ-Google Maps &rarr; Share &rarr; Embed a map.</p>';

    freshm_section_title( 'קרדיט' );
    freshm_field_text( $g, 'copyright', 'טקסט זכויות יוצרים', '© {year} FRESH. כל הזכויות שמורות.' );
    echo '<p class="freshm-info">💡 השתמשו ב-{year} כדי להציג את השנה הנוכחית אוטומטית.</p>';

    freshm_page_wrap_end();
}

/* ==========================================================================
   ENHANCED SERVICE META BOXES
   ========================================================================== */
add_action( 'add_meta_boxes', 'freshm_service_meta_boxes', 20 );
function freshm_service_meta_boxes() {
    // Remove old meta box if exists
    remove_meta_box( 'fresh_service_meta', 'fresh_service', 'normal' );

    add_meta_box(
        'freshm_service_full',
        '🎯 הגדרות שירות — FRESH Manager',
        'freshm_service_meta_callback',
        'fresh_service',
        'normal',
        'high'
    );
}

function freshm_service_meta_callback( $post ) {
    wp_nonce_field( 'freshm_service_save', 'freshm_service_nonce' );
    $m = function( $key, $default = '' ) use ( $post ) {
        $val = get_post_meta( $post->ID, $key, true );
        return $val !== '' ? $val : $default;
    };
    ?>
    <style>
        .freshm-meta { display: grid; gap: 1rem; padding: 1rem 0; }
        .freshm-meta .field { display: grid; gap: 0.3rem; }
        .freshm-meta label { font-weight: 600; font-size: 13px; }
        .freshm-meta input, .freshm-meta textarea, .freshm-meta select { width: 100%; padding: 8px; }
        .freshm-meta .grid-2 { display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; }
        .freshm-meta .grid-3 { display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 1rem; }
        .freshm-meta h3 { margin: 1rem 0 0.5rem; padding-top: 1rem; border-top: 1px solid #ddd; color: #c9a84c; }
        .freshm-meta .img-preview { max-width: 200px; margin-top: 5px; }
        .freshm-meta .img-preview img { max-width: 100%; height: auto; border-radius: 4px; }
    </style>
    <div class="freshm-meta">
        <h3>🎨 תצוגה בסיסית</h3>
        <div class="grid-2">
            <div class="field">
                <label>אייקון (אימוג'י)</label>
                <input type="text" name="service_icon" value="<?php echo esc_attr( $m('service_icon', '🎯') ); ?>">
            </div>
            <div class="field">
                <label>צבע אקסנט (אופציונלי)</label>
                <input type="text" name="service_accent_color" value="<?php echo esc_attr( $m('service_accent_color') ); ?>" placeholder="#c9a84c">
            </div>
        </div>

        <h3>🖼️ מדיה</h3>
        <div class="grid-2">
            <div class="field">
                <label>תמונת Hero / רקע</label>
                <input type="url" name="service_hero_image" value="<?php echo esc_url( $m('service_hero_image') ); ?>" placeholder="URL תמונה" class="freshm-meta-img-input">
                <button type="button" class="button freshm-meta-upload">📷 בחר תמונה</button>
                <?php if ( $m('service_hero_image') ) : ?><div class="img-preview"><img src="<?php echo esc_url( $m('service_hero_image') ); ?>"></div><?php endif; ?>
            </div>
            <div class="field">
                <label>וידאו (YouTube/Vimeo URL)</label>
                <input type="url" name="service_video_url" value="<?php echo esc_url( $m('service_video_url') ); ?>" placeholder="https://www.youtube.com/watch?v=..." dir="ltr">
            </div>
        </div>

        <h3>✨ יתרונות / תכונות</h3>
        <p style="color:#666;font-size:12px;">כל שורה = יתרון אחד. פורמט: אייקון | כותרת | תיאור</p>
        <textarea name="service_benefits" rows="6" placeholder="✅ | ניהול מלא | אנחנו מנהלים הכל מ-A עד Z&#10;📊 | דוחות שקופים | דוח ביצועים חודשי&#10;🎯 | מיקוד גיאוגרפי | פרסום ממוקד לאזור שלכם"><?php echo esc_textarea( $m('service_benefits') ); ?></textarea>

        <h3>⚙️ תהליך עבודה</h3>
        <p style="color:#666;font-size:12px;">כל שורה = שלב. פורמט: מספר | כותרת | תיאור</p>
        <textarea name="service_process" rows="5" placeholder="01 | אפיון | מכירים את המסעדה ומבינים את הצרכים&#10;02 | אסטרטגיה | בונים תוכנית עבודה&#10;03 | ביצוע | מתחילים לעבוד&#10;04 | צמיחה | מודדים ומשפרים"><?php echo esc_textarea( $m('service_process') ); ?></textarea>

        <h3>📊 תוצאות מספריות</h3>
        <p style="color:#666;font-size:12px;">כל שורה = מספר. פורמט: מספר | תיאור</p>
        <textarea name="service_results" rows="3" placeholder="+280% | גידול בהזמנות&#10;4-8x | ROI ממוצע&#10;90 ימים | לתוצאות ראשונות"><?php echo esc_textarea( $m('service_results') ); ?></textarea>

        <h3>🖼️ גלריה</h3>
        <p style="color:#666;font-size:12px;">URL של תמונות — שורה לכל תמונה</p>
        <textarea name="service_gallery" rows="3" placeholder="https://example.com/image1.jpg&#10;https://example.com/image2.jpg"><?php echo esc_textarea( $m('service_gallery') ); ?></textarea>

        <h3>📞 CTA (קריאה לפעולה)</h3>
        <div class="grid-3">
            <div class="field">
                <label>כותרת CTA</label>
                <input type="text" name="service_cta_title" value="<?php echo esc_attr( $m('service_cta_title', 'מתעניינים בשירות?') ); ?>">
            </div>
            <div class="field">
                <label>טקסט CTA</label>
                <input type="text" name="service_cta_text" value="<?php echo esc_attr( $m('service_cta_text', 'נשמח לשמוע על המסעדה שלכם ולהתאים חבילה.') ); ?>">
            </div>
            <div class="field">
                <label>טקסט כפתור</label>
                <input type="text" name="service_cta_btn" value="<?php echo esc_attr( $m('service_cta_btn', 'דברו איתנו ←') ); ?>">
            </div>
        </div>

        <h3>🔗 שירותים קשורים</h3>
        <p style="color:#666;font-size:12px;">מזהי שירותים להציג כ"שירותים קשורים" (מופרד בפסיקים). השאירו ריק לבחירה אוטומטית.</p>
        <input type="text" name="service_related_ids" value="<?php echo esc_attr( $m('service_related_ids') ); ?>" placeholder="ריק = אוטומטי">
    </div>

    <script>
    jQuery(function($) {
        $(document).on('click', '.freshm-meta-upload', function(e) {
            e.preventDefault();
            var $input = $(this).siblings('.freshm-meta-img-input');
            var frame = wp.media({ title: 'בחרו תמונה', multiple: false });
            frame.on('select', function() {
                var url = frame.state().get('selection').first().toJSON().url;
                $input.val(url);
                $(e.target).siblings('.img-preview').remove();
                $(e.target).after('<div class="img-preview"><img src="'+url+'"></div>');
            });
            frame.open();
        });
    });
    </script>
    <?php
}

add_action( 'save_post_fresh_service', 'freshm_save_service_meta' );
function freshm_save_service_meta( $post_id ) {
    if ( ! isset( $_POST['freshm_service_nonce'] ) || ! wp_verify_nonce( $_POST['freshm_service_nonce'], 'freshm_service_save' ) ) return;
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;

    $fields = array( 'service_icon', 'service_accent_color', 'service_hero_image', 'service_video_url', 'service_benefits', 'service_process', 'service_results', 'service_gallery', 'service_cta_title', 'service_cta_text', 'service_cta_btn', 'service_related_ids' );
    foreach ( $fields as $field ) {
        if ( isset( $_POST[ $field ] ) ) {
            update_post_meta( $post_id, $field, wp_kses_post( $_POST[ $field ] ) );
        }
    }
}

/* ==========================================================================
   ENHANCED CLIENT SUCCESS STORY META BOXES
   ========================================================================== */
add_action( 'add_meta_boxes', 'freshm_case_study_meta_boxes', 20 );
function freshm_case_study_meta_boxes() {
    remove_meta_box( 'fresh_case_study_meta', 'fresh_case_study', 'normal' );
    add_meta_box( 'freshm_case_study_full', '📁 פרטי הצלחת לקוח — FRESH Manager', 'freshm_case_study_meta_callback', 'fresh_case_study', 'normal', 'high' );
}

function freshm_case_study_meta_callback( $post ) {
    wp_nonce_field( 'freshm_cs_save', 'freshm_cs_nonce' );
    $m = function( $key, $default = '' ) use ( $post ) {
        $val = get_post_meta( $post->ID, $key, true );
        return $val !== '' ? $val : $default;
    };
    ?>
    <style>.freshm-meta .grid-4 { display: grid; grid-template-columns: repeat(4, 1fr); gap: 1rem; }</style>
    <div class="freshm-meta">
        <h3>👤 פרטי לקוח</h3>
        <div class="grid-3">
            <div class="field"><label>שם לקוח</label><input type="text" name="client_name" value="<?php echo esc_attr( $m('client_name') ); ?>"></div>
            <div class="field"><label>תעשייה</label><input type="text" name="industry" value="<?php echo esc_attr( $m('industry') ); ?>"></div>
            <div class="field"><label>משך הפרויקט</label><input type="text" name="duration" value="<?php echo esc_attr( $m('duration') ); ?>"></div>
        </div>

        <h3>📊 תוצאות</h3>
        <div class="field"><label>תוצאה מרכזית</label><input type="text" name="main_result" value="<?php echo esc_attr( $m('main_result') ); ?>" placeholder="+280% בהזמנות אונליין"></div>
        <div class="grid-4">
            <div class="field"><label>לידים</label><input type="text" name="result_leads" value="<?php echo esc_attr( $m('result_leads') ); ?>"></div>
            <div class="field"><label>ROI</label><input type="text" name="result_roi" value="<?php echo esc_attr( $m('result_roi') ); ?>"></div>
            <div class="field"><label>הכנסות</label><input type="text" name="result_revenue" value="<?php echo esc_attr( $m('result_revenue') ); ?>"></div>
            <div class="field"><label>אנגייג'מנט</label><input type="text" name="result_engagement" value="<?php echo esc_attr( $m('result_engagement') ); ?>"></div>
        </div>

        <h3>💬 ציטוט לקוח</h3>
        <div class="field"><label>ציטוט</label><textarea name="client_quote" rows="3"><?php echo esc_textarea( $m('client_quote') ); ?></textarea></div>
        <div class="field"><label>שם מצטט</label><input type="text" name="client_quote_author" value="<?php echo esc_attr( $m('client_quote_author') ); ?>"></div>

        <h3>🖼️ מדיה</h3>
        <div class="grid-2">
            <div class="field"><label>תמונת Hero</label><input type="url" name="cs_hero_image" value="<?php echo esc_url( $m('cs_hero_image') ); ?>" class="freshm-meta-img-input"><button type="button" class="button freshm-meta-upload">📷 בחר</button></div>
            <div class="field"><label>וידאו</label><input type="url" name="cs_video_url" value="<?php echo esc_url( $m('cs_video_url') ); ?>" placeholder="YouTube/Vimeo URL" dir="ltr"></div>
        </div>
        <div class="field"><label>גלריה (URL לכל שורה)</label><textarea name="cs_gallery" rows="3"><?php echo esc_textarea( $m('cs_gallery') ); ?></textarea></div>
    </div>

    <script>
    jQuery(function($) {
        $(document).on('click', '.freshm-meta-upload', function(e) {
            e.preventDefault();
            var $input = $(this).siblings('.freshm-meta-img-input');
            var frame = wp.media({ title: 'בחרו תמונה', multiple: false });
            frame.on('select', function() { $input.val(frame.state().get('selection').first().toJSON().url); });
            frame.open();
        });
    });
    </script>
    <?php
}

add_action( 'save_post_fresh_case_study', 'freshm_save_cs_meta' );
function freshm_save_cs_meta( $post_id ) {
    if ( ! isset( $_POST['freshm_cs_nonce'] ) || ! wp_verify_nonce( $_POST['freshm_cs_nonce'], 'freshm_cs_save' ) ) return;
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
    $fields = array( 'client_name', 'industry', 'duration', 'main_result', 'result_leads', 'result_roi', 'result_revenue', 'result_engagement', 'client_quote', 'client_quote_author', 'cs_hero_image', 'cs_video_url', 'cs_gallery' );
    foreach ( $fields as $f ) {
        if ( isset( $_POST[ $f ] ) ) update_post_meta( $post_id, $f, wp_kses_post( $_POST[ $f ] ) );
    }
}

/* ==========================================================================
   MEGA MENU — FRONT-END OUTPUT
   ========================================================================== */
add_filter( 'wp_nav_menu_items', 'freshm_mega_menu_inject', 10, 2 );
function freshm_mega_menu_inject( $items, $args ) {
    if ( ! isset( $args->theme_location ) || $args->theme_location !== 'primary' ) return $items;

    // Skip if theme's Fresh_Mega_Walker already handles mega menu injection
    if ( class_exists( 'Fresh_Mega_Walker' ) && ! empty( $args->walker ) && $args->walker instanceof Fresh_Mega_Walker ) {
        return $items;
    }

    $settings = freshm_get_all( 'services_settings' );
    $enabled = isset( $settings['freshm_megamenu_enabled'] ) ? $settings['freshm_megamenu_enabled'] : '1';
    if ( $enabled !== '1' ) return $items;

    // Find the "שירותים" menu item and add mega menu class + content
    $services_page = get_page_by_path( 'services' );
    if ( ! $services_page ) return $items;

    $services_url = get_permalink( $services_page->ID );

    // Build mega menu HTML
    $mega_html = freshm_build_mega_menu();

    // Replace the services link with mega menu version
    $pattern = '(<li[^>]*class="[^"]*menu-item[^"]*"[^>]*>\s*<a[^>]*href="' . preg_quote( $services_url, '/' ) . '"[^>]*>)([^<]*)(<\/a>)';
    $replacement = '$1$2$3' . $mega_html;

    $items = preg_replace( '/' . $pattern . '/u', '$1$2$3' . addcslashes( $mega_html, '\\$' ), $items, 1 );

    // Add mega-menu class to the LI
    $items = preg_replace(
        '/(<li[^>]*class="[^"]*)(menu-item[^"]*"[^>]*>\s*<a[^>]*href="' . preg_quote( $services_url, '/' ) . '")/u',
        '$1has-mega-menu $2',
        $items,
        1
    );

    return $items;
}

function freshm_build_mega_menu() {
    $settings = freshm_get_all( 'services_settings' );

    // Try plugin data first, then WP taxonomy fallback
    $plugin_cats = freshm_get_service_cats();
    $plugin_svcs = freshm_get_services();

    if ( empty( $plugin_cats ) && empty( $plugin_svcs ) ) {
        // Fallback to taxonomy
        $wp_cats = get_terms( array( 'taxonomy' => 'service_category', 'hide_empty' => true ) );
        if ( is_wp_error( $wp_cats ) || empty( $wp_cats ) ) return '';
    }

    ob_start();
    ?>
    <div class="mega-menu">
        <div class="mega-menu-inner">
            <div class="mega-menu-categories">
                <?php if ( ! empty( $plugin_cats ) ) :
                    // Plugin data path
                    usort( $plugin_cats, function( $a, $b ) { return ( $a['order'] ?? 0 ) - ( $b['order'] ?? 0 ); } );
                    foreach ( $plugin_cats as $cat ) :
                        $cat_services = array_filter( $plugin_svcs, function( $s ) use ( $cat ) {
                            return ( $s['category'] ?? '' ) === $cat['slug'];
                        } );
                        if ( empty( $cat_services ) ) continue;
                        usort( $cat_services, function( $a, $b ) { return ( $a['order'] ?? 0 ) - ( $b['order'] ?? 0 ); } );
                ?>
                <div class="mega-menu-col">
                    <h4 class="mega-menu-cat-title"><?php echo esc_html( $cat['name'] ); ?></h4>
                    <ul>
                        <?php foreach ( $cat_services as $sv ) : ?>
                        <li><a href="<?php echo esc_url( home_url( '/service/' . $sv['slug'] . '/' ) ); ?>"><span class="icon"><?php echo freshm_render_service_icon( $sv, 'mini' ); ?></span><?php echo esc_html( $sv['name'] ); ?></a></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endforeach;
                else :
                    // WP taxonomy fallback
                    foreach ( $wp_cats as $cat ) :
                        $services = get_posts( array(
                            'post_type'      => 'fresh_service',
                            'posts_per_page' => 10,
                            'tax_query'      => array( array( 'taxonomy' => 'service_category', 'field' => 'term_id', 'terms' => $cat->term_id ) ),
                            'orderby'        => 'menu_order',
                            'order'          => 'ASC',
                        ) );
                ?>
                <div class="mega-menu-col">
                    <h4 class="mega-menu-cat-title"><?php echo esc_html( $cat->name ); ?></h4>
                    <ul>
                        <?php foreach ( $services as $service ) :
                            $icon = get_post_meta( $service->ID, 'service_icon', true ) ?: '◆';
                        ?>
                        <li><a href="<?php echo get_permalink( $service->ID ); ?>"><span class="icon"><?php echo freshm_render_service_icon( $icon, 'mini' ); ?></span><?php echo esc_html( $service->post_title ); ?></a></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endforeach; endif; ?>
            </div>
            <?php
            $featured_img = $settings['freshm_megamenu_featured_image'] ?? '';
            $featured_title = $settings['freshm_megamenu_featured_title'] ?? '';
            if ( $featured_img || $featured_title ) :
            ?>
            <div class="mega-menu-featured">
                <?php if ( $featured_img ) : ?>
                    <img src="<?php echo esc_url( $featured_img ); ?>" alt="">
                <?php endif; ?>
                <?php if ( $featured_title ) : ?>
                    <h4><?php echo esc_html( $featured_title ); ?></h4>
                <?php endif; ?>
                <?php if ( ! empty( $settings['freshm_megamenu_featured_desc'] ) ) : ?>
                    <p><?php echo esc_html( $settings['freshm_megamenu_featured_desc'] ); ?></p>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            <?php
            $cta_text = $settings['freshm_megamenu_cta_text'] ?? '';
            $cta_url  = $settings['freshm_megamenu_cta_url'] ?? '/services/';
            if ( $cta_text ) :
            ?>
            <div class="mega-menu-footer">
                <a href="<?php echo esc_url( home_url( $cta_url ) ); ?>" class="mega-menu-cta"><?php echo esc_html( $cta_text ); ?></a>
            </div>
            <?php endif; ?>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

/* ==========================================================================
   MEGA MENU CSS (injected in head)
   ========================================================================== */
add_action( 'wp_head', 'freshm_mega_menu_css' );
function freshm_mega_menu_css() {
    // Skip plugin CSS when theme already handles mega menu styling
    if ( class_exists( 'Fresh_Mega_Walker' ) ) return;
    ?>
    <style id="freshm-mega-menu">
    /* Mega Menu */
    .has-mega-menu { position: relative; }
    .mega-menu {
        position: absolute;
        top: 100%;
        right: 50%;
        transform: translateX(50%) translateY(12px);
        min-width: 1000px;
        background: rgba(17, 17, 22, 0.98);
        backdrop-filter: blur(20px);
        -webkit-backdrop-filter: blur(20px);
        border: 1px solid rgba(201, 168, 76, 0.15);
        border-radius: 16px;
        padding: 0;
        z-index: 9999;
        opacity: 0;
        visibility: hidden;
        pointer-events: none;
        transition: opacity 0.35s cubic-bezier(0.23, 1, 0.32, 1), transform 0.35s cubic-bezier(0.23, 1, 0.32, 1), visibility 0s 0.35s;
        box-shadow: 0 25px 60px rgba(0,0,0,0.5);
    }
    .has-mega-menu:hover .mega-menu {
        opacity: 1;
        visibility: visible;
        pointer-events: auto;
        transform: translateX(50%) translateY(0);
        transition: opacity 0.35s cubic-bezier(0.23, 1, 0.32, 1), transform 0.35s cubic-bezier(0.23, 1, 0.32, 1), visibility 0s;
    }
    .mega-menu-inner { padding: 2rem; }
    .mega-menu-categories {
        display: grid;
        grid-template-columns: repeat(6, 1fr);
        gap: 1.5rem;
    }
    .mega-menu-cat-title {
        color: var(--gold, #c9a84c);
        font-size: 0.85rem;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 1px;
        margin-bottom: 0.8rem;
        padding-bottom: 0.5rem;
        border-bottom: 1px solid rgba(201, 168, 76, 0.2);
    }
    .mega-menu ul { list-style: none; padding: 0; margin: 0; }
    .mega-menu li { margin: 0; }
    .mega-menu li a {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.45rem 0;
        color: #a0a0aa;
        font-size: 0.82rem;
        text-decoration: none;
        transition: color 0.2s, padding-right 0.2s;
        white-space: nowrap;
    }
    .mega-menu li a:hover {
        color: var(--gold, #c9a84c);
        padding-right: 5px;
    }
    .mega-menu li a:hover::after { display: none; }
    .mega-menu li a .icon { font-size: 0.9rem; width: 1.2em; text-align: center; flex-shrink: 0; }
    .mega-menu li a .icon .service-icon { width: 18px; height: 18px; }
    .mega-menu li a .icon .service-icon svg { width: 100%; height: 100%; color: currentColor; }
    .mega-menu-featured {
        margin-top: 1.5rem;
        padding-top: 1.5rem;
        border-top: 1px solid rgba(201, 168, 76, 0.1);
        display: flex;
        align-items: center;
        gap: 1.5rem;
    }
    .mega-menu-featured img { width: 120px; height: 80px; object-fit: cover; border-radius: 10px; border: 1px solid rgba(201, 168, 76, 0.1); }
    .mega-menu-featured h4 { color: #E8E6E1; font-size: 0.95rem; margin: 0; }
    .mega-menu-featured p { color: #a0a0aa; font-size: 0.82rem; margin: 0.3rem 0 0; }
    .mega-menu-footer {
        margin-top: 1.5rem;
        padding-top: 1rem;
        border-top: 1px solid rgba(201, 168, 76, 0.1);
        text-align: center;
    }
    .mega-menu-cta {
        display: inline-block;
        color: var(--gold, #c9a84c);
        font-weight: 600;
        font-size: 0.85rem;
        text-decoration: none;
        padding: 0.5rem 1.5rem;
        border: 1px solid rgba(201, 168, 76, 0.3);
        border-radius: 8px;
        transition: all 0.3s;
    }
    .mega-menu-cta:hover {
        background: var(--gold, #c9a84c);
        color: #0A0A0C;
        border-color: var(--gold, #c9a84c);
    }
    @media (max-width: 1024px) {
        .mega-menu { display: none !important; }
    }
    </style>
    <?php
}

/* ==========================================================================
   DYNAMIC CSS FROM SETTINGS
   ========================================================================== */
add_action( 'wp_head', 'freshm_dynamic_css', 20 );
function freshm_dynamic_css() {
    $primary = fm( 'global', 'color_primary', '#c9a84c' );
    $bg      = fm( 'global', 'color_bg', '#0A0A0C' );
    $bg2     = fm( 'global', 'color_bg2', '#111116' );
    $text    = fm( 'global', 'color_text', '#E8E6E1' );

    if ( $primary === '#c9a84c' && $bg === '#0A0A0C' ) return; // defaults, skip
    ?>
    <style id="freshm-dynamic-colors">
    :root {
        --gold: <?php echo esc_attr( $primary ); ?>;
        --bg-primary: <?php echo esc_attr( $bg ); ?>;
        --bg-secondary: <?php echo esc_attr( $bg2 ); ?>;
        --text-primary: <?php echo esc_attr( $text ); ?>;
    }
    </style>
    <?php
}

/* ==========================================================================
   THEME HELPER FUNCTIONS (available in templates)
   ========================================================================== */

/**
 * Get global contact info
 */
function freshm_phone( $linked = true ) {
    $phone = fm( 'global', 'phone', '050-123-4567' );
    if ( $linked ) return '<a href="tel:+' . preg_replace( '/[^0-9]/', '', $phone ) . '">' . esc_html( $phone ) . '</a>';
    return esc_html( $phone );
}

function freshm_whatsapp_url() {
    $wa = fm( 'global', 'whatsapp', '972501234567' );
    return 'https://wa.me/' . preg_replace( '/[^0-9]/', '', $wa );
}

function freshm_email( $linked = true ) {
    $email = fm( 'global', 'email', 'hello@fresh.agency' );
    if ( $linked ) return '<a href="mailto:' . esc_attr( $email ) . '">' . esc_html( $email ) . '</a>';
    return esc_html( $email );
}

/**
 * Parse pipe-separated meta (benefits, process, results)
 */
function freshm_parse_pipe_meta( $text ) {
    if ( empty( $text ) ) return array();
    $items = array();
    $lines = array_filter( array_map( 'trim', explode( "\n", $text ) ) );
    foreach ( $lines as $line ) {
        $parts = array_map( 'trim', explode( '|', $line ) );
        $items[] = $parts;
    }
    return $items;
}

/**
 * Parse gallery URLs
 */
function freshm_parse_gallery( $text ) {
    if ( empty( $text ) ) return array();
    return array_filter( array_map( 'trim', explode( "\n", $text ) ) );
}

// ---- BLOG SETTINGS ----
function freshm_page_blog() {
    freshm_page_wrap_start( 'הגדרות בלוג', 'blog_settings' );
    $g = 'blog_settings';

    freshm_section_title( 'הפעלה/כיבוי' );
    freshm_field_select( $g, 'enabled', 'הצג בלוג באתר', array( '1' => 'מופעל', '0' => 'כבוי — מוסתר מהניווט והפוטר' ), '1' );

    freshm_section_title( 'עמוד בלוג - Hero' );
    freshm_field_text( $g, 'hero_title', 'כותרת (תומך HTML ל-<span>)', 'ה<span class="gold">בלוג</span> של FRESH' );
    freshm_field_text( $g, 'hero_subtitle', 'תת-כותרת', 'טיפים, תובנות ומגמות מעולם השיווק הדיגיטלי למסעדות.' );

    freshm_section_title( 'הגדרות תצוגה' );
    freshm_field_select( $g, 'layout', 'סוג תצוגה', array( 'grid' => 'רשת', 'list' => 'רשימה' ), 'grid' );
    freshm_field_text( $g, 'posts_per_page', 'פוסטים לעמוד', '9' );
    freshm_field_select( $g, 'show_sidebar', 'הצג סיידבר', array( '1' => 'כן', '0' => 'לא' ), '1' );
    freshm_field_select( $g, 'show_featured', 'הצג פוסט מובלט ראשון', array( '1' => 'כן', '0' => 'לא' ), '1' );

    freshm_page_wrap_end();
}

// ---- VIDEO CAROUSEL ----
function freshm_page_videos() {
    freshm_page_wrap_start( 'קרוסלת וידאו', 'video_carousel' );
    $g = 'video_carousel';

    freshm_section_title( 'קרוסלת וידאו - דף הבית', 'מחליפה את סקשן השואוריל. תומכת בעד 15 סרטונים.' );
    freshm_field_select( $g, 'enabled', 'הפעל קרוסלת וידאו', array( '1' => 'מופעל (מחליף שואוריל)', '0' => 'כבוי (שואוריל רגיל)' ), '1' );
    freshm_field_text( $g, 'section_label', 'תווית סקשן', 'ראו אותנו בפעולה' );
    freshm_field_text( $g, 'section_title', 'כותרת', 'הסרטונים' );
    freshm_field_text( $g, 'section_title_gold', 'כותרת (חלק זהב)', 'שלנו' );
    freshm_field_textarea( $g, 'section_subtitle', 'תת-כותרת', 'צפו בעבודות שלנו ותנו לתוצאות לדבר.' );

    freshm_field_repeater( $g, 'videos', 'סרטונים (עד 15)', array(
        'url'       => array( 'label' => 'קישור YouTube/Vimeo', 'placeholder' => 'https://www.youtube.com/watch?v=...' ),
        'title'     => array( 'label' => 'כותרת', 'placeholder' => 'שם הסרטון' ),
        'thumbnail' => array( 'label' => 'תמונה ממוזערת', 'type' => 'image' ),
    ) );

    freshm_page_wrap_end();
}

// ---- 404 PAGE ----
// ---- FAQ PAGE ----
function freshm_page_faq() {
    $saved = isset( $_GET['freshm_saved'] );
    ?>
    <div class="freshm-admin-layout">
        <?php freshm_render_sidebar( 'fresh-faq' ); ?>
        <div class="freshm-content">
    <div class="freshm-wrap">
        <div class="freshm-header">
            <h1>שאלות ותשובות</h1>
        </div>
        <?php if ( $saved ) : ?>
            <div class="freshm-notice freshm-notice-success">&#10004; השינויים נשמרו!</div>
        <?php endif; ?>
        <form method="post">
            <?php wp_nonce_field( 'freshm_faq_action', 'freshm_faq_nonce' ); ?>
            <input type="hidden" name="freshm_save_faq" value="1">

            <div class="freshm-cards-toolbar">
                <button type="button" class="freshm-btn freshm-add-faq">+ הוסף שאלה</button>
                <?php $faq = get_option( 'freshm_faq', array() ); ?>
                <span class="freshm-cards-count"><?php echo is_array( $faq ) ? count( $faq ) : 0; ?> שאלות</span>
            </div>

            <div class="freshm-cards-list" id="freshm-faq-list">
                <?php
                if ( is_array( $faq ) && ! empty( $faq ) ) :
                    foreach ( $faq as $idx => $item ) :
                        freshm_render_faq_card( $idx, $item );
                    endforeach;
                endif;
                ?>
            </div>

            <script type="text/html" id="freshm-faq-template">
                <?php freshm_render_faq_card( '__IDX__', array() ); ?>
            </script>

            <div class="freshm-save-bar">
                <button type="submit" class="freshm-btn-save">שמור שאלות ותשובות</button>
            </div>
        </form>

        <script>
        jQuery(function($){
            $(document).on('click', '.freshm-add-faq', function(){
                var html = $('#freshm-faq-template').html().replace(/__IDX__/g, Date.now());
                var $card = $(html);
                $card.addClass('open');
                $('#freshm-faq-list').append($card);
                $('html, body').animate({ scrollTop: $card.offset().top - 100 }, 300);
                var count = $('#freshm-faq-list').find('.freshm-card').length;
                $(this).closest('form').find('.freshm-cards-count').text(count + ' שאלות');
            });
            $(document).on('click', '.freshm-delete-faq', function(e){
                e.stopPropagation();
                if (!confirm('למחוק שאלה זו?')) return;
                var $card = $(this).closest('.freshm-card');
                var $list = $card.closest('.freshm-cards-list');
                $card.slideUp(300, function(){
                    $(this).remove();
                    var count = $list.find('.freshm-card').length;
                    $list.closest('form').find('.freshm-cards-count').text(count + ' שאלות');
                });
            });
            $(document).on('input', '#freshm-faq-list .freshm-faq-q', function(){
                var val = $(this).val() || 'שאלה חדשה';
                $(this).closest('.freshm-card').find('.freshm-card-title').text(val);
            });
            if ($.fn.sortable) {
                $('#freshm-faq-list').sortable({
                    handle: '.freshm-card-handle',
                    placeholder: 'ui-sortable-placeholder',
                    opacity: 0.8
                });
            }
        });
        </script>
    </div>
        </div>
    </div>
    <?php
}

function freshm_render_faq_card( $idx, $item ) {
    $item = wp_parse_args( $item, array( 'q' => '', 'a' => '' ) );
    $n = "freshm_faq[{$idx}]";
    ?>
    <div class="freshm-card" data-type="faq">
        <div class="freshm-card-header" onclick="this.parentElement.classList.toggle('open')">
            <span class="freshm-card-handle">&#9776;</span>
            <span class="freshm-card-title"><?php echo $item['q'] ? esc_html( mb_substr( $item['q'], 0, 60 ) ) : 'שאלה חדשה'; ?></span>
            <span class="freshm-card-toggle">&#9660;</span>
        </div>
        <div class="freshm-card-body">
            <div class="freshm-card-section">
                <div class="freshm-field">
                    <label>שאלה *</label>
                    <input type="text" name="<?php echo $n; ?>[q]" value="<?php echo esc_attr( $item['q'] ); ?>" class="freshm-faq-q" required>
                </div>
                <div class="freshm-field">
                    <label>תשובה (HTML מותר לקישורים)</label>
                    <textarea name="<?php echo $n; ?>[a]" rows="4" style="width:100%;min-height:100px;"><?php echo esc_textarea( $item['a'] ); ?></textarea>
                </div>
            </div>
            <div class="freshm-card-actions">
                <button type="button" class="freshm-delete-faq">מחק שאלה</button>
            </div>
        </div>
    </div>
    <?php
}

add_action( 'admin_init', 'freshm_save_faq_handler' );
function freshm_save_faq_handler() {
    if ( ! isset( $_POST['freshm_save_faq'] ) || ! current_user_can( 'manage_options' ) ) return;
    check_admin_referer( 'freshm_faq_action', 'freshm_faq_nonce' );

    $raw = isset( $_POST['freshm_faq'] ) ? $_POST['freshm_faq'] : array();
    $faq = array();
    foreach ( $raw as $item ) {
        if ( empty( $item['q'] ) ) continue;
        $faq[] = array(
            'q' => sanitize_text_field( wp_unslash( $item['q'] ) ),
            'a' => wp_kses( wp_unslash( $item['a'] ), array( 'a' => array( 'href' => array(), 'target' => array(), 'rel' => array() ), 'br' => array(), 'strong' => array(), 'em' => array() ) ),
        );
    }
    update_option( 'freshm_faq', $faq );
    wp_safe_redirect( add_query_arg( 'freshm_saved', '1' ) );
    exit;
}

function freshm_page_404() {
    freshm_page_wrap_start( 'דף 404', '404_page' );
    $g = '404_page';

    freshm_section_title( 'דף 404 - לא נמצא' );
    freshm_field_text( $g, 'title', 'כותרת', 'הדף לא נמצא' );
    freshm_field_textarea( $g, 'text', 'טקסט', 'נראה שהגעתם למקום הלא נכון. אולי הדף הוסר או שהכתובת שגויה.' );
    freshm_field_text( $g, 'btn_home', 'טקסט כפתור ראשי', 'חזרה לדף הבית ←' );
    freshm_field_text( $g, 'btn_contact', 'טקסט כפתור צור קשר', 'צור קשר' );
    freshm_field_image( $g, 'image', 'תמונה (אופציונלי)' );

    freshm_page_wrap_end();
}

// ---- TRACKING CODES ----
function freshm_page_tracking() {
    freshm_page_wrap_start( 'קודי מעקב', 'global' );
    $g = 'global';

    echo '<div class="freshm-notice" style="background:rgba(201,168,76,0.1);border-color:var(--gold,#c9a84c);color:#f0ebe3;margin-bottom:1.5rem;padding:1rem 1.2rem;border-radius:8px;border-right:4px solid var(--gold,#c9a84c);">';
    echo '<strong>&#9888;&#65039; שימו לב:</strong> קוד שגוי עלול לשבור את האתר. הדביקו רק קודים ממקורות מהימנים.';
    echo '</div>';

    freshm_section_title( 'קוד ל-Head', 'קוד לפני סגירת </head> - Google Analytics, Tag Manager, Meta Pixel וכו\'' );
    $code_head = fm( $g, 'code_head', '' );
    echo '<div class="freshm-field freshm-field-wide">';
    echo '<textarea name="freshm_code_head" rows="6" class="code-injection-field" dir="ltr" style="text-align:left;">' . esc_textarea( $code_head ) . '</textarea>';
    echo '</div>';

    freshm_section_title( 'קוד אחרי פתיחת Body', 'קוד אחרי פתיחת <body> - noscript של Tag Manager וכו\'' );
    $code_body = fm( $g, 'code_body_open', '' );
    echo '<div class="freshm-field freshm-field-wide">';
    echo '<textarea name="freshm_code_body_open" rows="6" class="code-injection-field" dir="ltr" style="text-align:left;">' . esc_textarea( $code_body ) . '</textarea>';
    echo '</div>';

    freshm_section_title( 'קוד לפני סגירת Body', 'קוד לפני סגירת </body> - קודי מעקב שצריכים להיטען בסוף העמוד' );
    $code_footer = fm( $g, 'code_footer', '' );
    echo '<div class="freshm-field freshm-field-wide">';
    echo '<textarea name="freshm_code_footer" rows="6" class="code-injection-field" dir="ltr" style="text-align:left;">' . esc_textarea( $code_footer ) . '</textarea>';
    echo '</div>';

    freshm_page_wrap_end();
}

/* ==========================================================================
   CODE INJECTION — Front-end Output
   ========================================================================== */
add_action( 'wp_head', function() {
    if ( is_admin() ) return;
    $code = fm( 'global', 'code_head', '' );
    if ( $code ) echo $code . "\n";
}, 99 );

add_action( 'wp_body_open', function() {
    $code = fm( 'global', 'code_body_open', '' );
    if ( $code ) echo $code . "\n";
}, 1 );

add_action( 'wp_footer', function() {
    if ( is_admin() ) return;
    $code = fm( 'global', 'code_footer', '' );
    if ( $code ) echo $code . "\n";
}, 99 );

/* ==========================================================================
   COOKIE CONSENT BANNER
   ========================================================================== */
add_action( 'wp_footer', 'freshm_cookie_banner', 50 );
function freshm_cookie_banner() {
    if ( is_admin() ) return;
    if ( fm( 'global', 'cookie_enabled', '0' ) !== '1' ) return;

    $text      = fm( 'global', 'cookie_text', 'אתר זה משתמש בעוגיות כדי לשפר את חוויית הגלישה שלכם.' );
    $btn_text  = fm( 'global', 'cookie_btn_text', 'הבנתי, אני מסכים/ה' );
    $learn_url = fm( 'global', 'cookie_learn_url', '' );
    $learn_txt = fm( 'global', 'cookie_learn_text', 'מדיניות פרטיות' );
    ?>
    <div id="freshm-cookie-banner" class="cookie-banner" style="display:none;">
        <div class="cookie-banner-inner">
            <p class="cookie-text"><?php echo esc_html( $text ); ?>
                <?php if ( $learn_url ) : ?>
                    <a href="<?php echo esc_url( $learn_url ); ?>" class="cookie-learn"><?php echo esc_html( $learn_txt ); ?></a>
                <?php endif; ?>
            </p>
            <button type="button" class="cookie-accept" id="freshm-cookie-accept"><?php echo esc_html( $btn_text ); ?></button>
        </div>
    </div>
    <script>
    (function(){
        if (localStorage.getItem('freshm_cookie_ok')) return;
        var b = document.getElementById('freshm-cookie-banner');
        if (!b) return;
        b.style.display = '';
        document.getElementById('freshm-cookie-accept').addEventListener('click', function(){
            localStorage.setItem('freshm_cookie_ok', '1');
            b.style.transform = 'translateY(100%)';
            setTimeout(function(){ b.style.display = 'none'; }, 400);
        });
    })();
    </script>
    <?php
}

/* ==========================================================================
   PLUGIN ACTIVATION
   ========================================================================== */
register_activation_hook( __FILE__, 'freshm_activate' );
function freshm_activate() {
    // Set some default options
    if ( ! get_option( 'freshm_global' ) ) {
        update_option( 'freshm_global', array(
            'freshm_phone'     => '050-123-4567',
            'freshm_whatsapp'  => '972501234567',
            'freshm_email'     => 'hello@fresh.agency',
            'freshm_address'   => 'רוטשילד 45, תל אביב',
            'freshm_hours'     => 'א׳-ה׳ 09:00-18:00',
            'freshm_color_primary' => '#c9a84c',
            'freshm_color_bg'  => '#0A0A0C',
            'freshm_color_bg2' => '#111116',
            'freshm_color_text'=> '#E8E6E1',
        ) );
    }

    // Seed default 12 services and 6 categories (only if empty)
    freshm_seed_default_services( false );

    // Seed default 14 category pages (only if empty)
    freshm_seed_category_pages( false );

    // Clear transient so category terms get created on next init
    delete_transient( 'freshm_cats_terms_checked' );

    // Flush rewrite rules so taxonomy URLs work
    flush_rewrite_rules();
}
